# 機能設計書 7-stdQuadTree

## 概要

本ドキュメントは、stdVBAライブラリにおけるQuadtreeデータ構造クラス `stdQuadTree` の機能設計を詳細に記述したものである。

### 本機能の処理概要

stdQuadTreeは、2次元空間における点データを効率的に格納・検索するためのQuadtree（四分木）データ構造を実装するクラスである。空間を再帰的に4つの象限に分割することで、範囲検索を高速に行うことができる。

**業務上の目的・背景**：地理情報システム（GIS）、衝突検出、画像処理など、2次元空間データを扱う場面では、全点の線形探索ではパフォーマンスが不足する。stdQuadTreeはこの課題を解決し、空間インデックスにより効率的な範囲検索を可能にする。Excel上での地図データ処理、座標データの空間検索などで活用できる。

**機能の利用シーン**：
- 地理座標データの空間検索（近傍検索、範囲検索）
- Excel上のプロット点の効率的な管理
- 衝突検出や近接判定
- 大量の2次元座標データのフィルタリング
- ヒートマップ生成のためのデータ集約

**主要な処理内容**：
1. Quadtreeの生成（Create, CreateFromBounds）
2. 点の挿入（insertPoint）
3. 範囲検索（queryRange, queryRangeBounds）
4. ノードの分割（SubDivide）
5. 境界ボックスの交差判定（AABB操作群）

**関連システム・外部連携**：特になし（スタンドアロンのデータ構造クラス）。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本クラスはUIを持たないライブラリクラス |

## 機能種別

データ構造（空間インデックス） / 空間検索 / パフォーマンス最適化

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| cx | Double | Yes | 中心X座標 | - |
| cy | Double | Yes | 中心Y座標 | - |
| halfDimension | Double | Yes | 半径（セルの半辺長） | 正の値 |
| capacityBeforeSubdivision | Long | No | 分割前の最大点数（デフォルト4） | 正の整数 |
| minimumHalfDimension | Double | No | 最小セルサイズ（デフォルト0.00001） | 正の値 |
| minX, minY, maxX, maxY | Double | Yes | 境界座標（CreateFromBounds時） | minX < maxX, minY < maxY |
| x, y | Double | Yes | 挿入する点の座標 | 境界内 |
| meta | Variant | No | 点に付随するメタデータ | - |

### 入力データソース

- ユーザー指定の座標値
- 点に付随するメタデータ（任意のVariant）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| insertPoint | Boolean | 挿入成功/失敗 |
| queryRange | Collection | 範囲内の点（Array(x, y, meta)のCollection） |
| queryRangeBounds | Collection | 範囲内の点（Array(x, y, meta)のCollection） |

### 出力先

- 各メソッドの戻り値

## 処理フロー

### 処理シーケンス

```
1. Create(cx, cy, halfDimension)
   └─ 新規stdQuadTreeインスタンス生成
   └─ protInit: LeafノードとしてAABB境界を設定

2. insertPoint(x, y, meta)
   └─ 境界内チェック（AABB_containsXY）
      └─ 範囲外: False返却
   └─ ノード種別判定
      └─ Leaf:
         └─ 容量未満: 点を配列に追加
         └─ 容量超過: SubDivide → Branch化
      └─ Branch:
         └─ 4象限の子ノードに再帰的に挿入

3. SubDivide()
   └─ Branch化（Type = QuadTreeBranch）
   └─ 4つの子ノード生成（NW, NE, SW, SE）
   └─ 既存点を子ノードに再挿入
   └─ 既存点配列を解放

4. queryRange(cx, cy, radius)
   └─ AABB作成
   └─ pQueryRange: 再帰的に範囲内点を収集
      └─ 境界交差チェック（AABB_intersects）
      └─ Leaf: 各点を範囲内チェック
      └─ Branch: 子ノードを再帰検索
```

### フローチャート

```mermaid
flowchart TD
    A[Create呼び出し] --> B[protInit]
    B --> C[Leafノードとして初期化]
    C --> D[境界(AABB)設定]
    D --> E[stdQuadTree返却]

    F[insertPoint呼び出し] --> G{境界内?}
    G -->|No| H[False返却]
    G -->|Yes| I{ノード種別}
    I -->|Leaf| J{容量超過?}
    J -->|No| K[点を追加]
    J -->|Yes| L[SubDivide]
    L --> M[Branch化]
    M --> N[子ノードに再挿入]
    I -->|Branch| O[子ノードに再帰挿入]
    K --> P[True返却]
    N --> O
    O --> Q{挿入成功?}
    Q -->|Yes| P
    Q -->|No| R[次の子ノードへ]

    S[queryRange呼び出し] --> T[AABB作成]
    T --> U{境界交差?}
    U -->|No| V[空Collection返却]
    U -->|Yes| W{ノード種別}
    W -->|Leaf| X[各点を範囲チェック]
    W -->|Branch| Y[子ノードを再帰検索]
    X --> Z[結果収集]
    Y --> AA[結果結合]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 境界チェック | 境界外の点は挿入不可 | insertPoint時 |
| BR-002 | 容量制限 | capacityBeforeSubdivision超過で分割 | insertPoint時 |
| BR-003 | 最小セルサイズ | minimumHalfDimension未満では分割しない | insertPoint時 |
| BR-004 | 4象限分割 | NW, NE, SW, SEの4象限に分割 | SubDivide時 |
| BR-005 | 正方形セル | halfDimensionは正方形の半辺長 | 全操作 |

### Quadtree構造

```
         Root (Branch)
         /  |  \  \
       NW  NE  SW  SE
       |   |   |   |
     Leaf Leaf... ...
```

**象限定義**:
```
NW: (center.x - r, center.y + r)  NE: (center.x + r, center.y + r)
SW: (center.x - r, center.y - r)  SE: (center.x + r, center.y - r)
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本クラスはDBアクセスを行わない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | 挿入エラー | SubDivide時に点が挿入できない | "Could not be inserted?" |
| - | 範囲外 | 境界外への挿入試行 | False返却（エラーなし） |
| - | スタックオーバーフロー | 同一座標の過剰な重複 | minimumHalfDimensionで防止 |

### リトライ仕様

リトライ仕様なし

## トランザクション仕様

トランザクション管理なし（メモリ内操作のみ）

## パフォーマンス要件

- insertPoint: O(log n) 平均、O(n) 最悪（重複点多数時）
- queryRange: O(log n + k) （kは結果点数）
- SubDivide: O(capacityBeforeSubdivision)
- メモリ: O(n) （点数に比例）

**比較**:
| 操作 | 線形探索 | Quadtree |
|------|---------|----------|
| 範囲検索 | O(n) | O(log n + k) |
| 挿入 | O(1) | O(log n) |

## セキュリティ考慮事項

- 外部入力の座標値の検証は呼び出し側の責任
- 大量の重複点によるスタックオーバーフローに注意（minimumHalfDimensionで緩和）

## 備考

- VB_PredeclaredId = True により、stdQuadTreeをインスタンス化せずにファクトリメソッドを呼び出し可能
- AABB（Axis-Aligned Bounding Box）による効率的な境界判定
- Loose Quadtreeやグリッドベースの代替手法も検討可能（devNoteコメント参照）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、Quadtreeの内部データ構造（XY, AABB, BranchInfo, LeafInfo, TThis）を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdQuadTree.cls | `src/stdQuadTree.cls` | 30-34行目: XY型（点座標+メタデータ） |
| 1-2 | stdQuadTree.cls | `src/stdQuadTree.cls` | 35-38行目: AABB型（境界ボックス） |
| 1-3 | stdQuadTree.cls | `src/stdQuadTree.cls` | 40-48行目: BranchIndex, BranchInfo（子ノード） |
| 1-4 | stdQuadTree.cls | `src/stdQuadTree.cls` | 49-56行目: LeafInfo, QuadTreeType |
| 1-5 | stdQuadTree.cls | `src/stdQuadTree.cls` | 60-77行目: TThis型（全体状態） |

**読解のコツ**:
- `XY`は点座標とメタデータを保持
- `AABB`は中心座標とhalfDimension（半辺長）で境界を表現
- `QuadTreeType`がLeaf/Branchを区別
- `TThis`が全体の状態を保持

#### Step 2: ファクトリメソッドを理解する

Create関数とprotInit関数を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdQuadTree.cls | `src/stdQuadTree.cls` | 88-91行目: Createメソッド |
| 2-2 | stdQuadTree.cls | `src/stdQuadTree.cls` | 102-111行目: CreateFromBoundsメソッド |
| 2-3 | stdQuadTree.cls | `src/stdQuadTree.cls` | 122-125行目: CreateChildメソッド |
| 2-4 | stdQuadTree.cls | `src/stdQuadTree.cls` | 135-151行目: protInit初期化処理 |

**主要処理フロー**:
1. **88行目**: `Create` - 中心座標とhalfDimensionから生成
2. **102行目**: `CreateFromBounds` - min/maxからの境界計算
3. **135-151行目**: `protInit` - Leafノードとして初期化、AABB設定

#### Step 3: 挿入処理を理解する

insertPoint関数とSubDivide関数を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdQuadTree.cls | `src/stdQuadTree.cls` | 159-196行目: insertPoint関数 |
| 3-2 | stdQuadTree.cls | `src/stdQuadTree.cls` | 252-286行目: SubDivide関数 |

**主要処理フロー**:
- **161行目**: 境界チェック（AABB_containsXY）
- **163-181行目**: Leafノードへの点追加または分割判定
- **182-191行目**: Branchノードでの再帰挿入
- **252-286行目**: SubDivide - 4象限に分割、既存点を再配置

#### Step 4: 検索処理を理解する

queryRange関数とpQueryRange関数を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdQuadTree.cls | `src/stdQuadTree.cls` | 203-206行目: queryRangeメソッド |
| 4-2 | stdQuadTree.cls | `src/stdQuadTree.cls` | 214-222行目: queryRangeBoundsメソッド |
| 4-3 | stdQuadTree.cls | `src/stdQuadTree.cls` | 232-249行目: pQueryRange関数 |

**主要処理フロー**:
- **232-249行目**: pQueryRange - 再帰的な範囲検索
- **234行目**: 交差チェック（AABB_intersects）
- **237-243行目**: Leafノードでの点ごとのチェック
- **245-247行目**: Branchノードでの再帰検索

#### Step 5: AABB操作を理解する

境界ボックス（AABB）操作関数群を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | stdQuadTree.cls | `src/stdQuadTree.cls` | 305-309行目: AABB_Create |
| 5-2 | stdQuadTree.cls | `src/stdQuadTree.cls` | 325-331行目: AABB_containsPoint |
| 5-3 | stdQuadTree.cls | `src/stdQuadTree.cls` | 338-344行目: AABB_containsXY |
| 5-4 | stdQuadTree.cls | `src/stdQuadTree.cls` | 351-361行目: AABB_intersects |

### プログラム呼び出し階層図

```
stdQuadTree.Create(cx, cy, halfDimension)
    │
    └─ stdQuadTree.protInit(cx, cy, halfDimension, ...)
           └─ Type = QuadTreeLeaf
           └─ AABB境界設定

stdQuadTree.insertPoint(x, y, meta)
    │
    ├─ AABB_containsXY(boundary, x, y)
    │
    └─ [QuadTreeLeaf]
    │      ├─ 容量内: points配列に追加
    │      └─ 容量超過: SubDivide()
    │                    │
    │                    ├─ Type = QuadTreeBranch
    │                    ├─ CreateChild x 4 (NW, NE, SW, SE)
    │                    └─ 既存点を子ノードに再挿入
    │
    └─ [QuadTreeBranch]
           └─ 子ノード.insertPoint(x, y, meta) [再帰]

stdQuadTree.queryRange(cx, cy, radius)
    │
    ├─ AABB_Create(cx, cy, radius)
    │
    └─ pQueryRange(rect)
           │
           ├─ AABB_intersects(boundary, rect)
           │
           ├─ [QuadTreeLeaf]
           │      └─ AABB_containsPoint → 結果収集
           │
           └─ [QuadTreeBranch]
                  └─ 子ノード.queryRange [再帰]
```

### データフロー図

```
[入力]                      [処理]                          [出力]

Create(cx,cy,r) ──────▶ protInit ───────────────────▶ stdQuadTree
                              │                          (Leaf)
                              └─▶ AABB境界設定

insertPoint(x,y) ────▶ 境界チェック ─────────────────▶ Boolean
                              │
                              ├─▶ [Leaf] 点追加
                              │     └─▶ SubDivide [容量超過時]
                              │            └─▶ 4象限生成
                              │
                              └─▶ [Branch] 子ノードへ再帰

queryRange(cx,cy,r) ─▶ AABB交差チェック ────────────▶ Collection
                              │
                              ├─▶ [Leaf] 点ごとチェック
                              │
                              └─▶ [Branch] 子ノードへ再帰
                                      │
                                      └─▶ 結果マージ
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdQuadTree.cls | `src/stdQuadTree.cls` | ソース | Quadtreeデータ構造クラス |
