# 機能設計書 9-stdRegex

## 概要

本ドキュメントは、stdVBAライブラリにおける正規表現ラッパークラス `stdRegex` の機能設計を詳細に記述したものである。

### 本機能の処理概要

stdRegexは、VBScript.RegExpオブジェクトをラップし、名前付きキャプチャグループや拡張フラグなどの追加機能を提供するクラスである。正規表現によるテキスト検索、マッチング、置換、リスト生成などの機能を提供する。

**業務上の目的・背景**：VBA標準のVBScript.RegExpは名前付きキャプチャグループをサポートしていないなど機能が限定的である。stdRegexはこれらの制限を解消し、より使いやすい正規表現操作を可能にする。

**機能の利用シーン**：
- テキストデータからのパターンマッチング抽出
- ログファイルの解析
- データ検証（メールアドレス、電話番号等）
- テキストの置換・整形
- 構造化データの抽出とExcelへの出力

**主要な処理内容**：
1. 正規表現オブジェクトの生成とキャッシュ（Create, protInit）
2. パターンマッチングのテスト（Test）
3. 単一マッチの取得（Match）
4. 全マッチの取得（MatchAll）
5. パターン置換（Replace）
6. フォーマット付きリスト生成（List, ListArr）

**関連システム・外部連携**：VBScript.RegExpオブジェクトに依存。stdErrorによるエラーハンドリング（オプション）。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本クラスはUIを持たないライブラリクラス |

## 機能種別

テキスト処理 / 正規表現 / パターンマッチング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pattern | String | Yes | 正規表現パターン | 有効な正規表現 |
| flags | String | No | フラグ文字列（i,g,m,x,c,s） | - |
| sHaystack | String | Yes | 検索対象テキスト | - |
| sReplacer | String | Yes | 置換フォーマット（Replace時） | - |
| sFormat | String | Yes | 出力フォーマット（List時） | - |
| vFormats | Variant | Yes | フォーマット配列（ListArr時） | Array |

### 入力データソース

- 正規表現パターン文字列
- 検索対象テキスト文字列

### フラグオプション

| フラグ | 名称 | 説明 |
|-------|------|------|
| i | Case insensitive | 大文字小文字を区別しない |
| g | Global | 全マッチを対象とする |
| m | Multiline | 複数行モード |
| x | Ignore whitespace | 空白を無視 |
| c | Ignore comment | #以降をコメントとして無視 |
| s | Singleline | `.`が改行も含む全文字にマッチ |

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Test | Boolean | パターンマッチの有無 |
| Match | Dictionary | 単一マッチ結果（名前付きキー含む） |
| MatchAll | Collection | 全マッチ結果のコレクション |
| Replace | String | 置換後の文字列 |
| List | String | フォーマット適用後のリスト文字列 |
| ListArr | Variant | 2次元配列形式のマッチ結果 |

### Match/MatchAll戻り値構造

```
Dictionary {
  0: "全体マッチ文字列",
  1: "サブマッチ1",
  2: "サブマッチ2",
  ...
  "名前1": "名前付きキャプチャ1",
  "名前2": "名前付きキャプチャ2",
  "$COUNT": サブマッチ数,
  "$RAW": VBScript.RegExpマッチオブジェクト
}
```

## 処理フロー

### 処理シーケンス

```
1. Create(pattern, flags)
   └─ キャッシュチェック
      └─ 存在: キャッシュから返却
      └─ 不在: protInit呼び出し → キャッシュ登録

2. protInit(pattern, flags)
   └─ NamesMatcher: 名前付きキャプチャグループ検出
   └─ NamesReplacer: 名前構文を除去
   └─ This.regex = VBScript.RegExp生成
   └─ フラグ処理（i,g,m,x,c,s）

3. Test(sHaystack)
   └─ This.regex.Test呼び出し
   └─ Boolean返却

4. Match(sHaystack)
   └─ This.regex.Execute
   └─ Dictionary生成
   └─ 名前付きキー設定
   └─ Dictionary返却

5. MatchAll(sHaystack)
   └─ Global一時設定
   └─ This.regex.Execute
   └─ 各マッチをDictionary化
   └─ Collection返却

6. Replace(sHaystack, sReplacer)
   └─ $name → $n 変換
   └─ This.regex.Replace
   └─ 文字列返却

7. List(sHaystack, sFormat)
   └─ MatchAll呼び出し
   └─ FormatFromMatch適用
   └─ 結合文字列返却

8. ListArr(sHaystack, vFormats)
   └─ MatchAll呼び出し
   └─ 2次元配列生成
   └─ 配列返却
```

### フローチャート

```mermaid
flowchart TD
    A[Create呼び出し] --> B{キャッシュ存在?}
    B -->|Yes| C[キャッシュから返却]
    B -->|No| D[protInit呼び出し]
    D --> E[NamesMatcher実行]
    E --> F[NamesReplacer実行]
    F --> G[VBScript.RegExp生成]
    G --> H{フラグ処理}
    H -->|i| I[ignoreCase=True]
    H -->|g| J[Global=True]
    H -->|m| K[MultiLine=True]
    H -->|x| L[空白除去]
    H -->|c| M[コメント除去]
    H -->|s| N[. → (?:.|\s)]
    I --> O[キャッシュ登録]
    J --> O
    K --> O
    L --> O
    M --> O
    N --> O
    O --> C

    P[Test] --> Q[This.regex.Test]
    Q --> R[Boolean返却]

    S[Match] --> T[This.regex.Execute]
    T --> U[Dictionary生成]
    U --> V[名前付きキー設定]
    V --> W[Dictionary返却]

    X[MatchAll] --> Y[Global一時設定]
    Y --> Z[This.regex.Execute]
    Z --> AA[各マッチをDictionary化]
    AA --> AB[Collection返却]

    AC[Replace] --> AD[$name→$n変換]
    AD --> AE[This.regex.Replace]
    AE --> AF[文字列返却]

    AG[List] --> AH[MatchAll呼び出し]
    AH --> AI[FormatFromMatch]
    AI --> AJ[結合文字列返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 名前付きキャプチャ | `(?<name>...)` 構文で名前付きキャプチャ | pattern内 |
| BR-002 | キャッシュ | 同一pattern+flagsはキャッシュから返却 | Create時 |
| BR-003 | インスタンスチェック | classでのメソッド呼び出しはエラー | 全メソッド |
| BR-004 | 名前置換 | $nameを$nに変換してReplace実行 | Replace時 |
| BR-005 | エスケープ処理 | SaveEscape/LoadEscapeで特殊文字保護 | フラグ処理時 |
| BR-006 | Singlelineモード | `.`を`(?:.|\s)`に置換 | sフラグ時 |

### 名前付きキャプチャグループ構文

```
標準VBScript.RegExp: サポートなし
stdRegex: (?<name>pattern) で名前付きキャプチャ

例：
  pattern: "(?<Site>\d{5})-(\w{2}\d)"
  対象: "12345-AA1"
  結果: {0:"12345-AA1", 1:"12345", 2:"AA1", "Site":"12345"}
```

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | 本クラスはDBアクセスを行わない |

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 1 | CriticalRaise | classでCreate以外のメソッド呼び出し | "Method/Property called on class not object" |
| 1 | CriticalRaise | objectでCreate呼び出し | "Constructor called on object not class" |
| 1 | CriticalRaise | classでprotInit呼び出し | "Cannot run init on class" |
| - | VBScript.RegExpエラー | 無効な正規表現パターン | VBAランタイムエラー |

### リトライ仕様

リトライ仕様なし

## トランザクション仕様

トランザクション管理なし（メモリ内操作のみ）

## パフォーマンス要件

- Create: O(1)（キャッシュヒット時）/ O(n)（初回パターン解析時）
- Test: O(n)（VBScript.RegExpに依存）
- Match: O(n)（VBScript.RegExpに依存）
- MatchAll: O(n*m)（n=文字列長, m=マッチ数）
- キャッシュにより同一パターンの再利用が高速化

## セキュリティ考慮事項

- 外部入力パターンによるReDoS（正規表現DoS）攻撃に注意
- パターンのバリデーションは呼び出し側の責任
- ユーザー入力を直接パターンに使用しない

## 備考

- VB_PredeclaredId = True により、stdRegexをインスタンス化せずにファクトリメソッドを呼び出し可能
- Static変数によるNamesMatcher, NamesReplacer等の初期化コスト削減
- SaveEscape/LoadEscapeでGUIDベースのプレースホルダーを使用してエスケープ処理
- stdErrorが利用可能な場合はstdError.Raiseを使用、そうでなければErr.Raise

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、stdRegexの内部データ構造（TThis型）を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdRegex.cls | `src/stdRegex.cls` | 89-97行目: TThis型定義 |

**読解のコツ**:
- `initialised`: 初期化済みフラグ
- `pattern`: 元のパターン文字列
- `cPattern`: コンパイル済みパターン（現在未使用）
- `flags`: フラグ文字列
- `namesDict`: 名前付きキャプチャの辞書
- `regex`: VBScript.RegExpオブジェクト

#### Step 2: ファクトリメソッドを理解する

Create関数とprotInit関数を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdRegex.cls | `src/stdRegex.cls` | 109-130行目: Createメソッド |
| 2-2 | stdRegex.cls | `src/stdRegex.cls` | 116行目: Staticキャッシュ |
| 2-3 | stdRegex.cls | `src/stdRegex.cls` | 142-226行目: protInit |
| 2-4 | stdRegex.cls | `src/stdRegex.cls` | 156-161行目: NamesMatcher初期化 |
| 2-5 | stdRegex.cls | `src/stdRegex.cls` | 164-169行目: NamesReplacer初期化 |
| 2-6 | stdRegex.cls | `src/stdRegex.cls` | 203-204行目: VBScript.RegExp生成 |
| 2-7 | stdRegex.cls | `src/stdRegex.cls` | 207-223行目: フラグ処理ループ |

**主要処理フロー**:
1. **116行目**: Static cacheでDictionaryキャッシュを管理
2. **117行目**: pattern & "&" & flags をキーとして使用
3. **159行目**: 名前付きキャプチャを検出する正規表現
4. **204行目**: 名前構文を除去してVBScript.RegExpに設定
5. **210-221行目**: 各フラグ(i,g,m,x,c,s)に応じた処理

#### Step 3: マッチング操作を理解する

Test, Match, MatchAllメソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdRegex.cls | `src/stdRegex.cls` | 286-293行目: Testメソッド |
| 3-2 | stdRegex.cls | `src/stdRegex.cls` | 322-354行目: Matchメソッド |
| 3-3 | stdRegex.cls | `src/stdRegex.cls` | 380-427行目: MatchAllメソッド |
| 3-4 | stdRegex.cls | `src/stdRegex.cls` | 344-349行目: 名前付きキー設定 |

**主要処理フロー**:
- **292行目**: Test - VBScript.RegExp.Testを直接呼び出し
- **330行目**: Match - Executeで結果取得
- **338-340行目**: Match - 基本プロパティ(0, $COUNT, $RAW)設定
- **347行目**: Match - 名前付きキャプチャをDictionaryに設定
- **387-395行目**: MatchAll - Global一時設定と復元

#### Step 4: 置換・リスト生成を理解する

Replace, List, ListArrメソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdRegex.cls | `src/stdRegex.cls` | 455-474行目: Replaceメソッド |
| 4-2 | stdRegex.cls | `src/stdRegex.cls` | 498-517行目: Listメソッド |
| 4-3 | stdRegex.cls | `src/stdRegex.cls` | 525-553行目: ListArrメソッド |
| 4-4 | stdRegex.cls | `src/stdRegex.cls` | 564-591行目: FormatFromMatch |

**主要処理フロー**:
- **468行目**: Replace - $name → $n 変換
- **473行目**: Replace - VBScript.RegExp.Replaceに委譲
- **506行目**: List - MatchAllで全マッチ取得
- **512行目**: List - FormatFromMatchでフォーマット適用
- **538行目**: ListArr - 2次元配列ReDim
- **572-577行目**: FormatFromMatch - \r, \nなどのエスケープ処理

### プログラム呼び出し階層図

```
stdRegex.Create(pattern, flags)
    │
    ├─ キャッシュチェック (116-118行目)
    │
    └─ stdRegex.protInit(pattern, flags)
           │
           ├─ NamesMatcher.Execute (199行目)
           │     └─ 名前付きキャプチャ検出
           │
           ├─ NamesReplacer.Replace (204行目)
           │     └─ 名前構文除去
           │
           └─ フラグ処理 (207-223行目)
                 ├─ i: ignoreCase = True
                 ├─ g: Global = True
                 ├─ m: MultiLine = True
                 ├─ x: OptionIgnoreWhiteSpace
                 ├─ c: OptionIgnoreComment
                 └─ s: OptionSingleLineMode

stdRegex.Test(sHaystack)
    └─ This.regex.Test (292行目)

stdRegex.Match(sHaystack)
    │
    ├─ This.regex.Execute (330行目)
    │
    └─ Dictionary生成 (335-349行目)
          ├─ 基本プロパティ設定
          └─ 名前付きキー設定

stdRegex.MatchAll(sHaystack)
    │
    ├─ Global一時設定 (387-388行目)
    │
    ├─ This.regex.Execute (392行目)
    │
    ├─ Global復元 (395行目)
    │
    └─ Collection生成 (399-423行目)
          └─ 各マッチをDictionary化

stdRegex.Replace(sHaystack, sReplacer)
    │
    ├─ $name → $n 変換 (462-470行目)
    │
    └─ This.regex.Replace (473行目)

stdRegex.List(sHaystack, sFormat)
    │
    ├─ MatchAll (506行目)
    │
    └─ FormatFromMatch (512行目)
          ├─ $& → 全体マッチ
          ├─ \r, \n エスケープ
          └─ $name, $n 置換

stdRegex.ListArr(sHaystack, vFormats)
    │
    ├─ MatchAll (533行目)
    │
    └─ 2次元配列構築 (538-549行目)
          └─ FormatFromMatch適用
```

### データフロー図

```
[入力]                      [処理]                          [出力]

Create(pattern,flags) ──▶ キャッシュチェック ──────────▶ stdRegex
                              │                          (キャッシュ済み)
                              └─▶ protInit
                                    ├─▶ NamesMatcher
                                    ├─▶ NamesReplacer
                                    └─▶ フラグ処理

Test(sHaystack) ────────▶ This.regex.Test ─────────────▶ Boolean

Match(sHaystack) ───────▶ This.regex.Execute ──────────▶ Dictionary
                              │                          {0, 1, 2, ...
                              └─▶ 名前付きキー設定          "name1", ...}

MatchAll(sHaystack) ────▶ This.regex.Execute ──────────▶ Collection
                              │                          [Dict1, Dict2, ...]
                              └─▶ 各マッチDictionary化

Replace(sHaystack, ─────▶ $name→$n変換 ────────────────▶ String
        sReplacer)            │                          (置換済み)
                              └─▶ This.regex.Replace

List(sHaystack, ────────▶ MatchAll ────────────────────▶ String
     sFormat)                 │                          (フォーマット済み)
                              └─▶ FormatFromMatch

ListArr(sHaystack, ─────▶ MatchAll ────────────────────▶ Variant()
        vFormats)             │                          [2次元配列]
                              └─▶ 2次元配列構築
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdRegex.cls | `src/stdRegex.cls` | ソース | 正規表現ラッパークラス |
| stdError.cls | `src/stdError.cls` | ソース | エラーハンドリング（オプション依存） |
