# 通知設計書 1-BeforeRun

## 概要

本ドキュメントは、stdVBAライブラリのstdCallbackクラスで提供されるBeforeRunイベントの設計仕様を定義する。このイベントはコールバック関数が実行される直前に発火し、実行前の処理介入やログ記録を可能にする。

### 本通知の処理概要

BeforeRunイベントは、stdCallbackクラスのRunExメソッド内でコールバック関数が実際に実行される直前に発火するVBAイベントである。このイベントにより、コールバック実行前の引数検証、ロギング、前処理などが可能になる。

**業務上の目的・背景**：VBAでコールバックパターンを実装する際、関数実行前にフック処理を挿入したいケースが多い。例えば、デバッグ時の引数ログ出力、バリデーション処理、AOP（アスペクト指向プログラミング）的な横断的関心事の実装などが挙げられる。BeforeRunイベントはこれらのニーズに応え、コールバック実行のライフサイクルに介入可能なポイントを提供する。

**通知の送信タイミング**：stdCallbackオブジェクトのRunまたはRunExメソッドが呼び出され、実際のコールバック関数（モジュール関数、オブジェクトメソッド、関数ポインタなど）が実行される直前。具体的にはstdCallback.cls行383（RunExメソッド内のRaiseEvent BeforeRun）で発火する。

**通知の受信者**：stdCallbackオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。開発者が明示的にイベントハンドラを実装する必要がある。

**通知内容の概要**：イベントハンドラには、実行されようとしているstdCallbackオブジェクト自身（callback引数）と、渡される予定の引数配列（args引数）がByRefで渡される。これにより、受信側でargs配列を変更してコールバックに渡す引数を動的に変更することも可能。

**期待されるアクション**：開発者はこのイベントを利用して、引数のバリデーション、ログ記録、条件付き実行制御、引数の変換・加工などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventによる同期呼び出し） |
| 優先度 | 高（コールバック実行前に必ず発火） |
| リトライ | 無（単発のイベント発火） |

### 送信先決定ロジック

WithEventsキーワードを使用してstdCallbackオブジェクトを宣言しているクラスモジュール内のイベントハンドラ（Sub callback_BeforeRun形式）に対して送信される。VBA標準のイベント機構に従い、イベントハンドラが定義されている場合のみ処理が実行される。

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | BeforeRun |
| クラス名 | stdCallback |
| シグネチャ | Public Event BeforeRun(ByRef callback As stdCallback, ByRef args As Variant) |

### イベントハンドラテンプレート

```vba
Private WithEvents myCallback As stdCallback

Private Sub myCallback_BeforeRun(ByRef callback As stdCallback, ByRef args As Variant)
    ' コールバック実行前の処理を記述
    Debug.Print "Callback about to run with " & UBound(args) + 1 & " arguments"
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| callback | 実行されるstdCallbackオブジェクト | RunExメソッドのMeキーワード | Yes |
| args | コールバックに渡される引数配列 | RunExメソッドのvArr変数 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| メソッド呼び出し | Run()またはRunEx()メソッド | stdCallbackが初期化済み | コールバックの実行メソッドが呼ばれた時点で発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| 未初期化状態 | This.InitialisedがFalseの場合、RunExは実行されずイベントも発火しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[Run/RunExメソッド呼び出し] --> B{初期化済み?}
    B -->|No| C[CriticalRaiseエラー]
    B -->|Yes| D{Boundタイプ?}
    D -->|Yes| E[引数配列をマージ]
    D -->|No| F[引数配列をそのまま使用]
    E --> G[RaiseEvent BeforeRun]
    F --> G
    G --> H[イベントハンドラ実行]
    H --> I[コールバック関数実行]
    I --> J[RaiseEvent AfterRun]
    J --> K[終了]
    C --> L[処理中断]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 未初期化エラー | stdCallbackが未初期化のままRunを呼び出し | "No callback specified"エラーがCriticalRaiseで発生 |
| イベントハンドラエラー | イベントハンドラ内でエラー発生 | VBA標準のエラーハンドリングに従う |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（VBAイベントのため制限なし） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（プログラム実行時に随時発火）

## セキュリティ考慮事項

- イベントハンドラはByRefで引数を受け取るため、引数の意図しない変更に注意が必要
- コールバックオブジェクトへの参照が渡されるため、不正なオブジェクト操作を防ぐ必要がある
- 外部から渡された引数の検証をBeforeRunハンドラで行うことを推奨

## 備考

- BeforeRunイベントはAfterRunイベントと対になっており、コールバック実行のライフサイクル全体を監視可能
- args引数をイベントハンドラ内で変更すると、実際のコールバック実行に影響する（ByRef渡し）
- stdICallableインターフェースを実装しているため、汎用的なコールバック機構として利用可能

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

stdCallbackクラスの内部データ構造を理解することが、イベントの動作を把握する第一歩である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdCallback.cls | `src/stdCallback.cls` | 行132-162のParentType列挙型とCallbackStruct構造体を理解する |
| 1-2 | stdCallback.cls | `src/stdCallback.cls` | 行148-157のCallbackStruct型でコールバック情報の保持方法を確認 |

**読解のコツ**: VBAのType文でユーザー定義型を定義している。ParentTypeはコールバックの種類（Module/Object/Formula/Bound/Pointer）を表し、CallbackStructはそれぞれの種類に応じたデータを保持する。

#### Step 2: エントリーポイントを理解する

イベントが発火するエントリーポイントであるRunExメソッドを特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdCallback.cls | `src/stdCallback.cls` | 行367-371のRunメソッドがユーザー向けエントリーポイント |
| 2-2 | stdCallback.cls | `src/stdCallback.cls` | 行376-522のRunExメソッドが実際の実行ロジック |

**主要処理フロー**:
1. **行377**: 初期化状態とコールバックタイプのチェック
2. **行378-380**: Boundタイプの場合、引数配列をマージ
3. **行383**: RaiseEvent BeforeRun(Me, vArr) - ここでBeforeRunイベント発火
4. **行387-509**: コールバックタイプに応じた実行処理
5. **行512**: RaiseEvent AfterRun(Me, vRet)

#### Step 3: イベント宣言を理解する

BeforeRunイベントの宣言と引数を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdCallback.cls | `src/stdCallback.cls` | 行166-169のBeforeRunイベント宣言 |

**主要処理フロー**:
- **行169**: `Public Event BeforeRun(ByRef callback As stdCallback, ByRef args As Variant)` - イベント宣言

### プログラム呼び出し階層図

```
ユーザーコード
    │
    ├─ stdCallback.Run(args...)
    │      │
    │      └─ stdCallback.RunEx(vArr)
    │             │
    │             ├─ RaiseEvent BeforeRun(Me, vArr)  ← イベント発火
    │             │      │
    │             │      └─ [イベントハンドラ実行]
    │             │
    │             ├─ [コールバック関数実行]
    │             │
    │             └─ RaiseEvent AfterRun(Me, vRet)
    │
    └─ [戻り値受け取り]
```

### データフロー図

```
[入力]               [処理]                    [出力]

args(引数配列) ───▶ RunExメソッド ───▶ BeforeRunイベント
                        │                    │
                        │                    ├─▶ callback (stdCallback参照)
                        │                    └─▶ args (引数配列参照)
                        │
                        └─▶ コールバック実行 ───▶ 戻り値
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdCallback.cls | `src/stdCallback.cls` | ソース | BeforeRunイベントを定義・発火するメインクラス |
| stdICallable.cls | `src/stdICallable.cls` | ソース | コールバックインターフェース定義 |
