# 通知設計書 10-RangeSelectionChange

## 概要

本ドキュメントは、stdVBAライブラリのstdXLEventsクラスで提供されるRangeSelectionChangeイベントの設計仕様を定義する。このイベントはExcelのセル範囲（Range）の選択が変更された際に発火し、セル選択に特化した処理を可能にする。

### 本通知の処理概要

RangeSelectionChangeイベントは、stdXLEventsクラスで実装されたセル範囲選択専用のイベントである。SelectionChangeイベントの発火後、選択がRange型の場合にのみ発火する。このイベントにより、セル選択変更に特化した処理をシンプルに実装できる。

**業務上の目的・背景**：Excelではセル範囲の選択変更は最も頻繁に発生する操作である。stdXLEventsの汎用的なSelectionChangeイベントでは全ての選択タイプを扱うが、多くのケースではセル選択のみに関心がある。RangeSelectionChangeイベントはこのニーズに応え、Range型の選択変更のみを効率的に監視する機能を提供する。これにより、セル選択に応じたプロパティパネルの更新、条件付き書式のプレビュー、選択範囲の統計表示などの機能を実装できる。

**通知の送信タイミング**：セル範囲の選択が変更され、旧選択または新選択のいずれかがRange型の場合。具体的にはstdXLEvents.cls行88（cbs_OnUpdateイベントハンドラ内のRaiseEvent RangeSelectionChange）で発火する。

**通知の受信者**：stdXLEventsオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。

**通知内容の概要**：イベントハンドラには、旧選択Range（rOldRange引数）と新選択Range（rNewRange引数）が渡される。選択がRange型でない場合、対応する引数はNothingとなる。

**期待されるアクション**：開発者はこのイベントを利用して、選択セルの統計情報表示、セル選択に応じたUIの更新、選択範囲の検証、自動計算の実行などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventによる同期呼び出し） |
| 優先度 | 中（SelectionChange後に発火） |
| リトライ | 無（単発のイベント発火） |

### 送信先決定ロジック

WithEventsキーワードを使用してstdXLEventsオブジェクトを宣言しているクラスモジュール内のイベントハンドラ（Sub events_RangeSelectionChange形式）に対して送信される。

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | RangeSelectionChange |
| クラス名 | stdXLEvents |
| シグネチャ | Public Event RangeSelectionChange(rOldRange As Range, rNewRange As Range) |

### イベントハンドラテンプレート

```vba
Private WithEvents xlEvents As stdXLEvents

Private Sub Class_Initialize()
    Set xlEvents = New stdXLEvents
End Sub

Private Sub xlEvents_RangeSelectionChange(rOldRange As Range, rNewRange As Range)
    ' セル範囲選択変更時の処理

    ' 旧選択範囲の確認
    If Not rOldRange Is Nothing Then
        Debug.Print "Previous selection: " & rOldRange.Address
    End If

    ' 新選択範囲の確認
    If Not rNewRange Is Nothing Then
        Debug.Print "New selection: " & rNewRange.Address

        ' 選択範囲の統計情報表示
        lblCellCount.Caption = "Cells: " & rNewRange.Cells.Count
        lblRowCount.Caption = "Rows: " & rNewRange.Rows.Count
        lblColCount.Caption = "Columns: " & rNewRange.Columns.Count

        ' 数値データの統計
        On Error Resume Next
        lblSum.Caption = "Sum: " & Application.Sum(rNewRange)
        lblAvg.Caption = "Avg: " & Application.Average(rNewRange)
        On Error GoTo 0
    Else
        ' Range以外が選択された場合（Shapeなど）
        Debug.Print "Non-range selection"
        Call ClearRangeStats()
    End If
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| rOldRange | 旧選択Range（Range型でない場合はNothing） | oldSelection.oObj（Range型の場合） | No |
| rNewRange | 新選択Range（Range型でない場合はNothing） | newSelection.oObj（Range型の場合） | No |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| セル選択変更 | Range選択 | 旧または新がRange型 | セルの選択が変更された時 |
| Range→Shape | 選択タイプ変更 | 旧がRange型 | Rangeから別のオブジェクトに選択が変わった時 |
| Shape→Range | 選択タイプ変更 | 新がRange型 | 別のオブジェクトからRangeに選択が変わった時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| Range以外同士の変更 | 旧選択も新選択もRange型でない場合、イベントは発火しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[SelectionChange発火後] --> B{旧or新がRange?}
    B -->|No| C[終了・イベントなし]
    B -->|Yes| D[RaiseEvent RangeSelectionChange]
    D --> E[イベントハンドラ実行]
    E --> F{新がRange?}
    F -->|Yes| G{旧もRange?}
    F -->|No| H[RaiseEvent RangeDeselected]
    G -->|Yes| I[RaiseEvent RangeDeselected旧]
    G -->|No| J[スキップ]
    I --> K[RaiseEvent RangeSelected新]
    J --> K
    H --> L[終了]
    K --> L
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| Range参照エラー | 削除されたセルへの参照 | 例外処理でNothingに置換 |
| ワークシート変更 | 別シートへの切り替え | 正常にイベント発火 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（セル選択変更に応じて発火） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（セル選択変更時に発火）

## セキュリティ考慮事項

- rOldRange/rNewRangeを通じてワークブックやワークシートにアクセス可能なため、操作には注意
- 大量セル選択時はパフォーマンスに注意（Cells.Countの評価など）

## 備考

- RangeSelectionChangeはSelectionChangeイベントの後に発火する
- Range→Range、Range→Shape、Shape→Rangeの3パターンで発火する
- RangeSelected/RangeDeselectedイベントも併せて発火する
- stdXLEventsはWIP（Work In Progress）ディレクトリに配置されており、開発中の機能
- IIfを使用して条件付きでRange参照またはNothingを渡している

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: イベント宣言を理解する

RangeSelectionChangeイベントの宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行12のRangeSelectionChangeイベント宣言 |
| 1-2 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行13-14のRangeSelected/RangeDeselectedイベント宣言 |

**読解のコツ**: RangeSelectionChangeはRange型の引数を持つが、選択がRange以外の場合はNothingが渡される。

#### Step 2: 選択タイプ判定を理解する

ESelectionType列挙型でのRange判定を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行27-66のESelectionType列挙型 |
| 2-2 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行257-301のgetObjTypeメソッド |

**主要処理フロー**:
- **行56**: Range = ESelectionType.Rangeとして定義
- **行260**: TypeName(obj) = "Range"の場合、ESelectionType.Rangeを返す

#### Step 3: イベント発火処理を理解する

cbs_OnUpdateでのRangeSelectionChange発火ロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行86-97のRange関連イベント発火処理 |

**主要処理フロー**:
- **行87**: `If oldSelection.iType = Range Or newSelection.iType = Range Then` - Range関連チェック
- **行88**: `RaiseEvent RangeSelectionChange(...)` - RangeSelectionChange発火
- **行88**: IIf関数で条件付きRange/Nothing渡し
- **行89-96**: RangeSelected/RangeDeselectedの条件付き発火

### プログラム呼び出し階層図

```
[SelectionChange発火後]
    │
    └─ cbs_OnUpdate（行86-97）
           │
           └─ If oldSelection.iType = Range Or newSelection.iType = Range
                  │
                  └─ RaiseEvent RangeSelectionChange  ← イベント発火
                         │
                         ├─▶ rOldRange = IIf(oldType=Range, oldObj, Nothing)
                         │
                         └─▶ rNewRange = IIf(newType=Range, newObj, Nothing)
                                │
                                └─ [後続処理]
                                       │
                                       ├─ RangeDeselected（旧がRangeの場合）
                                       │
                                       └─ RangeSelected（新がRangeの場合）
```

### データフロー図

```
[入力]                      [処理]                    [出力]

SelectionChangeイベント
    │
    └─▶ Range判定
              │
              ├─ oldSelection.iType == Range?
              │
              └─ newSelection.iType == Range?
                        │
                        └─▶ [いずれかがRange]
                                │
                                └─▶ RangeSelectionChangeイベント
                                       │
                                       ├─▶ rOldRange (Range or Nothing)
                                       │
                                       └─▶ rNewRange (Range or Nothing)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | ソース | RangeSelectionChangeイベントを定義・発火するメインクラス |
