# 通知設計書 2-AfterRun

## 概要

本ドキュメントは、stdVBAライブラリのstdCallbackクラスで提供されるAfterRunイベントの設計仕様を定義する。このイベントはコールバック関数が実行された直後に発火し、実行結果の処理介入やログ記録を可能にする。

### 本通知の処理概要

AfterRunイベントは、stdCallbackクラスのRunExメソッド内でコールバック関数が実行された直後に発火するVBAイベントである。このイベントにより、コールバック実行後の結果検証、ロギング、後処理、結果の変更などが可能になる。

**業務上の目的・背景**：VBAでコールバックパターンを実装する際、関数実行後にフック処理を挿入したいケースが多い。例えば、戻り値のログ出力、結果の検証・加工、キャッシュ処理、パフォーマンス計測の終了処理などが挙げられる。AfterRunイベントはこれらのニーズに応え、コールバック実行のライフサイクルに介入可能なポイントを提供する。BeforeRunイベントと組み合わせることで、AOP（アスペクト指向プログラミング）的な処理が可能となる。

**通知の送信タイミング**：stdCallbackオブジェクトのRunExメソッド内で、コールバック関数（モジュール関数、オブジェクトメソッド、関数ポインタなど）が実行された直後。具体的にはstdCallback.cls行512（RunExメソッド内のRaiseEvent AfterRun）で発火する。

**通知の受信者**：stdCallbackオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。開発者が明示的にイベントハンドラを実装する必要がある。

**通知内容の概要**：イベントハンドラには、実行されたstdCallbackオブジェクト自身（callback引数）と、コールバックの戻り値（result引数）がByRefで渡される。これにより、受信側でresultを変更して最終的な戻り値を動的に変更することも可能。

**期待されるアクション**：開発者はこのイベントを利用して、戻り値の検証、ログ記録、結果の加工・変換、キャッシュへの保存、パフォーマンス計測などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventによる同期呼び出し） |
| 優先度 | 高（コールバック実行後に必ず発火） |
| リトライ | 無（単発のイベント発火） |

### 送信先決定ロジック

WithEventsキーワードを使用してstdCallbackオブジェクトを宣言しているクラスモジュール内のイベントハンドラ（Sub callback_AfterRun形式）に対して送信される。VBA標準のイベント機構に従い、イベントハンドラが定義されている場合のみ処理が実行される。

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | AfterRun |
| クラス名 | stdCallback |
| シグネチャ | Public Event AfterRun(ByRef callback As stdCallback, ByRef result As Variant) |

### イベントハンドラテンプレート

```vba
Private WithEvents myCallback As stdCallback

Private Sub myCallback_AfterRun(ByRef callback As stdCallback, ByRef result As Variant)
    ' コールバック実行後の処理を記述
    Debug.Print "Callback completed with result: " & CStr(result)

    ' 必要に応じて結果を変更可能
    ' result = ProcessResult(result)
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| callback | 実行されたstdCallbackオブジェクト | RunExメソッドのMeキーワード | Yes |
| result | コールバックの戻り値 | RunExメソッドのvRet変数 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| メソッド呼び出し | Run()またはRunEx()メソッド完了 | コールバック実行が正常完了 | コールバックの実行が完了した時点で発火 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| 未初期化状態 | This.InitialisedがFalseの場合、RunExは実行されずイベントも発火しない |
| 実行時エラー | コールバック実行中にエラーが発生した場合、AfterRunは発火しない可能性がある |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[Run/RunExメソッド呼び出し] --> B{初期化済み?}
    B -->|No| C[CriticalRaiseエラー]
    B -->|Yes| D[RaiseEvent BeforeRun]
    D --> E{コールバックタイプ判定}
    E -->|Module| F[Application.Run実行]
    E -->|Object| G[rtcCallByName実行]
    E -->|Bound| H[再帰的RunEx実行]
    E -->|Pointer| I[DispCallFunc実行]
    F --> J[vRetに結果格納]
    G --> J
    H --> J
    I --> J
    J --> K[RaiseEvent AfterRun]
    K --> L[イベントハンドラ実行]
    L --> M[CopyVariant RunEx, vRet]
    M --> N[終了]
    C --> O[処理中断]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 未初期化エラー | stdCallbackが未初期化のままRunを呼び出し | "No callback specified"エラーがCriticalRaiseで発生 |
| イベントハンドラエラー | イベントハンドラ内でエラー発生 | VBA標準のエラーハンドリングに従う |
| コールバック実行エラー | コールバック関数内でエラー発生 | エラーラベル(ErrorInRTCCallByName)でCriticalRaiseに転送 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（VBAイベントのため制限なし） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（プログラム実行時に随時発火）

## セキュリティ考慮事項

- イベントハンドラはByRefでresultを受け取るため、戻り値の意図しない変更に注意が必要
- コールバックオブジェクトへの参照が渡されるため、不正なオブジェクト操作を防ぐ必要がある
- 機密情報が戻り値に含まれる場合、ログ出力時に注意が必要

## 備考

- AfterRunイベントはBeforeRunイベントと対になっており、コールバック実行のライフサイクル全体を監視可能
- result引数をイベントハンドラ内で変更すると、最終的な戻り値に影響する（ByRef渡し）
- BeforeRunで開始したパフォーマンス計測をAfterRunで終了するなどの用途に活用可能
- stdICallableインターフェースを実装しているため、汎用的なコールバック機構として利用可能

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

stdCallbackクラスの内部データ構造を理解することが、イベントの動作を把握する第一歩である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdCallback.cls | `src/stdCallback.cls` | 行132-162のParentType列挙型とCallbackStruct構造体を理解する |
| 1-2 | stdCallback.cls | `src/stdCallback.cls` | 行158-162のTThis型で状態管理を確認 |

**読解のコツ**: VBAのType文でユーザー定義型を定義している。TThis.Callbackには実行対象のコールバック情報が格納される。

#### Step 2: エントリーポイントを理解する

イベントが発火するエントリーポイントであるRunExメソッドを特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdCallback.cls | `src/stdCallback.cls` | 行367-371のRunメソッドがユーザー向けエントリーポイント |
| 2-2 | stdCallback.cls | `src/stdCallback.cls` | 行376-522のRunExメソッドが実際の実行ロジック |

**主要処理フロー**:
1. **行377**: 初期化状態とコールバックタイプのチェック
2. **行383**: RaiseEvent BeforeRun(Me, vArr)
3. **行386-509**: コールバックタイプに応じた実行処理、結果をvRetに格納
4. **行512**: RaiseEvent AfterRun(Me, vRet) - ここでAfterRunイベント発火
5. **行515**: CopyVariant RunEx, vRet - 戻り値を設定

#### Step 3: イベント宣言を理解する

AfterRunイベントの宣言と引数を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdCallback.cls | `src/stdCallback.cls` | 行171-174のAfterRunイベント宣言 |

**主要処理フロー**:
- **行174**: `Public Event AfterRun(ByRef callback As stdCallback, ByRef result As Variant)` - イベント宣言

### プログラム呼び出し階層図

```
ユーザーコード
    │
    ├─ stdCallback.Run(args...)
    │      │
    │      └─ stdCallback.RunEx(vArr)
    │             │
    │             ├─ RaiseEvent BeforeRun(Me, vArr)
    │             │
    │             ├─ [コールバック関数実行] → vRet
    │             │
    │             ├─ RaiseEvent AfterRun(Me, vRet)  ← イベント発火
    │             │      │
    │             │      └─ [イベントハンドラ実行]
    │             │             （resultの変更可能）
    │             │
    │             └─ return vRet
    │
    └─ [戻り値受け取り]
```

### データフロー図

```
[入力]               [処理]                    [出力]

args(引数配列) ───▶ RunExメソッド
                        │
                        ├─▶ コールバック実行 ───▶ vRet(戻り値)
                        │                            │
                        │                            ▼
                        │                    AfterRunイベント
                        │                            │
                        │                            ├─▶ callback (stdCallback参照)
                        │                            └─▶ result (戻り値参照、変更可能)
                        │
                        └─▶ 最終戻り値（result変更後の値）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdCallback.cls | `src/stdCallback.cls` | ソース | AfterRunイベントを定義・発火するメインクラス |
| stdICallable.cls | `src/stdICallable.cls` | ソース | コールバックインターフェース定義 |
