# 通知設計書 3-EventRaised

## 概要

本ドキュメントは、stdVBAライブラリのstdWebSocketクラスで提供されるEventRaisedイベントの設計仕様を定義する。このイベントはWebSocket通信における全てのイベント（Open, Close, Error, Message）が発生した際に共通して発火する汎用イベントである。

### 本通知の処理概要

EventRaisedイベントは、stdWebSocketクラスのhandleEventメソッド内で、WebSocketの各種イベント（Open, Close, Error, Message）が発生した際に最初に発火するVBAイベントである。このイベントにより、イベントタイプに関わらず一元的にWebSocket通信イベントを監視・処理することが可能になる。

**業務上の目的・背景**：WebSocket通信では複数のイベントタイプ（接続開始、切断、エラー、メッセージ受信）が存在するが、ログ記録やモニタリングなど全てのイベントに対して共通処理を行いたいケースが多い。EventRaisedイベントはこれらのニーズに応え、個別のイベント（OnOpen, OnClose等）とは別に、全イベントを一箇所で処理可能なハブとして機能する。これによりイベントハンドラの重複コードを削減できる。

**通知の送信タイミング**：WebSocket接続がopen, close, error, messageのいずれかのイベントを受信した際。具体的にはstdWebSocket.cls行153（handleEventメソッド内のRaiseEvent EventRaised）で発火する。JavaScriptのWebSocket APIからイベントが発生すると、handleEventメソッドが呼び出され、そこでEventRaisedが発火する。

**通知の受信者**：stdWebSocketオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。また、ListenersコレクションにstdICallableオブジェクトを追加している場合も受信対象となる。

**通知内容の概要**：イベントハンドラには、イベント名（sEventName引数："Open", "Close", "Error", "Message"のいずれか）と、JavaScriptから渡されるイベントデータ（eventData引数）が渡される。

**期待されるアクション**：開発者はこのイベントを利用して、全WebSocketイベントの一元ログ記録、イベント統計収集、共通エラーハンドリング、デバッグ出力などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventによる同期呼び出し） |
| 優先度 | 高（WebSocketイベント発生時に最初に発火） |
| リトライ | 無（単発のイベント発火） |

### 送信先決定ロジック

1. WithEventsキーワードを使用してstdWebSocketオブジェクトを宣言しているクラスモジュール内のイベントハンドラ
2. Listenersコレクションに追加されたstdICallableオブジェクトのRunメソッド

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | EventRaised |
| クラス名 | stdWebSocket |
| シグネチャ | Public Event EventRaised(ByVal sEventName As String, ByVal eventData As Variant) |

### イベントハンドラテンプレート

```vba
Private WithEvents ws As stdWebSocket

Private Sub ws_EventRaised(ByVal sEventName As String, ByVal eventData As Variant)
    ' 全WebSocketイベントの共通処理
    Debug.Print "[" & Now & "] WebSocket Event: " & sEventName

    ' イベントタイプに応じた処理分岐も可能
    Select Case sEventName
        Case "Open"
            Debug.Print "Connection established"
        Case "Close"
            Debug.Print "Connection closed"
        Case "Error"
            Debug.Print "Error occurred"
        Case "Message"
            Debug.Print "Message received: " & eventData.data
    End Select
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| sEventName | イベント名（"Open", "Close", "Error", "Message"） | handleEventメソッドの引数 | Yes |
| eventData | JavaScriptイベントオブジェクト | handleEventメソッドの引数（oEventData） | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| WebSocket onopen | 接続確立 | WebSocket接続成功 | 接続が確立された時 |
| WebSocket onclose | 接続切断 | WebSocket切断 | 接続が閉じられた時 |
| WebSocket onerror | エラー発生 | WebSocketエラー | 通信エラーが発生した時 |
| WebSocket onmessage | メッセージ受信 | データ受信 | サーバーからメッセージを受信した時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | handleEventが呼び出されれば必ず発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[JavaScriptイベント発生] --> B[handleEvent呼び出し]
    B --> C[RaiseEvent EventRaised]
    C --> D[イベントハンドラ実行]
    D --> E{イベントタイプ判定}
    E -->|Open| F[RaiseEvent OnOpen]
    E -->|Close| G[RaiseEvent OnClose]
    E -->|Error| H[RaiseEvent OnError]
    E -->|Message| I[RaiseEvent OnMessage]
    F --> J[Listenersへ通知]
    G --> J
    H --> J
    I --> J
    J --> K[終了]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| IEオブジェクトエラー | Internet Explorerが利用不可 | CreateObjectエラーが発生 |
| WebSocket接続エラー | 無効なURL | OnErrorイベント経由でエラー通知 |
| イベントハンドラエラー | ハンドラ内でエラー発生 | VBA標準のエラーハンドリング |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（イベント発生に応じて発火） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（WebSocket通信中に随時発火）

## セキュリティ考慮事項

- Internet Explorerのインスタンスを内部で使用しており、IEのセキュリティ設定に依存
- WebSocket URLは検証されずに使用されるため、信頼できるURLのみを使用すべき
- eventDataには外部サーバーからのデータが含まれる可能性があり、処理時に検証が必要
- クロスサイトスクリプティング（XSS）への注意が必要

## 備考

- EventRaisedは汎用イベントであり、OnOpen/OnClose/OnError/OnMessageの個別イベントと併用可能
- Listenersコレクションを使用することで、クラスモジュール以外からもイベントを監視可能
- Internet Explorerを内部で使用しているため、Windowsプラットフォーム限定
- WebSocket接続はpURL変数で管理される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

stdWebSocketクラスの内部変数とイベント宣言を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行57-60の内部変数（ie, document, pURL, Listeners）を確認 |
| 1-2 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行36-55のイベント宣言を確認 |

**読解のコツ**: ieはInternet Explorerオブジェクト、documentはそのドキュメント、pURLはWebSocket接続先URL、ListenersはstdICallableオブジェクトのコレクション。

#### Step 2: 初期化処理を理解する

WebSocket接続の初期化処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行67-70のCreateファクトリメソッド |
| 2-2 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行78-119のprotInitメソッド |

**主要処理フロー**:
1. **行80**: Internet Explorerオブジェクト作成
2. **行83-86**: 空ページへナビゲート
3. **行92-95**: bindヘルパー関数をJavaScriptで定義
4. **行98-101**: VBオブジェクトとURLをブラウザにバインド
5. **行111-118**: WebSocket作成とイベントハンドラをJavaScriptで設定

#### Step 3: イベント処理を理解する

handleEventメソッドとイベント発火の仕組みを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行152-167のhandleEventメソッド |
| 3-2 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行39のEventRaisedイベント宣言 |

**主要処理フロー**:
- **行153**: `RaiseEvent EventRaised(sEventName, oEventData)` - EventRaisedイベント発火
- **行154-159**: Select Caseで個別イベント（OnOpen, OnClose, OnError, OnMessage）を発火
- **行161-166**: Listenersコレクション内の各リスナーにイベントを通知

### プログラム呼び出し階層図

```
JavaScript WebSocket
    │
    ├─ ws.onopen/onclose/onerror/onmessage
    │      │
    │      └─ vb_obj.handleEvent(eventName, event)
    │             │
    │             ├─ RaiseEvent EventRaised(sEventName, oEventData)  ← イベント発火
    │             │      │
    │             │      └─ [VBAイベントハンドラ実行]
    │             │
    │             ├─ RaiseEvent OnOpen/OnClose/OnError/OnMessage
    │             │
    │             └─ For Each listener In Listeners
    │                    └─ listener.Run(sEventName, oEventData)
```

### データフロー図

```
[入力]                      [処理]                    [出力]

WebSocket通信
    │
    └─▶ JavaScript Event ───▶ handleEvent ───▶ EventRaisedイベント
                                    │              │
                                    │              ├─▶ sEventName (イベント名)
                                    │              └─▶ eventData (イベントデータ)
                                    │
                                    ├─▶ 個別イベント（OnOpen等）
                                    │
                                    └─▶ Listeners.Run()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdWebSocket.cls | `src/stdWebSocket.cls` | ソース | EventRaisedイベントを定義・発火するメインクラス |
| stdICallable.cls | `src/stdICallable.cls` | ソース | Listenersに追加可能なコールバックインターフェース |
