# 通知設計書 4-OnOpen

## 概要

本ドキュメントは、stdVBAライブラリのstdWebSocketクラスで提供されるOnOpenイベントの設計仕様を定義する。このイベントはWebSocket接続が確立された際に発火し、接続完了後の初期化処理を可能にする。

### 本通知の処理概要

OnOpenイベントは、stdWebSocketクラスのhandleEventメソッド内で、WebSocket接続が正常に確立された際に発火するVBAイベントである。このイベントにより、接続確立後の初期メッセージ送信、UIの更新、接続状態の管理などが可能になる。

**業務上の目的・背景**：WebSocket通信では、接続が確立されたタイミングを正確に把握することが重要である。これにより、接続確認メッセージの送信、認証トークンの送付、接続状態インジケータの更新などの初期処理を適切なタイミングで実行できる。OnOpenイベントはWebSocket API標準のonopenイベントに対応し、VBAアプリケーションからリアルタイム通信を制御する基盤を提供する。

**通知の送信タイミング**：WebSocket接続が確立され、JavaScriptのonopenイベントが発火した際。具体的にはstdWebSocket.cls行155（handleEventメソッド内のRaiseEvent OnOpen）で発火する。

**通知の受信者**：stdWebSocketオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。

**通知内容の概要**：イベントハンドラには、JavaScriptから渡されるイベントデータ（eventData引数）が渡される。このデータにはWebSocket接続に関する情報が含まれる。

**期待されるアクション**：開発者はこのイベントを利用して、接続確立のログ記録、初期メッセージの送信、UI状態の更新（接続中インジケータなど）、認証処理の開始などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventによる同期呼び出し） |
| 優先度 | 高（接続確立時に発火） |
| リトライ | 無（単発のイベント発火） |

### 送信先決定ロジック

WithEventsキーワードを使用してstdWebSocketオブジェクトを宣言しているクラスモジュール内のイベントハンドラ（Sub ws_OnOpen形式）に対して送信される。

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | OnOpen |
| クラス名 | stdWebSocket |
| シグネチャ | Public Event OnOpen(ByVal eventData As Variant) |

### イベントハンドラテンプレート

```vba
Private WithEvents ws As stdWebSocket

Private Sub ws_OnOpen(ByVal eventData As Variant)
    ' 接続確立後の処理
    Debug.Print "WebSocket connection established!"

    ' 接続後の初期メッセージ送信例
    ws.Send "{""type"":""auth"",""token"":""xxx""}"

    ' UI更新例
    lblStatus.Caption = "Connected"
    lblStatus.ForeColor = vbGreen
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| eventData | JavaScriptイベントオブジェクト | handleEventメソッドの引数（oEventData） | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| WebSocket onopen | 接続確立 | WebSocket接続成功 | サーバーとの接続が正常に確立された時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| 接続失敗 | 接続に失敗した場合、OnOpenは発火せずOnErrorが発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[stdWebSocket.Create呼び出し] --> B[protInit実行]
    B --> C[IEオブジェクト作成]
    C --> D[WebSocket作成]
    D --> E{接続試行}
    E -->|成功| F[JavaScriptのonopen発火]
    E -->|失敗| G[JavaScriptのonerror発火]
    F --> H[handleEvent'Open'呼び出し]
    H --> I[RaiseEvent EventRaised]
    I --> J[RaiseEvent OnOpen]
    J --> K[イベントハンドラ実行]
    K --> L[終了]
    G --> M[OnErrorイベント発火]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 接続タイムアウト | サーバー無応答 | OnErrorイベントが代わりに発火 |
| 無効なURL | URLフォーマット不正 | OnErrorイベントが発火 |
| ネットワークエラー | 通信不可 | OnErrorイベントが発火 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（アプリケーション側で再接続を実装） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（接続確立時に1回のみ発火） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（接続確立時に発火）

## セキュリティ考慮事項

- 接続確立後に認証情報を送信する場合、暗号化された接続（wss://）を使用すべき
- eventDataに含まれる情報は外部サーバー由来の可能性があるため検証が必要
- 接続確立直後の認証処理では、機密情報の取り扱いに注意

## 備考

- OnOpenイベントはEventRaisedイベントの後に発火する
- 1つのWebSocket接続に対して、OnOpenは正常なライフサイクルで1回のみ発火
- 再接続時は新しいstdWebSocketインスタンスを作成する必要がある

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: イベント宣言を理解する

OnOpenイベントの宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行41-43のOnOpenイベント宣言とコメント |

**読解のコツ**: ByVal eventData As Variantで、JavaScriptイベントオブジェクトがVariant型として渡される。

#### Step 2: WebSocket作成と接続を理解する

protInitメソッドでのWebSocket作成処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行111-118のJavaScriptコード |

**主要処理フロー**:
- **行112**: `window.ws = new WebSocket(websocket_url)` - WebSocket作成
- **行113**: `ws.onopen = function(event){ vb_obj.handleEvent('Open', event); }` - onopenハンドラ設定

#### Step 3: イベント発火処理を理解する

handleEventメソッドでのOnOpen発火処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行152-167のhandleEventメソッド |

**主要処理フロー**:
- **行153**: `RaiseEvent EventRaised(sEventName, oEventData)` - 汎用イベント発火
- **行155**: `Case "Open": RaiseEvent OnOpen(oEventData)` - OnOpen発火

### プログラム呼び出し階層図

```
stdWebSocket.Create(url)
    │
    └─ protInit(url)
           │
           ├─ CreateObject("InternetExplorer.Application")
           │
           ├─ JavaScript: new WebSocket(url)
           │      │
           │      └─ ws.onopen = function(event){...}
           │
           └─ [接続確立時]
                  │
                  └─ vb_obj.handleEvent("Open", event)
                         │
                         ├─ RaiseEvent EventRaised("Open", event)
                         │
                         └─ RaiseEvent OnOpen(event)  ← イベント発火
                                │
                                └─ [VBAイベントハンドラ実行]
```

### データフロー図

```
[入力]                      [処理]                    [出力]

WebSocket URL
    │
    └─▶ new WebSocket(url)
              │
              └─▶ 接続確立 ───▶ onopen ───▶ handleEvent
                                                │
                                                └─▶ OnOpenイベント
                                                        │
                                                        └─▶ eventData (接続情報)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdWebSocket.cls | `src/stdWebSocket.cls` | ソース | OnOpenイベントを定義・発火するメインクラス |
