# 通知設計書 5-OnClose

## 概要

本ドキュメントは、stdVBAライブラリのstdWebSocketクラスで提供されるOnCloseイベントの設計仕様を定義する。このイベントはWebSocket接続が閉じられた際に発火し、切断後のクリーンアップ処理や再接続ロジックの実行を可能にする。

### 本通知の処理概要

OnCloseイベントは、stdWebSocketクラスのhandleEventメソッド内で、WebSocket接続が閉じられた際に発火するVBAイベントである。このイベントにより、切断理由の確認、リソースの解放、ユーザーへの通知、再接続処理の開始などが可能になる。

**業務上の目的・背景**：WebSocket通信では、接続が切断されたタイミングを正確に把握することが重要である。これにより、切断原因のログ記録、UIへの状態反映、自動再接続の実装、リソースの適切な解放が可能となる。OnCloseイベントはWebSocket API標準のoncloseイベントに対応し、接続ライフサイクルの終了を適切に処理する基盤を提供する。切断はサーバー側からの切断、クライアント側からのCloseSocket呼び出し、ネットワーク障害など様々な原因で発生する。

**通知の送信タイミング**：WebSocket接続が閉じられ、JavaScriptのoncloseイベントが発火した際。具体的にはstdWebSocket.cls行156（handleEventメソッド内のRaiseEvent OnClose）で発火する。

**通知の受信者**：stdWebSocketオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。

**通知内容の概要**：イベントハンドラには、JavaScriptから渡されるCloseEventオブジェクト（eventData引数）が渡される。このデータには切断コード（code）、切断理由（reason）、正常切断かどうか（wasClean）などの情報が含まれる。

**期待されるアクション**：開発者はこのイベントを利用して、切断のログ記録、UI状態の更新（切断中インジケータなど）、再接続ロジックの起動、リソースの解放などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventによる同期呼び出し） |
| 優先度 | 高（接続切断時に発火） |
| リトライ | 無（単発のイベント発火） |

### 送信先決定ロジック

WithEventsキーワードを使用してstdWebSocketオブジェクトを宣言しているクラスモジュール内のイベントハンドラ（Sub ws_OnClose形式）に対して送信される。

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | OnClose |
| クラス名 | stdWebSocket |
| シグネチャ | Public Event OnClose(ByVal eventData As Variant) |

### イベントハンドラテンプレート

```vba
Private WithEvents ws As stdWebSocket

Private Sub ws_OnClose(ByVal eventData As Variant)
    ' 接続切断後の処理
    Debug.Print "WebSocket connection closed!"

    ' 切断コードと理由を取得
    Dim closeCode As Long
    Dim closeReason As String

    On Error Resume Next
    closeCode = eventData.code
    closeReason = eventData.reason
    On Error GoTo 0

    Debug.Print "Close code: " & closeCode
    Debug.Print "Close reason: " & closeReason

    ' UI更新例
    lblStatus.Caption = "Disconnected"
    lblStatus.ForeColor = vbRed

    ' 再接続ロジック例（必要に応じて）
    ' Call ReconnectWebSocket()
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| eventData | JavaScript CloseEventオブジェクト | handleEventメソッドの引数（oEventData） | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| WebSocket onclose | 接続切断 | WebSocket接続終了 | サーバーとの接続が閉じられた時 |
| CloseSocket呼び出し | 明示的切断 | CloseSocketメソッド実行 | クライアント側から切断を要求した時 |
| Disconnect呼び出し | 完全切断 | Disconnectメソッド実行 | 切断とリソース解放を行った時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | 接続が閉じられれば必ず発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[切断イベント発生] --> B{切断原因}
    B -->|サーバー切断| C[JavaScriptのonclose発火]
    B -->|CloseSocket呼び出し| D[document.ws.Close実行]
    B -->|Disconnect呼び出し| E[CloseSocket + ie.Quit]
    D --> C
    E --> C
    C --> F[handleEvent'Close'呼び出し]
    F --> G[RaiseEvent EventRaised]
    G --> H[RaiseEvent OnClose]
    H --> I[イベントハンドラ実行]
    I --> J[終了]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| eventDataアクセスエラー | CloseEventプロパティアクセス失敗 | On Error Resume Nextで保護 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（アプリケーション側で再接続を実装） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（切断時に1回のみ発火） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（接続切断時に発火）

## セキュリティ考慮事項

- 切断理由（reason）に機密情報が含まれる可能性があるため、ログ出力時に注意
- 自動再接続を実装する場合、無限ループを防ぐためのバックオフ機構を検討

## 備考

- OnCloseイベントはEventRaisedイベントの後に発火する
- 1つのWebSocket接続に対して、OnCloseは正常なライフサイクルで1回のみ発火
- CloseSocket/Disconnectメソッドを呼ばずに接続が切断された場合も発火する
- CloseEventのcodeプロパティでステータスコードを確認可能（1000=正常終了など）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: イベント宣言を理解する

OnCloseイベントの宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行45-47のOnCloseイベント宣言とコメント |

**読解のコツ**: ByVal eventData As VariantでJavaScript CloseEventオブジェクトが渡される。code, reason, wasCleanプロパティにアクセス可能。

#### Step 2: 切断処理を理解する

CloseSocketとDisconnectメソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行136-138のCloseSocketメソッド |
| 2-2 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行141-145のDisconnectメソッド |

**主要処理フロー**:
- **行137**: `Call document.ws.Close(Code, reason)` - WebSocket切断
- **行142**: `Call CloseSocket` - Disconnect内でCloseSocket呼び出し
- **行143**: `Call document.Bind("vb_obj", 0)` - VBオブジェクト参照解除
- **行144**: `Call ie.Quit` - IEオブジェクト終了

#### Step 3: イベント発火処理を理解する

handleEventメソッドでのOnClose発火処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行152-167のhandleEventメソッド |

**主要処理フロー**:
- **行153**: `RaiseEvent EventRaised(sEventName, oEventData)` - 汎用イベント発火
- **行156**: `Case "Close": RaiseEvent OnClose(oEventData)` - OnClose発火

### プログラム呼び出し階層図

```
[切断トリガー]
    │
    ├─ サーバー側切断
    │      │
    │      └─ ws.onclose発火
    │
    ├─ CloseSocket(code, reason)
    │      │
    │      └─ document.ws.Close(code, reason) → ws.onclose発火
    │
    └─ Disconnect()
           │
           ├─ CloseSocket() → ws.onclose発火
           │
           ├─ document.Bind("vb_obj", 0)
           │
           └─ ie.Quit

[ws.onclose発火後]
    │
    └─ vb_obj.handleEvent("Close", event)
           │
           ├─ RaiseEvent EventRaised("Close", event)
           │
           └─ RaiseEvent OnClose(event)  ← イベント発火
                  │
                  └─ [VBAイベントハンドラ実行]
```

### データフロー図

```
[入力]                      [処理]                    [出力]

切断イベント
    │
    ├─▶ サーバー切断
    │
    ├─▶ CloseSocket(code, reason)
    │
    └─▶ Disconnect()
              │
              └─▶ onclose ───▶ handleEvent
                                    │
                                    └─▶ OnCloseイベント
                                            │
                                            └─▶ eventData
                                                   │
                                                   ├─ code (ステータスコード)
                                                   ├─ reason (切断理由)
                                                   └─ wasClean (正常切断フラグ)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdWebSocket.cls | `src/stdWebSocket.cls` | ソース | OnCloseイベントを定義・発火するメインクラス |
