# 通知設計書 6-OnError

## 概要

本ドキュメントは、stdVBAライブラリのstdWebSocketクラスで提供されるOnErrorイベントの設計仕様を定義する。このイベントはWebSocket通信でエラーが発生した際に発火し、エラーハンドリングや障害対応処理を可能にする。

### 本通知の処理概要

OnErrorイベントは、stdWebSocketクラスのhandleEventメソッド内で、WebSocket通信においてエラーが発生した際に発火するVBAイベントである。このイベントにより、エラーの検出、ログ記録、ユーザーへの通知、復旧処理の開始などが可能になる。

**業務上の目的・背景**：WebSocket通信では、ネットワーク障害、サーバーエラー、プロトコルエラーなど様々な原因でエラーが発生する可能性がある。これらのエラーを適切に検出し対処することは、堅牢なリアルタイムアプリケーションを構築する上で不可欠である。OnErrorイベントはWebSocket API標準のonerrorイベントに対応し、エラー発生時の適切な対応を可能にする。このイベントをハンドルすることで、ユーザーへの適切なエラーメッセージ表示、自動リトライ、フォールバック処理などが実装できる。

**通知の送信タイミング**：WebSocket通信でエラーが発生し、JavaScriptのonerrorイベントが発火した際。具体的にはstdWebSocket.cls行157（handleEventメソッド内のRaiseEvent OnError）で発火する。

**通知の受信者**：stdWebSocketオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。

**通知内容の概要**：イベントハンドラには、JavaScriptから渡されるErrorEventオブジェクト（eventData引数）が渡される。ただし、WebSocket APIのonerrorは詳細なエラー情報を提供しないため、eventDataの内容は限定的である場合が多い。

**期待されるアクション**：開発者はこのイベントを利用して、エラーのログ記録、ユーザーへのエラー通知、UI状態の更新、再接続試行、フォールバック処理の開始などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventによる同期呼び出し） |
| 優先度 | 高（エラー発生時に即座に発火） |
| リトライ | 無（単発のイベント発火） |

### 送信先決定ロジック

WithEventsキーワードを使用してstdWebSocketオブジェクトを宣言しているクラスモジュール内のイベントハンドラ（Sub ws_OnError形式）に対して送信される。

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | OnError |
| クラス名 | stdWebSocket |
| シグネチャ | Public Event OnError(ByVal eventData As Variant) |

### イベントハンドラテンプレート

```vba
Private WithEvents ws As stdWebSocket

Private Sub ws_OnError(ByVal eventData As Variant)
    ' エラー発生時の処理
    Debug.Print "WebSocket error occurred!"

    ' エラーログ記録
    Call LogError("WebSocket", "Connection error at " & Now)

    ' UI更新例
    lblStatus.Caption = "Error"
    lblStatus.ForeColor = vbRed

    ' ユーザーへの通知
    MsgBox "WebSocket connection error. Please check your network.", vbExclamation

    ' 再接続試行例（必要に応じて）
    ' Call ScheduleReconnect(5) ' 5秒後に再接続
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| eventData | JavaScript ErrorEventオブジェクト | handleEventメソッドの引数（oEventData） | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| WebSocket onerror | 通信エラー | エラー発生 | WebSocket通信中にエラーが発生した時 |
| 接続失敗 | 接続エラー | 接続不可 | サーバーへの接続に失敗した時 |
| プロトコルエラー | ハンドシェイク失敗 | プロトコル不一致 | WebSocketプロトコルのハンドシェイクに失敗した時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | エラーが発生すれば必ず発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[WebSocket操作] --> B{エラー発生?}
    B -->|No| C[正常処理継続]
    B -->|Yes| D[JavaScriptのonerror発火]
    D --> E[handleEvent'Error'呼び出し]
    E --> F[RaiseEvent EventRaised]
    F --> G[RaiseEvent OnError]
    G --> H[イベントハンドラ実行]
    H --> I{OnClose発火?}
    I -->|Yes| J[OnCloseイベント]
    I -->|No| K[終了]
    J --> K
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 接続エラー | サーバー到達不可 | OnErrorで検出、再接続試行 |
| ネットワークエラー | 通信途絶 | OnErrorで検出、ユーザー通知 |
| SSL/TLSエラー | 証明書不正 | OnErrorで検出、セキュリティ警告 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（アプリケーション側で再接続を実装） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（エラー発生時に発火） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（エラー発生時に発火）

## セキュリティ考慮事項

- エラーメッセージに機密情報が含まれる可能性があるため、ログ出力やユーザー表示時に注意
- エラー発生時の再接続処理では、認証情報の再送信に注意
- 連続エラー時の無限ループを防ぐためのバックオフ機構を検討

## 備考

- OnErrorイベントはEventRaisedイベントの後に発火する
- WebSocket APIの仕様上、onerrorは詳細なエラー情報を提供しないことが多い
- OnErrorの後にOnCloseが発火することが一般的（エラーにより接続が切断される場合）
- エラー原因の詳細を取得するには、サーバー側のログ確認が必要な場合がある

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: イベント宣言を理解する

OnErrorイベントの宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行49-51のOnErrorイベント宣言とコメント |

**読解のコツ**: ByVal eventData As VariantでJavaScript ErrorEventオブジェクトが渡される。ただし、WebSocket APIの制限により詳細情報は限定的。

#### Step 2: JavaScriptイベントハンドラを理解する

JavaScript側のonerrorハンドラ設定を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行111-118のJavaScriptコード |

**主要処理フロー**:
- **行115**: `ws.onerror = function(event){ vb_obj.handleEvent('Error', event); }` - onerrorハンドラ設定

#### Step 3: イベント発火処理を理解する

handleEventメソッドでのOnError発火処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行152-167のhandleEventメソッド |

**主要処理フロー**:
- **行153**: `RaiseEvent EventRaised(sEventName, oEventData)` - 汎用イベント発火
- **行157**: `Case "Error": RaiseEvent OnError(oEventData)` - OnError発火

### プログラム呼び出し階層図

```
[エラー発生源]
    │
    ├─ ネットワーク障害
    │
    ├─ サーバーエラー
    │
    └─ 接続失敗
           │
           └─ JavaScript: ws.onerror発火
                  │
                  └─ vb_obj.handleEvent("Error", event)
                         │
                         ├─ RaiseEvent EventRaised("Error", event)
                         │
                         └─ RaiseEvent OnError(event)  ← イベント発火
                                │
                                └─ [VBAイベントハンドラ実行]
                                       │
                                       └─ [エラー処理・復旧処理]
```

### データフロー図

```
[入力]                      [処理]                    [出力]

エラー発生
    │
    └─▶ JavaScript onerror
              │
              └─▶ handleEvent("Error", event)
                        │
                        └─▶ OnErrorイベント
                                │
                                └─▶ eventData (ErrorEvent)
                                       │
                                       └─▶ [VBAイベントハンドラ]
                                              │
                                              ├─ エラーログ記録
                                              ├─ UI更新
                                              └─ 復旧処理
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdWebSocket.cls | `src/stdWebSocket.cls` | ソース | OnErrorイベントを定義・発火するメインクラス |
