# 通知設計書 7-OnMessage

## 概要

本ドキュメントは、stdVBAライブラリのstdWebSocketクラスで提供されるOnMessageイベントの設計仕様を定義する。このイベントはWebSocket経由でメッセージを受信した際に発火し、受信データの処理を可能にする。

### 本通知の処理概要

OnMessageイベントは、stdWebSocketクラスのhandleEventメソッド内で、WebSocketサーバーからメッセージを受信した際に発火するVBAイベントである。このイベントにより、リアルタイムに受信したデータの解析、処理、UIへの反映などが可能になる。

**業務上の目的・背景**：WebSocket通信の主要な目的は双方向のリアルタイムデータ通信であり、サーバーからのプッシュ通知やリアルタイムデータ配信を受信することが重要なユースケースである。OnMessageイベントはWebSocket API標準のonmessageイベントに対応し、受信したデータをVBAアプリケーションで処理する基盤を提供する。これにより、株価のリアルタイム更新、チャットメッセージの受信、IoTセンサーデータの受信など様々なリアルタイムアプリケーションを構築できる。

**通知の送信タイミング**：WebSocketサーバーからメッセージを受信し、JavaScriptのonmessageイベントが発火した際。具体的にはstdWebSocket.cls行158（handleEventメソッド内のRaiseEvent OnMessage）で発火する。

**通知の受信者**：stdWebSocketオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。

**通知内容の概要**：イベントハンドラには、JavaScriptから渡されるMessageEventオブジェクト（eventData引数）が渡される。このオブジェクトのdataプロパティに実際の受信メッセージが格納されている。

**期待されるアクション**：開発者はこのイベントを利用して、受信データの解析（JSON解析など）、ビジネスロジックの実行、UIの更新、データの永続化などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventによる同期呼び出し） |
| 優先度 | 高（メッセージ受信時に即座に発火） |
| リトライ | 無（単発のイベント発火） |

### 送信先決定ロジック

WithEventsキーワードを使用してstdWebSocketオブジェクトを宣言しているクラスモジュール内のイベントハンドラ（Sub ws_OnMessage形式）に対して送信される。

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | OnMessage |
| クラス名 | stdWebSocket |
| シグネチャ | Public Event OnMessage(ByVal eventData As Variant) |

### イベントハンドラテンプレート

```vba
Private WithEvents ws As stdWebSocket

Private Sub ws_OnMessage(ByVal eventData As Variant)
    ' メッセージ受信時の処理
    Dim message As String
    message = eventData.data

    Debug.Print "Received: " & message

    ' JSONデータの場合の解析例
    ' Dim json As Object
    ' Set json = JsonConverter.ParseJson(message)

    ' UIへの反映例
    txtMessages.Text = txtMessages.Text & vbCrLf & message

    ' データに応じた処理分岐
    If Left(message, 1) = "{" Then
        ' JSONデータとして処理
        Call ProcessJsonMessage(message)
    Else
        ' テキストデータとして処理
        Call ProcessTextMessage(message)
    End If
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| eventData | JavaScript MessageEventオブジェクト | handleEventメソッドの引数（oEventData） | Yes |
| eventData.data | 受信メッセージ本体 | MessageEvent.dataプロパティ | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| WebSocket onmessage | メッセージ受信 | データ受信成功 | サーバーからメッセージを受信した時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | メッセージを受信すれば必ず発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[サーバーからメッセージ送信] --> B[WebSocketでデータ受信]
    B --> C[JavaScriptのonmessage発火]
    C --> D[handleEvent'Message'呼び出し]
    D --> E[RaiseEvent EventRaised]
    E --> F[RaiseEvent OnMessage]
    F --> G[イベントハンドラ実行]
    G --> H{データ形式判定}
    H -->|JSON| I[JSON解析処理]
    H -->|Text| J[テキスト処理]
    H -->|Binary| K[バイナリ処理]
    I --> L[ビジネスロジック実行]
    J --> L
    K --> L
    L --> M[UI更新/データ保存]
    M --> N[終了]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| データアクセスエラー | eventData.dataアクセス失敗 | On Error Resume Nextで保護 |
| JSON解析エラー | 不正なJSON形式 | エラーハンドリングで対処 |
| 型変換エラー | 予期しないデータ型 | 型チェックを実施 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（メッセージ受信に応じて発火） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（メッセージ受信時に発火）

## セキュリティ考慮事項

- 受信データは外部サーバーからの入力であるため、必ず検証・サニタイズが必要
- JSON解析時はインジェクション攻撃に注意
- 機密データを受信する場合は暗号化された接続（wss://）を使用
- 大量データ受信によるメモリ枯渇に注意

## 備考

- OnMessageイベントはEventRaisedイベントの後に発火する
- eventData.dataにはテキストまたはバイナリデータが格納される
- 高頻度のメッセージ受信時はパフォーマンスに注意
- VBAでのJSON解析にはJsonConverterなどの外部ライブラリが必要

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: イベント宣言を理解する

OnMessageイベントの宣言を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行53-55のOnMessageイベント宣言とコメント |

**読解のコツ**: ByVal eventData As VariantでJavaScript MessageEventオブジェクトが渡される。eventData.dataで実際のメッセージにアクセス。

#### Step 2: JavaScriptイベントハンドラを理解する

JavaScript側のonmessageハンドラ設定を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行111-118のJavaScriptコード |

**主要処理フロー**:
- **行116**: `ws.onmessage = function(event){ vb_obj.handleEvent('Message', event); }` - onmessageハンドラ設定

#### Step 3: イベント発火処理を理解する

handleEventメソッドでのOnMessage発火処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行152-167のhandleEventメソッド |

**主要処理フロー**:
- **行153**: `RaiseEvent EventRaised(sEventName, oEventData)` - 汎用イベント発火
- **行158**: `Case "Message": RaiseEvent OnMessage(oEventData)` - OnMessage発火

#### Step 4: データ送信を理解する

メッセージ送信側のSendメソッドを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdWebSocket.cls | `src/stdWebSocket.cls` | 行129-131のSendメソッド |

**主要処理フロー**:
- **行130**: `Call document.ws.Send(data)` - WebSocketでデータ送信

### プログラム呼び出し階層図

```
[サーバー]
    │
    └─ WebSocketメッセージ送信
           │
           └─ [クライアント: JavaScript WebSocket]
                  │
                  └─ ws.onmessage発火
                         │
                         └─ vb_obj.handleEvent("Message", event)
                                │
                                ├─ RaiseEvent EventRaised("Message", event)
                                │
                                └─ RaiseEvent OnMessage(event)  ← イベント発火
                                       │
                                       └─ [VBAイベントハンドラ実行]
                                              │
                                              └─ event.data でメッセージ取得
```

### データフロー図

```
[入力]                      [処理]                    [出力]

サーバーからのメッセージ
    │
    └─▶ JavaScript onmessage
              │
              └─▶ handleEvent("Message", event)
                        │
                        └─▶ OnMessageイベント
                                │
                                └─▶ eventData
                                       │
                                       └─▶ eventData.data (受信メッセージ)
                                              │
                                              ├─▶ JSON解析
                                              ├─▶ UI更新
                                              └─▶ データ保存
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdWebSocket.cls | `src/stdWebSocket.cls` | ソース | OnMessageイベントを定義・発火するメインクラス |
