# 通知設計書 8-Tick

## 概要

本ドキュメントは、stdVBAライブラリのstdTimerクラスで提供されるTickイベントの設計仕様を定義する。このイベントは指定されたミリ秒間隔で定期的に発火し、タイマー処理やポーリング処理を可能にする。

### 本通知の処理概要

Tickイベントは、stdTimerクラスで実装された定期タイマーイベントである。別プロセスのExcelアプリケーション（リモートワークブック）内でループ処理を行い、COMイベント（WorksheetChange）を通じてVBAアプリケーションに通知を送信する。このイベントにより、定期的なUIの更新、ポーリング処理、タイムアウト監視などが可能になる。

**業務上の目的・背景**：VBAには標準のタイマー機構が存在しないため、定期的な処理の実行が困難である。stdTimerクラスはこの課題を解決し、ミリ秒単位の精度で定期イベントを発火させる機能を提供する。これにより、リアルタイムのデータ更新、アニメーション、定期バックアップ、接続監視など様々なタイマーベースの機能を実装できる。Cristian BuseのSafeTimersから着想を得ており、より柔軟なコールバック機構（stdICallable）との連携を実現している。

**通知の送信タイミング**：stdTimer.Createで指定されたミリ秒間隔で定期的に発火する。具体的にはstdTimer.cls行180（xlRemoteWorkbook_SheetChangeイベントハンドラ内のRaiseEvent Tick）で発火する。

**通知の受信者**：stdTimerオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。また、Createメソッドでcallback引数を指定している場合は、そのstdICallableオブジェクトのRunメソッドも呼び出される。

**通知内容の概要**：Tickイベントは引数を持たない単純なイベントである。イベント発火時刻は内部的には保持されていないため、必要に応じてNow関数等で取得する。

**期待されるアクション**：開発者はこのイベントを利用して、定期的なUIの更新、外部データのポーリング、タイムアウトチェック、アニメーション処理、進捗状況の更新などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）/ stdICallableコールバック

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventおよびcallback.Run） |
| 優先度 | 中（定期実行、タイミングは厳密ではない） |
| リトライ | 無（ループ処理で継続的に発火） |

### 送信先決定ロジック

1. WithEventsキーワードを使用してstdTimerオブジェクトを宣言しているクラスモジュール内のイベントハンドラ
2. Createメソッドのcallback引数で指定されたstdICallableオブジェクトのRunメソッド

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | Tick |
| クラス名 | stdTimer |
| シグネチャ | Public Event Tick() |

### イベントハンドラテンプレート

```vba
Private WithEvents tmr As stdTimer

Private Sub InitializeTimer()
    ' 1000ミリ秒（1秒）間隔でTickイベントを発火
    Set tmr = stdTimer.Create(1000)
End Sub

Private Sub tmr_Tick()
    ' 定期処理
    Debug.Print "Tick at " & Now

    ' UIの更新例
    lblTime.Caption = Format(Now, "hh:mm:ss")

    ' データのポーリング例
    Call RefreshData()

    ' タイムアウトチェック例
    If DateDiff("s", startTime, Now) > 60 Then
        Call HandleTimeout()
    End If
End Sub
```

### コールバック方式のテンプレート

```vba
' モジュール内で使用する場合
Public Sub StartTimerWithCallback()
    Dim tmr As stdTimer
    Dim cb As stdCallback

    ' コールバック作成
    Set cb = stdCallback.CreateFromModule("MyModule", "OnTick")

    ' タイマー作成（500ミリ秒間隔）
    Set tmr = stdTimer.Create(500, cb)
End Sub

Public Sub OnTick()
    Debug.Print "Timer callback at " & Now
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

該当なし（Tickイベントは引数を持たない）

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| タイマー | ミリ秒間隔経過 | タイマー動作中 | 指定間隔ごとに発火 |
| SheetChange | リモートセル変更 | ワークブック変更検出 | リモートワークブックのセル変更イベントを検出 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| stdTimer終了 | Class_Terminateが呼ばれると停止 |
| MsgBox/InputBox表示中 | モーダルダイアログ表示中はTickが発火しない |
| デバッグ中 | VBAコードのデバッグ中はTickが発火しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[stdTimer.Create] --> B[protInit呼び出し]
    B --> C{VBOM有効?}
    C -->|No| D[VBOM有効化]
    C -->|Yes| E[リモートワークブック作成]
    D --> E
    E --> F[タイマーコード注入]
    F --> G[StartTimer実行]
    G --> H[MainLoopスケジュール]
    H --> I[MainLoop開始]
    I --> J[Sheets1.Range-A1変更]
    J --> K[xlRemoteWorkbook_SheetChange発火]
    K --> L[RaiseEvent Tick]
    L --> M[イベントハンドラ実行]
    M --> N{callback設定?}
    N -->|Yes| O[callback.Run実行]
    N -->|No| P[Sleep]
    O --> P
    P --> Q{bRunning?}
    Q -->|Yes| J
    Q -->|No| R[ワークブック閉じる]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| VBOM無効 | VBE拡張が無効 | 自動でレジストリ設定を変更 |
| リモートワークブック作成失敗 | Excel新規インスタンス作成失敗 | エラー発生 |
| タイマー精度低下 | システム負荷高 | 仕様として許容 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし（継続的なループ処理） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 最小間隔 | 技術的制限なし（ただし1ms未満は非推奨） |
| 推奨間隔 | 100ms以上 |

### 配信時間帯

制限なし（タイマー動作中は継続的に発火）

## セキュリティ考慮事項

- VBOM（Visual Basic for Applications Object Model）へのアクセスを有効化するためにレジストリを変更する
- 別プロセスのExcelアプリケーションが作成されるため、リソース管理に注意
- タイマー停止時には適切にリソースを解放する必要がある

## 備考

- MsgBoxやInputBox表示中はTickイベントが発火しない（SafeTimersとの違い）
- Edit/Pointモード中もTickイベントは発火する（SafeTimers-Nativeとの違い）
- VBAデバッグ中はTickイベントが発火しない
- タイマー精度は保証されず、実際の間隔には誤差が生じる可能性がある
- Class_Terminateでタイマーが自動停止し、リモートワークブックも自動クローズされる

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: アーキテクチャを理解する

stdTimerの設計コンセプトを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdTimer.cls | `src/stdTimer.cls` | 行11-38のコメントでアーキテクチャ説明を確認 |

**読解のコツ**: stdTimerは別プロセスのExcelインスタンス（リモートワークブック）を作成し、そのSheetChangeイベントを通じてTickを通知する仕組み。これによりVBAの制限を回避している。

#### Step 2: 初期化処理を理解する

タイマーの初期化処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdTimer.cls | `src/stdTimer.cls` | 行56-59のCreateファクトリメソッド |
| 2-2 | stdTimer.cls | `src/stdTimer.cls` | 行65-72のprotInitメソッド |
| 2-3 | stdTimer.cls | `src/stdTimer.cls` | 行76-90のcreateRemoteWorkbookメソッド |

**主要処理フロー**:
- **行66**: pCallbackにコールバックを保存
- **行67-69**: VBOMの有効化チェックと設定
- **行68**: リモートワークブック作成
- **行70**: タイマーコード注入
- **行71**: タイマー開始

#### Step 3: タイマーコードを理解する

注入されるVBAコードを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdTimer.cls | `src/stdTimer.cls` | 行136-168のgetTimerCodeメソッド |

**主要処理フロー**:
- **行152-158**: MainLoopプロシージャ（Sheets(1).Range("A1")を変更してイベント発火）
- **行160-163**: StartTimerプロシージャ
- **行164-166**: StopTimerプロシージャ

#### Step 4: イベント発火処理を理解する

Tickイベントの発火処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | stdTimer.cls | `src/stdTimer.cls` | 行45のTickイベント宣言 |
| 4-2 | stdTimer.cls | `src/stdTimer.cls` | 行179-182のxlRemoteWorkbook_SheetChangeイベントハンドラ |

**主要処理フロー**:
- **行180**: `RaiseEvent Tick` - Tickイベント発火
- **行181**: `If Not pCallback Is Nothing Then pCallback.Run` - コールバック実行

### プログラム呼び出し階層図

```
stdTimer.Create(iMilliseconds, callback)
    │
    └─ protInit(iMilliseconds, callback)
           │
           ├─ getVBOM() / setVBOM(True)
           │
           ├─ createRemoteWorkbook()
           │      │
           │      └─ CreateObject("Excel.Application")
           │             │
           │             └─ app.Workbooks.Add
           │
           ├─ injectCode(wb, "TimerMain", code)
           │      │
           │      └─ VBProject.VBComponents.Add
           │
           └─ Application.Run("TimerMain.StartTimer")
                  │
                  └─ [リモートExcel内]
                         │
                         └─ MainLoop()
                                │
                                └─ While bRunning
                                       │
                                       ├─ Sheets(1).Range("A1") = "Tick"
                                       │      │
                                       │      └─ [元ワークブック]
                                       │             │
                                       │             └─ xlRemoteWorkbook_SheetChange
                                       │                    │
                                       │                    ├─ RaiseEvent Tick  ← イベント発火
                                       │                    │
                                       │                    └─ pCallback.Run
                                       │
                                       └─ Sleep(iMilliseconds)
```

### データフロー図

```
[入力]                      [処理]                    [出力]

iMilliseconds
    │
    └─▶ タイマー設定
              │
              └─▶ リモートExcel MainLoop
                        │
                        └─▶ Sheets(1).Range("A1")変更
                                │
                                └─▶ SheetChangeイベント
                                        │
                                        └─▶ Tickイベント
                                               │
                                               ├─▶ イベントハンドラ
                                               │
                                               └─▶ コールバック実行
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdTimer.cls | `src/stdTimer.cls` | ソース | Tickイベントを定義・発火するメインクラス |
| stdICallable.cls | `src/stdICallable.cls` | ソース | コールバックインターフェース定義 |
| stdCallback.cls | `src/stdCallback.cls` | ソース | コールバック実装クラス（callback引数で使用可能） |
