# 通知設計書 9-SelectionChange

## 概要

本ドキュメントは、stdVBAライブラリのstdXLEventsクラスで提供されるSelectionChangeイベントの設計仕様を定義する。このイベントはExcelのセル選択範囲が変更された際に発火し、選択変更の監視と対応処理を可能にする。

### 本通知の処理概要

SelectionChangeイベントは、stdXLEventsクラスで実装された汎用的な選択変更イベントである。Application.CommandBarsのOnUpdateイベントをフックし、選択が変更されたことを検出してVBAアプリケーションに通知する。このイベントにより、Range、Shape、Chartなど全ての種類の選択変更を一元的に監視できる。

**業務上の目的・背景**：Excelでは様々なオブジェクト（セル、図形、グラフなど）を選択できるが、標準のVBAではこれらの選択変更を一元的に監視する仕組みがない。stdXLEventsクラスはCommandBars.OnUpdateイベントを利用してこの課題を解決し、あらゆる選択変更を検出・通知する機能を提供する。これにより、コンテキスト依存のUIの実装、選択に応じたプロパティパネルの更新、監査ログの記録など様々な機能を実装できる。

**通知の送信タイミング**：Excelで任意のオブジェクトの選択が変更され、CommandBarsのOnUpdateイベントが発火した際。具体的にはstdXLEvents.cls行84（cbs_OnUpdateイベントハンドラ内のRaiseEvent SelectionChange）で発火する。

**通知の受信者**：stdXLEventsオブジェクトをWithEventsキーワードで宣言しているVBAクラスモジュールの当該イベントハンドラが受信者となる。

**通知内容の概要**：イベントハンドラには、旧選択オブジェクト（selOld引数）と新選択オブジェクト（selNew引数）が渡される。これらはObject型であり、Range、Shape、ChartAreaなど様々な型のオブジェクトが渡される可能性がある。

**期待されるアクション**：開発者はこのイベントを利用して、選択変更のログ記録、コンテキスト依存UIの更新、プロパティインスペクタの更新、選択に基づくアクションの実行などを行うことが期待される。

## 通知種別

VBA Event（クラスイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（RaiseEventによる同期呼び出し） |
| 優先度 | 中（CommandBars.OnUpdate経由） |
| リトライ | 無（単発のイベント発火） |

### 送信先決定ロジック

WithEventsキーワードを使用してstdXLEventsオブジェクトを宣言しているクラスモジュール内のイベントハンドラ（Sub events_SelectionChange形式）に対して送信される。

## 通知テンプレート

### VBAイベントの場合

| 項目 | 内容 |
|-----|------|
| イベント名 | SelectionChange |
| クラス名 | stdXLEvents |
| シグネチャ | Public Event SelectionChange(selOld As Object, selNew As Object) |

### イベントハンドラテンプレート

```vba
Private WithEvents xlEvents As stdXLEvents

Private Sub Class_Initialize()
    Set xlEvents = New stdXLEvents
End Sub

Private Sub xlEvents_SelectionChange(selOld As Object, selNew As Object)
    ' 選択変更時の処理
    Debug.Print "Selection changed"

    ' 旧選択の型を確認
    If Not selOld Is Nothing Then
        Debug.Print "Old selection type: " & TypeName(selOld)
    End If

    ' 新選択の型を確認
    Debug.Print "New selection type: " & TypeName(selNew)

    ' 選択タイプに応じた処理
    Select Case TypeName(selNew)
        Case "Range"
            Call HandleRangeSelection(selNew)
        Case "Shape", "Rectangle", "Oval"
            Call HandleShapeSelection(selNew)
        Case "ChartArea"
            Call HandleChartSelection(selNew)
    End Select
End Sub
```

### 添付ファイル

該当なし（VBAイベントのためファイル添付の概念なし）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| selOld | 旧選択オブジェクト | getSelection().oObjの前回値 | No（初回はNothing） |
| selNew | 新選択オブジェクト | getSelection().oObjの現在値 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| CommandBars.OnUpdate | UIアップデート | 選択変更検出 | 選択が変更されたと判定された時 |
| Range選択 | セル選択変更 | Range型の選択 | セルの選択が変更された時 |
| Shape選択 | 図形選択変更 | Shape型の選択 | 図形の選択が変更された時 |
| Chart選択 | グラフ選択変更 | Chart関連型の選択 | グラフの選択が変更された時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| 選択未変更 | selectionsAreDifferentがFalseを返す場合、イベントは発火しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[ユーザー操作] --> B[CommandBars.OnUpdate発火]
    B --> C[cbs_OnUpdate呼び出し]
    C --> D[getSelection呼び出し]
    D --> E[SelectionInfo構造体作成]
    E --> F{selectionsAreDifferent?}
    F -->|No| G[終了・イベントなし]
    F -->|Yes| H[RaiseEvent SelectionChange]
    H --> I[イベントハンドラ実行]
    I --> J{Range選択?}
    J -->|Yes| K[RaiseEvent RangeSelectionChange等]
    J -->|No| L{Shape選択?}
    L -->|Yes| M[RaiseEvent ShapeSelectionChange等]
    L -->|No| N[終了]
    K --> N
    M --> N
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（VBAイベントのためデータベース操作なし）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| 選択取得エラー | Selection参照失敗 | 例外発生時にスキップ |
| 型判定エラー | 未知の選択タイプ | デフォルト処理にフォールスルー |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（選択変更に応じて発火） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし（選択変更時に発火）

## セキュリティ考慮事項

- selOld/selNewにはワークブックやワークシートへの参照が含まれる可能性があるため、オブジェクト操作に注意
- CommandBars.OnUpdateは頻繁に発火する可能性があるため、ハンドラ内の処理は軽量に保つべき

## 備考

- SelectionChangeはRangeSelectionChange、ShapeSelectionChange等の個別イベントの前に発火する
- selOld/selNewはObject型のため、TypeNameでの型チェックが必要
- stdXLEventsはWIP（Work In Progress）ディレクトリに配置されており、開発中の機能
- 多数の選択タイプ（約40種類）に対応している

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

SelectionInfo構造体と関連する列挙型を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行27-66のESelectionType列挙型 |
| 1-2 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行67-76のSelectionInfo構造体 |

**読解のコツ**: ESelectionTypeは約40種類の選択タイプを定義。SelectionInfoは選択の完全な状態（型、オブジェクト、ワークブック、ワークシート、ID、パス）を保持する。

#### Step 2: 初期化処理を理解する

クラスの初期化とCommandBarsへのフック設定を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行23-24のWithEvents cbs変数 |
| 2-2 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行150-152のClass_Initialize |

**主要処理フロー**:
- **行151**: `Set cbs = Application.CommandBars` - CommandBarsへの参照取得

#### Step 3: イベント発火処理を理解する

cbs_OnUpdateでの選択変更検出とイベント発火を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行80-148のcbs_OnUpdateメソッド |
| 3-2 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行156-163のgetSelectionメソッド |
| 3-3 | stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | 行237-254のselectionsAreDifferentメソッド |

**主要処理フロー**:
- **行81**: Static oldSelection - 前回の選択状態を保持
- **行82**: newSelection = getSelection() - 現在の選択状態を取得
- **行83-84**: selectionsAreDifferentでチェック、違えばSelectionChangeイベント発火
- **行84**: `RaiseEvent SelectionChange(oldSelection.oObj, newSelection.oObj)`

### プログラム呼び出し階層図

```
[Excel UI操作]
    │
    └─ 選択変更
           │
           └─ CommandBars.OnUpdate発火
                  │
                  └─ cbs_OnUpdate()
                         │
                         ├─ getSelection() → newSelection
                         │      │
                         │      ├─ getObjAncestor(Selection, "Workbook")
                         │      ├─ getObjAncestor(Selection, "Worksheet")
                         │      ├─ getObjID(Selection)
                         │      ├─ getObjPath(Selection)
                         │      └─ getObjType(Selection)
                         │
                         └─ selectionsAreDifferent(oldSelection, newSelection)
                                │
                                └─ [異なる場合]
                                       │
                                       └─ RaiseEvent SelectionChange  ← イベント発火
                                              │
                                              ├─▶ selOld (前回のSelection)
                                              │
                                              └─▶ selNew (現在のSelection)
```

### データフロー図

```
[入力]                      [処理]                    [出力]

ユーザー操作
    │
    └─▶ 選択変更
              │
              └─▶ CommandBars.OnUpdate
                        │
                        └─▶ cbs_OnUpdate
                                │
                                ├─▶ getSelection() → SelectionInfo
                                │
                                └─▶ selectionsAreDifferent
                                        │
                                        └─▶ SelectionChangeイベント
                                               │
                                               ├─▶ selOld (Object)
                                               │
                                               └─▶ selNew (Object)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stdXLEvents.cls | `src/WIP/stdExcelLibraries/stdXLEvents.cls` | ソース | SelectionChangeイベントを定義・発火するメインクラス |
