# stdVBA 開発者運用マニュアル

## 目次

**【基本操作】**
1. [システム概要](#1-システム概要)
2. [インストール・セットアップ手順](#2-インストールセットアップ手順)
3. [基本構造（クラス設計パターン）](#3-基本構造クラス設計パターン)

**【業務フローに沿った機能】**
4. [【STEP1】データ準備（配列・コレクション操作）](#4-step1データ準備配列コレクション操作)
5. [【STEP2-3】データ変換・処理（関数型プログラミング）](#5-step2-3データ変換処理関数型プログラミング)
6. [【STEP3-4】テキスト処理・正規表現](#6-step3-4テキスト処理正規表現)
7. [【STEP4-5】システム連携・自動化](#7-step4-5システム連携自動化)

**【サポート機能】**
8. [パフォーマンス計測機能](#8-パフォーマンス計測機能)
9. [クリップボード操作機能](#9-クリップボード操作機能)
10. [HTTP通信・WebSocket機能](#10-http通信websocket機能)

**【運用ガイド】**
11. [日常開発の流れ](#11-日常開発の流れ)
12. [よくある質問と対処法](#12-よくある質問と対処法)
13. [トラブルシューティング](#13-トラブルシューティング)

---

## 1. システム概要

### 1.1 stdVBAとは

stdVBA（Standard VBA Library）は、モダンなVBAアプリケーション開発のための共通標準ライブラリコレクションです。JavaScriptやPythonのような現代的なプログラミング言語のAPIに強くインスパイアされており、VBAの古い言語仕様を補完し、より効率的で保守性の高いコードの記述を可能にします。

### 1.2 本ライブラリで行う主要業務

本ライブラリは、主に以下の開発業務を支援します：

| 業務カテゴリ | 具体的な業務内容 |
|------------|----------------|
| **データ操作** | 動的配列管理、配列要素の追加・削除・検索・ソート、コレクション操作 |
| **関数型プログラミング** | Map/Filter/Reduce処理、ラムダ関数の動的生成、コールバック処理 |
| **テキスト処理** | 正規表現マッチング、名前付きキャプチャ、テキスト置換・抽出 |
| **システム連携** | ウィンドウ操作、プロセス管理、クリップボード操作、UI自動化 |
| **開発支援** | パフォーマンス計測、COM操作、レジストリ操作 |

### 1.3 業務フロー概要

stdVBAを使用した典型的な開発業務は、以下の流れで進行します：

```
+-----------------------------------------------------------------+
|                    stdVBA開発フロー                               |
+-----------------------------------------------------------------+
|                                                                 |
|  【STEP 1】ライブラリのインポートと初期設定                        |
|      |                                                          |
|  【STEP 2】データの準備（stdArray/stdEnumeratorで構造化）          |
|      |                                                          |
|  【STEP 3】データ変換・処理（stdLambda/stdCallbackで変換）          |
|      |                                                          |
|  【STEP 4】テキスト処理・外部連携（stdRegex/stdWindow等）           |
|      |                                                          |
|  【STEP 5】結果の出力と後処理                                     |
|                                                                 |
+-----------------------------------------------------------------+
```

**役割別の業務概要：**

| 役割 | 主な業務 |
|------|---------|
| **VBA開発者** | ライブラリの選択・インポート、コード記述、デバッグ |
| **チームリーダー** | コーディング規約の策定、コードレビュー、ライブラリ更新管理 |

### 1.4 主な機能

| 機能 | 説明 | 業務フローでの位置 |
|------|------|------------------|
| stdArray | JavaScriptライクな動的配列クラス | STEP 2 |
| stdEnumerator | イテレータ・列挙子クラス | STEP 2 |
| stdLambda | インラインラムダ関数生成 | STEP 2-3 |
| stdCallback | コールバック関数オブジェクト生成 | STEP 2-3 |
| stdRegex | 正規表現操作（名前付きキャプチャ対応） | STEP 3-4 |
| stdJSON | JSON操作 | STEP 3-4 |
| stdWindow | ウィンドウ操作 | STEP 4 |
| stdProcess | プロセス管理 | STEP 4 |
| stdClipboard | クリップボード操作 | STEP 4-5 |
| stdPerformance | パフォーマンス計測 | 随時利用 |

### 1.5 ユーザー種別

ライブラリには以下のユーザー種別が想定されます：

| ユーザー種別 | 説明 |
|-------------|------|
| 一般VBA開発者 | 日常的なVBAアプリケーション開発でライブラリを使用 |
| 上級VBA開発者 | ライブラリの拡張やカスタマイズを行う |
| ライブラリ開発者 | stdVBAライブラリ自体の開発・保守を行う |

### 1.6 対応環境

**推奨環境：**
- Microsoft Excel 2010以降
- Microsoft Word 2010以降
- Microsoft PowerPoint 2010以降
- Microsoft Access 2010以降

**対応プラットフォーム：**
- Windows（32bit/64bit）
- Mac（一部機能に制限あり）

---

## 2. インストール・セットアップ手順

### 2.1 ライブラリのダウンロード

1. GitHubリポジトリ（https://github.com/sancarn/stdVBA）にアクセスします
2. 「Code」ボタンをクリックし、「Download ZIP」を選択します
3. ダウンロードしたZIPファイルを解凍します

### 2.2 VBAプロジェクトへのインポート

**方法1: ドラッグ&ドロップ**

1. VBAエディタ（Alt+F11）を開きます
2. Windowsエクスプローラーで解凍した「src」フォルダを開きます
3. 使用したいクラスファイル（.cls）をVBAプロジェクトエクスプローラーにドラッグ&ドロップします

**方法2: ファイルインポート**

1. VBAエディタのメニューから「ファイル」>「ファイルのインポート」を選択します
2. 使用したいクラスファイル（.cls）を選択します
3. 「開く」をクリックします

**必要なファイルの組み合わせ：**

| 使用機能 | 必要ファイル |
|---------|-------------|
| stdArray | stdArray.cls, stdICallable.cls |
| stdLambda | stdLambda.cls, stdICallable.cls |
| stdCallback | stdCallback.cls, stdICallable.cls |
| stdRegex | stdRegex.cls |
| stdEnumerator | stdEnumerator.cls, stdICallable.cls |
| stdWindow | stdWindow.cls |
| stdProcess | stdProcess.cls |
| stdClipboard | stdClipboard.cls |

### 2.3 初期動作確認

インポート後、以下のコードで動作確認を行います：

```vb
Sub TestStdVBA()
    ' stdArrayのテスト
    Dim arr As stdArray
    Set arr = stdArray.Create(1, 2, 3, 4, 5)
    Debug.Print "配列要素数: " & arr.Length        ' => 5
    Debug.Print "配列結合: " & arr.Join(",")       ' => 1,2,3,4,5

    ' stdLambdaのテスト
    Dim doubled As stdArray
    Set doubled = arr.Map(stdLambda.Create("$1 * 2"))
    Debug.Print "2倍した配列: " & doubled.Join(",") ' => 2,4,6,8,10
End Sub
```

### 2.4 重要な設定

以下の設定がクラスファイルに適用されていることを確認してください：

```
Attribute VB_PredeclaredId = True
```

この設定により、`New`キーワードを使用せずにファクトリメソッドを呼び出すことができます。

---

## 3. 基本構造（クラス設計パターン）

### 3.1 ファクトリクラスパターン

stdVBAのすべてのクラスは、統一されたファクトリパターンを使用しています：

```vb
' 正しい使用方法（ファクトリメソッド）
Dim arr As stdArray
Set arr = stdArray.Create(1, 2, 3, 4, 5)

' 誤った使用方法（Newキーワードは使用しない）
' Set arr = New stdArray  ' これは動作しません
```

### 3.2 stdICallableインターフェース

多くのstdVBAクラスは`stdICallable`インターフェースを実装しています。これにより、関数オブジェクトとして扱うことができます：

| メソッド | 説明 |
|---------|------|
| Run(params) | パラメータを渡して関数を実行 |
| RunEx(params) | 配列形式でパラメータを渡して実行 |
| Bind(params) | パラメータをバインドした新しい関数を返却 |

### 3.3 メソッドチェーン

stdArrayなどのクラスはメソッドチェーンをサポートしています：

```vb
Dim result As stdArray
Set result = stdArray.Create(1, 2, 3, 4, 5, 6, 7, 8, 9, 10) _
    .Filter(stdLambda.Create("$1 mod 2 = 0")) _  ' 偶数のみ
    .Map(stdLambda.Create("$1 * 2")) _           ' 2倍
    .Sort(stdLambda.Create("$1 - $2"))           ' 昇順ソート

Debug.Print result.Join(",")  ' => 4,8,12,16,20
```

---

## 4. 【STEP1】データ準備（配列・コレクション操作）

> **業務フローでの位置**: データを収集・整理し、処理可能な形式に構造化するステップです。

### 4.1 stdArrayの基本操作

**配列の作成：**

```vb
' 値から作成
Dim arr As stdArray
Set arr = stdArray.Create(1, 2, 3, 4, 5)

' VBA配列から作成
Dim vbaArr() As Variant
vbaArr = Array(1, 2, 3, 4, 5)
Set arr = stdArray.CreateFromArray(vbaArr)

' コレクションから作成
Dim col As Collection
Set col = New Collection
col.Add 1: col.Add 2: col.Add 3
Set arr = stdArray.CreateFromCollection(col)
```

**要素の追加・削除：**

| 操作 | コード | 説明 |
|------|--------|------|
| 末尾に追加 | `arr.Push 6` | 末尾に要素を追加 |
| 先頭に追加 | `arr.Unshift 0` | 先頭に要素を追加 |
| 末尾から削除 | `val = arr.Pop` | 末尾要素を取得・削除 |
| 先頭から削除 | `val = arr.Shift` | 先頭要素を取得・削除 |
| 指定位置に挿入 | `arr.Insert 3, "新要素"` | 指定位置に要素を挿入 |
| 指定位置を削除 | `arr.Remove 3` | 指定位置の要素を削除 |

### 4.2 stdEnumeratorの基本操作

**列挙子の作成：**

```vb
' コレクションから作成
Dim enum As stdEnumerator
Set enum = stdEnumerator.Create(Application.Workbooks)

' ExcelのListObjectから作成
Set enum = stdEnumerator.CreateFromListObject(Sheet1.ListObjects(1))
```

**基本的な反復処理：**

```vb
' ForEachを使用した反復
Dim item As Variant
For Each item In enum
    Debug.Print item.Name
Next

' forEach メソッドを使用
Call enum.forEach(stdLambda.Create("Debug.Print $1.Name"))
```

### 4.3 配列データの検索・確認

```vb
Dim arr As stdArray
Set arr = stdArray.Create("apple", "banana", "cherry", "date")

' 要素の存在確認
Debug.Print arr.includes("banana")  ' => True
Debug.Print arr.includes("grape")   ' => False

' インデックスの取得
Debug.Print arr.indexOf("cherry")   ' => 3 (1-based)
Debug.Print arr.indexOf("grape")    ' => 0 (見つからない)

' 最後のインデックス
Set arr = stdArray.Create(1, 2, 3, 2, 1)
Debug.Print arr.lastIndexOf(2)      ' => 4
```

---

## 5. 【STEP2-3】データ変換・処理（関数型プログラミング）

> **業務フローでの位置**: 収集したデータをstdLambda/stdCallbackを使用して変換・処理するステップです。

### 5.1 stdLambdaの基本操作

**ラムダ式の作成と実行：**

```vb
' 基本的なラムダ式
Dim add As stdLambda
Set add = stdLambda.Create("$1 + $2")
Debug.Print add.Run(3, 5)  ' => 8

' 引数参照
' $1, $2, $3... で引数を参照
Set lambda = stdLambda.Create("$1 * $2 + $3")
Debug.Print lambda.Run(2, 3, 4)  ' => 10
```

**対応する演算子：**

| 種別 | 演算子 |
|------|--------|
| 算術 | `+`, `-`, `*`, `/`, `^`, `mod` |
| 比較 | `=`, `<>`, `<`, `<=`, `>`, `>=`, `like`, `is` |
| 論理 | `and`, `or`, `not`, `xor` |
| 文字列 | `&` |

**制御フロー構文：**

```vb
' if-then-else
Set lambda = stdLambda.Create("if $1 > 0 then ""positive"" else ""non-positive""")
Debug.Print lambda.Run(5)   ' => positive
Debug.Print lambda.Run(-3)  ' => non-positive

' プロパティアクセス
Set lambda = stdLambda.Create("$1.Name")
Debug.Print lambda.Run(ThisWorkbook)  ' => ブック名

' メソッド呼び出し
Set lambda = stdLambda.Create("$1.Sheets($2)")
Set sheet = lambda.Run(ThisWorkbook, 1)
```

### 5.2 高階関数によるデータ変換

**Map（各要素を変換）：**

```vb
Dim arr As stdArray
Set arr = stdArray.Create(1, 2, 3, 4, 5)

' 各要素を2倍
Dim doubled As stdArray
Set doubled = arr.Map(stdLambda.Create("$1 * 2"))
Debug.Print doubled.Join(",")  ' => 2,4,6,8,10

' オブジェクトのプロパティを抽出
Dim names As stdArray
Set names = stdEnumerator.Create(Application.Workbooks) _
    .Map(stdLambda.Create("$1.Name")).AsArray
```

**Filter（条件に合う要素を抽出）：**

```vb
' 偶数のみ抽出
Dim evens As stdArray
Set evens = arr.Filter(stdLambda.Create("$1 mod 2 = 0"))
Debug.Print evens.Join(",")  ' => 2,4

' 特定条件のオブジェクトを抽出
Dim openBooks As stdEnumerator
Set openBooks = stdEnumerator.Create(Application.Workbooks) _
    .Filter(stdLambda.Create("$1.Saved = False"))
```

**Reduce（集約処理）：**

```vb
' 合計を計算
Dim sum As Long
sum = arr.Reduce(stdLambda.Create("$1 + $2"))
Debug.Print sum  ' => 15

' 最大値を取得
Dim maxVal As Long
maxVal = arr.Reduce(stdLambda.Create("Application.WorksheetFunction.Max($1, $2)"))
Debug.Print maxVal  ' => 5
```

### 5.3 stdCallbackによるコールバック

**モジュール関数からコールバックを作成：**

```vb
' モジュール内の関数
Public Function CalculateArea(radius As Double) As Double
    CalculateArea = 3.14159 * radius * radius
End Function

' コールバックとして使用
Dim cb As stdCallback
Set cb = stdCallback.CreateFromModule("Module1", "CalculateArea")

Dim areas As stdArray
Set areas = stdArray.Create(1, 2, 3, 4, 5).Map(cb)
Debug.Print areas.Join(",")  ' => 3.14159,12.56636,...
```

**引数のバインド：**

```vb
' 引数をバインド
Dim addTen As stdLambda
Set addTen = stdLambda.Create("$1 + $2").Bind(10)
Debug.Print addTen.Run(5)  ' => 15
```

---

## 6. 【STEP3-4】テキスト処理・正規表現

> **業務フローでの位置**: テキストデータのパターンマッチング、抽出、置換を行うステップです。

### 6.1 stdRegexの基本操作

**正規表現オブジェクトの作成：**

```vb
Dim regex As stdRegex
Set regex = stdRegex.Create("(?<county>[A-Z])-(?<city>\d+)-(?<street>\d+)", "i")
```

**フラグオプション：**

| フラグ | 名称 | 説明 |
|-------|------|------|
| i | Case insensitive | 大文字小文字を区別しない |
| g | Global | 全マッチを対象とする |
| m | Multiline | 複数行モード |
| x | Ignore whitespace | 空白を無視 |
| c | Ignore comment | #以降をコメントとして無視 |
| s | Singleline | `.`が改行も含む全文字にマッチ |

### 6.2 パターンマッチング

**テスト（マッチ有無の確認）：**

```vb
Dim regex As stdRegex
Set regex = stdRegex.Create("\d{3}-\d{4}")

Debug.Print regex.Test("123-4567")  ' => True
Debug.Print regex.Test("abc-defg")  ' => False
```

**単一マッチの取得：**

```vb
Dim regex As stdRegex
Set regex = stdRegex.Create("(?<county>[A-Z])-(?<city>\d+)-(?<street>\d+)", "i")

Dim result As Object
Set result = regex.Match("D-040-1425")

Debug.Print result(0)        ' => D-040-1425（全体マッチ）
Debug.Print result(1)        ' => D（サブマッチ1）
Debug.Print result("county") ' => D（名前付きキャプチャ）
Debug.Print result("city")   ' => 040
Debug.Print result("street") ' => 1425
```

**全マッチの取得：**

```vb
Dim regex As stdRegex
Set regex = stdRegex.Create("\d+", "g")

Dim matches As Collection
Set matches = regex.MatchAll("Item1: 100, Item2: 200, Item3: 300")

Dim match As Variant
For Each match In matches
    Debug.Print match(0)  ' => 1, 100, 2, 200, 3, 300
Next
```

### 6.3 テキスト置換

**基本的な置換：**

```vb
Dim regex As stdRegex
Set regex = stdRegex.Create("(\w+)@(\w+)\.com", "g")

Dim result As String
result = regex.Replace("contact: user@example.com", "$1 AT $2 DOT com")
Debug.Print result  ' => contact: user AT example DOT com
```

**名前付きキャプチャを使用した置換：**

```vb
Dim regex As stdRegex
Set regex = stdRegex.Create("(?<name>\w+)@(?<domain>\w+)\.com", "g")

Dim result As String
result = regex.Replace("user@example.com", "$domain/$name")
Debug.Print result  ' => example/user
```

### 6.4 リスト形式での出力

**フォーマット付きリスト：**

```vb
Dim regex As stdRegex
Set regex = stdRegex.Create("(?<Site>\d{5})-(\w{2}\d)", "g")

Dim listStr As String
listStr = regex.List("12345-AA1;67890-BB2", "$Site: $1\n")
Debug.Print listStr
' => 12345: AA1
' => 67890: BB2
```

**Excelへの出力用2次元配列：**

```vb
Dim regex As stdRegex
Set regex = stdRegex.Create("(?<county>[A-Z])-(?<city>\d+)", "g")

Dim arr As Variant
arr = regex.ListArr("A-100;B-200;C-300", Array("$county", "$city"))

' Excelセルに出力
Range("A1:B3").Value = arr
' A1: A, B1: 100
' A2: B, B2: 200
' A3: C, B3: 300
```

---

## 7. 【STEP4-5】システム連携・自動化

> **業務フローでの位置**: ウィンドウ操作、プロセス管理、UI自動化など、システム連携を行うステップです。

### 7.1 stdWindowによるウィンドウ操作

**ウィンドウの取得：**

```vb
' デスクトップウィンドウ
Dim desktop As stdWindow
Set desktop = stdWindow.CreateFromDesktop

' アクティブウィンドウ
Dim activeWin As stdWindow
Set activeWin = stdWindow.CreateFromHwnd(Application.hwnd)

' ウィンドウの検索
Dim notepad As stdWindow
Set notepad = stdWindow.CreateFromQuery(stdLambda.Create("$1.Class like ""Notepad*"""))
```

**ウィンドウの操作：**

```vb
Dim win As stdWindow

' ウィンドウの表示/非表示
win.Visible = True
win.Visible = False

' ウィンドウの位置・サイズ
win.x = 100
win.y = 100
win.Width = 800
win.Height = 600

' ウィンドウの状態
win.State = SW_MAXIMIZE  ' 最大化
win.State = SW_MINIMIZE  ' 最小化
win.State = SW_RESTORE   ' 通常

' アクティブ化
win.Activate
```

### 7.2 stdProcessによるプロセス管理

**プロセスの起動：**

```vb
' プロセスを起動
Dim proc As stdProcess
Set proc = stdProcess.Create("notepad.exe")

' 引数付きで起動
Set proc = stdProcess.Create("notepad.exe ""C:\test.txt""")
```

**プロセス情報の取得：**

```vb
' PIDからプロセスを取得
Dim proc As stdProcess
Set proc = stdProcess.CreateFromProcessId(12345)

' プロセス情報
Debug.Print proc.Name       ' => notepad.exe
Debug.Print proc.Path       ' => C:\Windows\System32\notepad.exe
Debug.Print proc.ProcessId  ' => 12345
```

**プロセスの検索：**

```vb
' 条件に一致するプロセスを検索
Dim proc As stdProcess
Set proc = stdProcess.CreateFromQuery(stdLambda.Create("$1.Name = ""notepad.exe"""))

' 複数プロセスを取得
Dim procs As Collection
Set procs = stdProcess.CreateManyFromQuery(stdLambda.Create("$1.Name like ""*chrome*"""))
```

**プロセスの終了：**

```vb
' 終了待機
proc.waitTilClose

' 強制終了
proc.forceQuit
```

### 7.3 stdAccによるUI自動化

**アクセシビリティ要素の取得：**

```vb
' ウィンドウからアクセシビリティ要素を取得
Dim acc As stdAcc
Set acc = stdAcc.CreateFromHwnd(hwnd)

' 子要素の検索
Dim button As stdAcc
Set button = acc.FindFirst(stdLambda.Create("$1.Role = ""PushButton"" and $1.Name = ""OK"""))
```

**UI操作：**

```vb
' クリック
button.DoDefaultAction

' テキスト設定
textBox.Value = "入力テキスト"

' フォーカス設定
element.SetFocus
```

---

## 8. パフォーマンス計測機能

### 8.1 stdPerformanceの基本操作

**実行時間の計測：**

```vb
' 計測開始
Call stdPerformance.MeasureStart

' 処理を実行
Dim i As Long
For i = 1 To 1000000
    ' 処理
Next

' 計測結果を取得
Dim elapsed As Double
elapsed = stdPerformance.MeasureEnd
Debug.Print "実行時間: " & elapsed & " ms"
```

**Excel最適化設定の一括制御：**

```vb
' 最適化設定（計算・更新を停止）
Call stdPerformance.Optimise

' 処理を実行
' ...

' 設定を復元
Call stdPerformance.UnOptimise
```

### 8.2 stdSentryによるスコープ管理

**Withブロックでの自動復元：**

```vb
' 設定を保存し、終了時に自動復元
With stdSentry.Create()
    Application.ScreenUpdating = False
    Application.Calculation = xlCalculationManual

    ' 処理を実行
    ' ...

End With  ' 設定が自動的に復元される
```

---

## 9. クリップボード操作機能

### 9.1 テキストの操作

**テキストのコピー・ペースト：**

```vb
' テキストをクリップボードに設定
stdClipboard.Text = "Hello, World!"

' クリップボードからテキストを取得
Dim text As String
text = stdClipboard.Text
Debug.Print text  ' => Hello, World!
```

### 9.2 ファイルの操作

**ファイルパスのコピー・ペースト：**

```vb
' ファイルをクリップボードに設定
Dim files As Collection
Set files = New Collection
files.Add "C:\File1.txt"
files.Add "C:\File2.txt"
Set stdClipboard.files = files

' クリップボードからファイルパスを取得
Dim clipFiles As Collection
Set clipFiles = stdClipboard.files
```

### 9.3 画像の操作

**画像のコピー・ペースト：**

```vb
' Excelチャートをコピー
Sheets("Sheet1").ChartObjects(1).Copy

' クリップボードから画像を取得
Dim pic As IPicture
Set pic = stdClipboard.Picture

' 画像をファイルに保存
Call pic.SaveAsFile("C:\chart.bmp", False, Null)
```

---

## 10. HTTP通信・WebSocket機能

### 10.1 stdHTTPの基本操作（HOLD状態）

**注意**: stdHTTPは現在HOLD状態であり、機能が限定的です。

```vb
' GETリクエスト
Dim http As stdHTTP
Set http = stdHTTP.Create("https://api.example.com/data")

Dim response As String
response = http.Get()
```

### 10.2 stdWebSocketの基本操作

**WebSocket接続：**

```vb
' WebSocket接続
Dim ws As stdWebSocket
Set ws = stdWebSocket.Create("wss://echo.websocket.org")

' メッセージ送信
ws.Send "Hello, WebSocket!"

' メッセージ受信（イベント）
' OnMessage イベントで受信
```

---

## 11. 日常開発の流れ

本セクションでは、業務フローに沿った日常開発の流れを説明します。

```
開発フロー概要（再掲）
======================================================
【STEP 1】ライブラリのインポートと初期設定 -> 2章参照
    |
【STEP 2】データの準備（stdArray/stdEnumeratorで構造化） -> 4章参照
    |
【STEP 3】データ変換・処理（stdLambda/stdCallbackで変換） -> 5章参照
    |
【STEP 4】テキスト処理・外部連携（stdRegex/stdWindow等） -> 6,7章参照
    |
【STEP 5】結果の出力と後処理
======================================================
```

### 11.1 【開発者】STEP1: 必要なライブラリのインポート

プロジェクト開始時に、使用するライブラリをインポートします。

1. プロジェクトで使用する機能を特定します
2. 必要なクラスファイルをsrcフォルダから選択します
3. VBAエディタにインポートします
4. 動作確認テストを実行します

### 11.2 【開発者】STEP2: データの準備

データを収集し、stdArrayまたはstdEnumeratorで構造化します。

```vb
' Excelシートからデータを取得
Dim data As stdArray
Set data = stdArray.CreateFromArray(Range("A1:A100").Value)

' または ListObject から取得
Dim rows As stdEnumerator
Set rows = stdEnumerator.CreateFromListObject(Sheet1.ListObjects("Table1"))
```

### 11.3 【開発者】STEP3: データの変換・処理

stdLambdaを使用してデータを変換・フィルタリングします。

```vb
' データのフィルタリングと変換
Dim processed As stdArray
Set processed = data _
    .Filter(stdLambda.Create("$1 <> """"")) _  ' 空白を除外
    .Map(stdLambda.Create("UCase($1)")) _       ' 大文字に変換
    .Unique                                      ' 重複を除去
```

### 11.4 【開発者】STEP4-5: 外部連携と出力

処理結果を出力し、必要に応じて外部システムと連携します。

```vb
' 結果をExcelに出力
Range("B1:B" & processed.Length).Value = Application.Transpose(processed.arr)

' クリップボードにコピー
stdClipboard.Text = processed.Join(vbCrLf)
```

---

## 12. よくある質問と対処法

### 12.1 インポートに関するQ&A

**Q: クラスをインポートしたが「ユーザー定義型は定義されていません」エラーが発生します**

A: 依存するクラスが不足しています。以下を確認してください：
- stdArray/stdLambda/stdCallbackを使用する場合は、必ずstdICallable.clsもインポートしてください
- 依存関係については2.2節の表を参照してください

**Q: 「オブジェクト変数またはWithブロック変数が設定されていません」エラーが発生します**

A: Createメソッドを使用してオブジェクトを生成してください：
```vb
' 正しい
Set arr = stdArray.Create(1, 2, 3)

' 誤り
Set arr = New stdArray
```

### 12.2 stdLambdaに関するQ&A

**Q: ラムダ式で文字列を扱う方法は？**

A: ダブルクォートを2つ重ねて記述します：
```vb
Set lambda = stdLambda.Create("$1 & "" - "" & $2")
```

**Q: ラムダ式でオブジェクトのメソッドを呼び出す方法は？**

A: ドット記法でアクセスできます：
```vb
Set lambda = stdLambda.Create("$1.Cells(1,1).Value")
```

### 12.3 パフォーマンスに関するQ&A

**Q: 大量データ処理時に処理が遅いです**

A: 以下の対策を検討してください：
1. stdPerformance.Optimiseで画面更新と計算を停止
2. チャンキングサイズを調整（stdArray.Create時）
3. stdLambdaのパフォーマンスキャッシュを有効化

---

## 13. トラブルシューティング

### 13.1 エラーメッセージと対処法

| エラーメッセージ | 原因 | 対処法 |
|----------------|------|--------|
| Constructor called on object not class | Createをインスタンスに対して呼び出した | クラス（静的）に対してCreateを呼び出す |
| Method/Property called on class not object | メソッドをクラスに対して呼び出した | Createでインスタンスを生成してから呼び出す |
| Argument N not supplied | stdLambdaで存在しない引数を参照 | 引数の数と$Nの数を確認する |
| Cannot run protInit() on static class | protInitをクラスに対して呼び出した | Createメソッドを使用する |

### 13.2 よくあるトラブルと解決方法

**トラブル: For Eachでオブジェクトが列挙されない**

解決方法: stdEnumeratorを使用するか、NewEnumプロパティが正しく設定されていることを確認してください。

**トラブル: 正規表現がマッチしない**

解決方法:
1. フラグ（i, g, m等）が正しく設定されているか確認
2. 名前付きキャプチャの構文（`(?<name>...)`）が正しいか確認
3. Testメソッドでまずマッチするかどうかをテスト

**トラブル: Mac環境で一部機能が動作しない**

解決方法: Mac環境では以下の機能に制限があります：
- Win32 APIを使用する機能（stdWindow, stdProcess等）
- 一部のCOM操作

### 13.3 サポートへの連絡方法

**GitHub Issues**
- バグ報告: https://github.com/sancarn/stdVBA/issues
- 機能リクエスト: 新しいIssueを作成

**コミュニティ**
- GitHub Discussions: https://github.com/sancarn/stdVBA/discussions
- サンプルコード: https://github.com/sancarn/stdVBA-examples

---

## 改訂履歴

| 版数 | 改訂日 | 改訂内容 |
|-----|--------|---------|
| 1.0 | 2026-01-15 | 初版作成 |

---

*本マニュアルは、stdVBAライブラリの開発者向け運用マニュアルです。*
*ライブラリの更新により、機能や使用方法が変更される場合があります。*
*最新情報はGitHubリポジトリを参照してください。*
