# アーキテクチャ設計書

## 概要

本ドキュメントは、Symfony Framework 8.1（バージョン 8.1.0-DEV）のアーキテクチャ設計について記述する。Symfonyは、PHPで構築された再利用可能なコンポーネント群と、それらを統合するフルスタックWebアプリケーションフレームワークである。本書では、コンポーネントベースのモジュラーアーキテクチャ、レイヤー構成、ミドルウェア構成、データフロー、および横断的関心事について詳述する。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| リレーショナルDB（MySQL, PostgreSQL等） | Doctrine DBAL/ORM（Bridge経由） | データ永続化・クエリ実行 |
| Redis | Predis/phpredis拡張 | キャッシュ、セッション、Lock、Messenger Transport |
| Memcached | ext-memcached | キャッシュ、セッション |
| AMQP（RabbitMQ等） | Messenger Transport | 非同期メッセージング |
| SMTP / メールサービス | Mailer Component / Transport | メール送信 |
| LDAP | Ldap Component（ext-ldap） | ディレクトリサービス認証 |
| 外部HTTP API | HttpClient Component | HTTP通信 |
| 通知サービス（Slack, SMS等） | Notifier Component / Channel | プッシュ通知・メッセージ送信 |
| Twig テンプレートエンジン | Twig Bridge / Twig Bundle | HTMLレンダリング |
| PSR-7 HTTPメッセージ | PsrHttpMessage Bridge | PSR-7互換HTTP処理 |
| Monolog | Monolog Bridge | ログ出力 |

## レイヤー構成

### アーキテクチャスタイル

Symfonyは**コンポーネントベースのレイヤードアーキテクチャ**を採用している。各コンポーネントはデカップルされた独立したライブラリとして設計されており、それぞれが単独でも利用可能であるが、Bundle機構とDependency Injectionコンテナによって統合される。イベント駆動アーキテクチャ（Event-Driven Architecture）を中核パターンとして採用しており、HttpKernelの処理フロー全体がイベントディスパッチにより制御される。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Presentation（プレゼンテーション層） | HTTPリクエスト/レスポンスの処理、ルーティング、コントローラー実行、テンプレートレンダリング | HttpFoundation, HttpKernel, Routing, Twig Bridge/Bundle |
| Application（アプリケーション層） | ビジネスロジックの調整、コマンド/メッセージハンドリング、フォーム処理、バリデーション | Messenger, Form, Validator, Workflow, Scheduler |
| Domain（ドメイン層） | ドメインモデル、シリアライゼーション、式言語、型情報 | Serializer, ExpressionLanguage, TypeInfo, PropertyAccess, PropertyInfo |
| Infrastructure（インフラストラクチャ層） | データアクセス、キャッシュ、ファイルシステム、外部サービス連携、ロック、セマフォ | Cache, Doctrine Bridge, HttpClient, Mailer, Notifier, Lock, Semaphore, Filesystem |
| Cross-Cutting（横断的関心事層） | DI、イベントディスパッチ、セキュリティ、エラーハンドリング、設定管理、ロギング | DependencyInjection, EventDispatcher, Security, ErrorHandler, Config, Monolog Bridge |

### レイヤー間の依存関係ルール

1. **上位レイヤーから下位レイヤーへの依存のみ許可**: Presentation層はApplication層・Infrastructure層に依存できるが、逆方向は禁止
2. **コンポーネント間の疎結合**: 各コンポーネントはインターフェースを通じて連携し、具象実装への直接依存を回避する（PSR標準インターフェースの実装: PSR-3 Logger, PSR-6/PSR-16 Cache, PSR-7 HTTP Message, PSR-11 Container, PSR-14 Event Dispatcher, PSR-20 Clock）
3. **Contracts（契約）による抽象化**: `symfony/contracts`パッケージにより、コンポーネント間のインターフェース契約を定義
4. **Bridge層による外部ライブラリ統合**: Doctrine, Monolog, Twig等の外部ライブラリとの連携はBridge層を通じて行い、フレームワークコアとの直接結合を避ける
5. **Bundle層による機能統合**: Bundleはコンポーネントをフレームワークに統合するための設定・DI定義を提供するが、ビジネスロジックを含まない

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| HTTP処理 | リクエスト/レスポンスの抽象化とカーネル処理 | HttpFoundation, HttpKernel, Routing |
| セキュリティ | 認証・認可・CSRF防御・パスワードハッシュ | Security Core, Security Http, Security Csrf, PasswordHasher, Security Bundle |
| メッセージング | 同期/非同期メッセージバス、トランスポート | Messenger |
| データシリアライゼーション | オブジェクトの変換・正規化 | Serializer, JsonStreamer, ObjectMapper |
| テンプレーティング | ビューレンダリング | Twig Bridge, Twig Bundle |
| フォーム処理 | フォームの構築・検証・レンダリング | Form, Validator |
| コンソール | CLI コマンド処理 | Console |
| キャッシュ | データキャッシュ管理 | Cache |
| 通知 | マルチチャネル通知 | Notifier, Mailer |
| スケジューリング | 定期実行タスク管理 | Scheduler |
| ワークフロー | 状態遷移管理 | Workflow |
| 国際化 | 多言語対応 | Translation, Intl |
| 設定管理 | 設定ファイルの読み込み・検証 | Config, DependencyInjection, Dotenv, OptionsResolver, Yaml |
| アセット管理 | 静的ファイル管理 | Asset, AssetMapper |
| UID管理 | 一意識別子の生成 | Uid |
| リソース制御 | レート制限・ロック・セマフォ | RateLimiter, Lock, Semaphore |

### パッケージ構造

```
src/Symfony/
├── Bridge/                          # 外部ライブラリ統合層
│   ├── Doctrine/                    # Doctrine ORM/DBAL との統合
│   ├── Monolog/                     # Monolog ロガーとの統合
│   ├── PsrHttpMessage/              # PSR-7 HTTP Message との統合
│   └── Twig/                        # Twig テンプレートエンジンとの統合
├── Bundle/                          # フレームワーク統合バンドル
│   ├── DebugBundle/                 # デバッグ機能の統合
│   ├── FrameworkBundle/             # コアフレームワーク機能の統合
│   ├── SecurityBundle/              # セキュリティ機能の統合
│   ├── TwigBundle/                  # Twig テンプレート機能の統合
│   └── WebProfilerBundle/           # Webプロファイラーの統合
└── Component/                       # 独立コンポーネント群
    ├── Asset/                       # アセットバージョニング
    ├── AssetMapper/                 # JavaScriptアセットマッピング
    ├── BrowserKit/                  # ブラウザシミュレーション
    ├── Cache/                       # PSR-6/PSR-16 キャッシュ実装
    ├── Clock/                       # PSR-20 クロック実装
    ├── Config/                      # 設定ファイル管理
    ├── Console/                     # CLIコマンドフレームワーク
    ├── CssSelector/                 # CSS→XPath変換
    ├── DependencyInjection/         # DIコンテナ
    ├── DomCrawler/                  # DOM操作
    ├── Dotenv/                      # .env ファイル管理
    ├── Emoji/                       # 絵文字処理
    ├── ErrorHandler/                # エラーハンドリング
    ├── EventDispatcher/             # PSR-14 イベントディスパッチ
    ├── ExpressionLanguage/          # 式言語エンジン
    ├── Filesystem/                  # ファイルシステム操作
    ├── Finder/                      # ファイル検索
    ├── Form/                        # フォーム処理
    ├── HtmlSanitizer/               # HTMLサニタイズ
    ├── HttpClient/                  # HTTP クライアント
    ├── HttpFoundation/              # HTTP リクエスト/レスポンス抽象化
    ├── HttpKernel/                  # HTTP カーネル（リクエスト処理の中核）
    ├── Intl/                        # 国際化データ
    ├── JsonPath/                    # JSONPath クエリ
    ├── JsonStreamer/                 # JSONストリーム処理
    ├── Ldap/                        # LDAP クライアント
    ├── Lock/                        # 分散ロック
    ├── Mailer/                      # メール送信
    ├── Messenger/                   # メッセージバス
    ├── Mime/                        # MIMEメッセージ構築
    ├── Notifier/                    # マルチチャネル通知
    ├── ObjectMapper/                # オブジェクトマッピング
    ├── OptionsResolver/             # オプション検証
    ├── PasswordHasher/              # パスワードハッシュ
    ├── Process/                     # プロセス管理
    ├── PropertyAccess/              # プロパティアクセス
    ├── PropertyInfo/                # プロパティ情報抽出
    ├── RateLimiter/                 # レート制限
    ├── RemoteEvent/                 # リモートイベント
    ├── Routing/                     # URL ルーティング
    ├── Scheduler/                   # タスクスケジューリング
    ├── Security/                    # セキュリティ
    │   ├── Core/                    # 認証・認可コア
    │   ├── Csrf/                    # CSRF 防御
    │   └── Http/                    # HTTP セキュリティ層（Firewall, Authenticator）
    ├── Semaphore/                   # セマフォ
    ├── Serializer/                  # シリアライゼーション
    ├── Stopwatch/                   # パフォーマンス計測
    ├── String/                      # 文字列操作
    ├── Translation/                 # 翻訳
    ├── TypeInfo/                    # 型情報
    ├── Uid/                         # UUID/ULID生成
    ├── Validator/                   # バリデーション
    ├── VarDumper/                   # 変数ダンプ
    ├── VarExporter/                 # 変数エクスポート
    ├── WebLink/                     # Web Link（HTTP/2 Push等）
    ├── Webhook/                     # Webhook処理
    ├── Workflow/                    # ワークフロー/ステートマシン
    └── Yaml/                        # YAML パーサー
```

### コンポーネント依存関係

主要コンポーネント間の依存関係は以下の通り:

1. **HttpKernel** は HttpFoundation, EventDispatcher, DependencyInjection, Config, ErrorHandler に依存する
2. **FrameworkBundle** は HttpKernel を中心に、ほぼ全てのコンポーネントを統合する中核Bundle
3. **Messenger** は EventDispatcher に依存し、Transport層を通じて外部メッセージブローカーと接続する
4. **Security** は HttpFoundation, EventDispatcher に依存し、Firewall/Authenticator パターンでHTTPリクエストを保護する
5. **Cache** は PSR-6/PSR-16 を実装し、複数のAdapter（Redis, Memcached, Filesystem, Doctrine DBAL等）を提供する
6. **Doctrine Bridge** は Doctrine DBAL/ORM と Symfony コンポーネント（Form, Validator, Security等）を接続する

## ミドルウェア構成

### データベース

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| RDB | Doctrine DBAL | ^4.3 | データベース抽象化レイヤー（MySQL, PostgreSQL, SQLite等） |
| RDB | Doctrine ORM | ^3.4 | オブジェクト-リレーショナルマッピング |
| NoSQL | MongoDB | （ext-mongodb経由） | ドキュメントストア（Lock, Cache等で利用可能） |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| Redis（Predis/phpredis） | predis ^1.1/^2.0 | 分散キャッシュ、セッション、タグ付きキャッシュ | 設定可能 |
| Memcached（ext-memcached） | - | 分散キャッシュ、セッション | 設定可能 |
| APCu（ext-apcu） | - | ローカルインメモリキャッシュ | 設定可能 |
| Filesystem | - | ファイルベースキャッシュ | 設定可能 |
| Couchbase | - | 分散キャッシュ | 設定可能 |
| Doctrine DBAL / PDO | - | データベースベースキャッシュ | 設定可能 |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| AMQP（RabbitMQ等） | ext-amqp | 非同期メッセージトランスポート |
| Redis Streams | predis/phpredis | 非同期メッセージトランスポート |
| Doctrine DBAL | ^4.3 | データベースベーストランスポート |
| Amazon SQS | async-aws/sqs ^1.0/^2.0 | クラウドメッセージキュー |
| Amazon SNS | async-aws/sns ^1.0 | クラウドPub/Sub通知 |
| Beanstalkd | pda/pheanstalk ^5.1/^7.0/^8.0 | 軽量メッセージキュー |
| In-Memory | - | テスト・開発用同期トランスポート |
| Sync | - | 同期メッセージ処理 |

### 検索エンジン

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| - | - | フレームワーク本体には検索エンジン統合は含まれない |

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Twig | ^3.21 | テンプレートエンジン |
| Monolog | ^3.0 | ログ管理 |
| Zookeeper（ext-zookeeper） | - | 分散ロック |
| LDAP（ext-ldap） | - | ディレクトリサービス認証 |
| DynamoDB | async-aws/dynamo-db ^3.0 | 分散ロック・キャッシュバックエンド |

## データフロー

### リクエスト処理の流れ

Symfony HttpKernel はイベント駆動で同期リクエストを処理する。`HttpKernel::handleRaw()` メソッドが中核処理を担う。

1. **Request受信**: `HttpKernel::handle()` が `Request` オブジェクトを受け取り、`RequestStack` にプッシュする
2. **kernel.request イベント**: `EventDispatcher` が `RequestEvent` をディスパッチする。ルーティング（`RouterListener`）はこの段階で実行され、リクエスト属性にルート情報をセットする。Securityの `Firewall` もこのイベントで認証処理を行う。リスナーがレスポンスを設定した場合、以降のステップをスキップする
3. **コントローラー解決**: `ControllerResolver` がリクエストから対応するコントローラーを特定する
4. **kernel.controller イベント**: `ControllerEvent` をディスパッチし、コントローラーの変更を許可する
5. **引数解決**: `ArgumentResolver` がコントローラーの引数を解決する（型宣言に基づく自動注入を含む）
6. **kernel.controller_arguments イベント**: `ControllerArgumentsEvent` をディスパッチし、引数の変更を許可する
7. **コントローラー実行**: コントローラーを呼び出し、Responseを取得する
8. **kernel.view イベント**: コントローラーの戻り値がResponseでない場合、`ViewEvent` をディスパッチしてResponse変換を試みる
9. **kernel.response イベント**: `ResponseEvent` をディスパッチし、レスポンスの修正を許可する（ヘッダー追加、キャッシュ制御等）
10. **レスポンス返却**: 完成した `Response` オブジェクトを返す
11. **kernel.terminate イベント**: レスポンス送信後、`TerminateEvent` をディスパッチし、後処理（ログ書き込み、メール送信等）を実行する

### 非同期処理の流れ

Messenger コンポーネントがメッセージベースの非同期処理を提供する。

1. **メッセージ作成**: アプリケーションコードがメッセージオブジェクトを生成する
2. **MessageBus::dispatch()**: メッセージを `Envelope` でラップし、ミドルウェアチェーンを通過させる
3. **ミドルウェア処理**: `SendMessageMiddleware` がトランスポート設定に基づき、メッセージを非同期トランスポートに送信する。同期処理の場合は `HandleMessageMiddleware` がハンドラーを直接実行する
4. **トランスポート送信**: `TransportInterface::send()` を通じてメッセージブローカー（AMQP, Redis Streams, Database等）にメッセージを投入する
5. **Worker消費**: `Worker` がトランスポートからメッセージを取得し、MessageBus を通じてハンドラーを実行する
6. **リトライ/失敗処理**: 処理失敗時は `RetryStrategy` に基づきリトライを実行し、最終的に失敗トランスポートに移動する

### データ永続化の流れ

Symfony自体はORM/データベースアクセスを直接提供せず、Doctrine Bridge を通じた連携を行う。

1. **エンティティ操作**: アプリケーションコードがドメインエンティティを操作する
2. **EntityManager**: Doctrine ORM の EntityManager がエンティティの変更を追跡する（Unit of Work パターン）
3. **flush**: EntityManager::flush() で変更をSQLに変換し、DBAL を通じてデータベースに反映する
4. **キャッシュ連携**: Cache コンポーネントがクエリ結果やメタデータをキャッシュし、パフォーマンスを向上させる

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| Firewall（イベントリスナー） | Security Http `Firewall` クラス | 全HTTPリクエスト（kernel.request イベントで動作） |
| Authenticator パターン | Security Http `AuthenticatorInterface` 実装群 | FormLogin, HttpBasic, JsonLogin, AccessToken, RememberMe, X509, RemoteUser, LoginLink |
| Voter ベースの認可 | Security Core `Authorization` | リソースアクセス制御 |
| CSRF トークン | Security Csrf `CsrfTokenManager` | フォーム送信の保護 |
| パスワードハッシュ | PasswordHasher Component | ユーザーパスワードの安全な保存 |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | Monolog Bridge（PSR-3 LoggerInterface） | ファイル、stdout、外部サービス（設定依存） |
| デバッグログ | WebProfilerBundle / DebugBundle DataCollector | メモリ/ファイル（開発環境） |
| アクセスログ | Webサーバー側（Symfony外） | Webサーバーログ |
| イベントトレース | TraceableEventDispatcher / Stopwatch | Profiler（開発環境） |

### エラーハンドリング

ErrorHandler コンポーネントがPHPエラーと例外を統合的に処理する。

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| HTTP例外（HttpException） | kernel.exception イベントでキャッチ | 対応するHTTPステータスコード（404, 403, 500等） |
| PHP致命的エラー | ErrorHandler がシャットダウン時に検出 | 500 Internal Server Error |
| バリデーションエラー | Validator Component / Form Component | 422 Unprocessable Entity またはフォーム再表示 |
| PHPワーニング/ノーティス | ErrorHandler が ErrorException に変換 | 開発環境: 詳細エラー、本番: 500 |
| 未捕捉例外 | HttpKernel::handleThrowable() | kernel.exceptionイベント経由でエラーレスポンス生成 |

### トランザクション管理

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| データベーストランザクション | Doctrine DBAL Connection / EntityManager | データベースエンジンのデフォルト（設定可能） |
| Messenger トランザクション | DoctrineTransactionMiddleware（任意） | データベースエンジンのデフォルト |
| 分散ロック | Lock Component（Redis, Database, Zookeeper等） | N/A（排他ロック） |
| セマフォ | Semaphore Component | N/A（並行性制御） |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 依存性逆転の原則（DIP） | 全コンポーネント | インターフェース（PSR標準含む）を通じた依存、ContainerBuilder による DI |
| 単一責任の原則（SRP） | コンポーネント分割 | 各コンポーネントが単一の関心事に特化（HttpFoundation: HTTP抽象化、Cache: キャッシュ等） |
| 開放閉鎖の原則（OCP） | EventDispatcher | イベントリスナー/サブスクライバーによる拡張ポイント（コア変更不要で動作追加可能） |
| コンポジション優先 | Messenger Middleware | ミドルウェアチェーンパターンによる処理の合成 |
| Contract（契約）パターン | symfony/contracts | コンポーネント間のインターフェース契約を独立パッケージとして定義 |
| Decorator パターン | Cache ChainAdapter, TraceableAdapter | 既存アダプターをラップして機能追加 |
| Strategy パターン | Security Authenticator, Cache Adapter | 同一インターフェースで異なる実装を切り替え |
| Observer パターン | EventDispatcher | イベントリスナーによる疎結合なイベント通知 |
| Middleware パターン | Messenger MiddlewareInterface | メッセージ処理チェーン |

### コーディング規約

- **PHP バージョン要件**: PHP >= 8.4
- **PSR準拠**: PSR-1（基本コーディング標準）, PSR-4（オートローディング）, PSR-12（拡張コーディングスタイル）
- **名前空間規約**: `Symfony\Component\{ComponentName}`, `Symfony\Bundle\{BundleName}`, `Symfony\Bridge\{BridgeName}`
- **テスト**: 各コンポーネントに `Tests/` ディレクトリを持ち、PHPUnit によるユニットテストを実施。`Symfony\Bridge\PhpUnit` を拡張として利用
- **後方互換性ポリシー**: セマンティックバージョニングを採用し、UPGRADE ファイル（`UPGRADE-8.0.md`, `UPGRADE-8.1.md`）でマイグレーションガイドを提供
- **非推奨管理**: `@deprecated` アノテーションと `trigger_deprecation()` によるソフトな移行パスの提供
- **型安全性**: PHP 8.4 の型機能（Union Types, Intersection Types, readonly プロパティ、Enums等）を積極活用

## 備考

- 本プロジェクトは Symfony Framework 本体（モノレポ構成）であり、個別アプリケーションではなくフレームワークライブラリそのものである。`replace` セクションにより、モノレポから個別コンポーネントパッケージとして分割リリースされる
- `splitsh.json` の存在から、Git subtree split によるコンポーネント分割リリースが行われていることが推測される
- Symfony 8.1 は PHP 8.4 以上を要求しており、最新のPHP言語機能を前提とした設計である
- `END_OF_MAINTENANCE` および `END_OF_LIFE` は共に `01/2027` と定義されている
