# セキュリティ設計書

## 概要

本ドキュメントは、Symfony 8.1フレームワークのセキュリティアーキテクチャを網羅的に記載したセキュリティ設計書である。Symfonyはセキュリティコンポーネントとして`Security Core`、`Security HTTP`、`Security CSRF`、`PasswordHasher`、`Validator`、`RateLimiter`等の複数サブコンポーネントを提供し、認証・認可・入出力対策・暗号化・セッション管理・監査ログなどの多層防御を実現している。本書では各セキュリティ機能の設計方針、実装構造、および設定可能なパラメータについて記述する。

---

## 認証設計

### 認証方式

Symfony 8.1は`AuthenticatorInterface`に基づく統一的な認証システム（Authenticator-based Security）を採用している。各認証方式はAuthenticatorとして実装され、`AuthenticatorManager`により管理される。Firewall単位で複数のAuthenticatorを組み合わせることが可能である。

#### サポートされる認証方式一覧

| 認証方式 | Authenticatorクラス | 概要 |
| --- | --- | --- |
| フォームログイン | `FormLoginAuthenticator` | HTMLフォームからのユーザー名・パスワード認証。CSRF保護オプション付き。`post_only`、`form_only`の制約が設定可能 |
| HTTP Basic認証 | `HttpBasicAuthenticator` | RFC7617に基づくBasic認証。`PHP_AUTH_USER`/`PHP_AUTH_PW`ヘッダーから資格情報を取得 |
| JSONログイン | `JsonLoginAuthenticator` | JSON形式のリクエストボディからユーザー名・パスワードを取得するステートレス認証 |
| アクセストークン認証 | `AccessTokenAuthenticator` | RFC6750に基づくBearerトークン認証。`AccessTokenExtractorInterface`と`AccessTokenHandlerInterface`により拡張可能 |
| Remember Me認証 | `RememberMeAuthenticator` | セッション期限切れ後のCookie基盤の再認証。`SignatureRememberMeHandler`または`PersistentRememberMeHandler`と連携 |
| リモートユーザー認証 | `RemoteUserAuthenticator` | リバースプロキシ等の外部認証済みユーザーの受け入れ |
| X.509証明書認証 | `X509Authenticator` | クライアント証明書に基づく認証 |
| ログインリンク認証 | `LoginLinkAuthenticator` | ワンタイムURLによるパスワードレス認証。`SignatureHasher`によるHMAC-SHA256署名で保護 |
| OIDC認証 | `OidcTokenHandler` | OpenID Connectトークンの検証。JWS署名検証、JWE復号、クレーム検証（iss, aud, exp, nbf, iat）を実施。JWKS Discovery対応 |
| OAuth2認証 | `Oauth2TokenHandler` | OAuth2トークンイントロスペクションによる認証 |

#### 認証フロー（Passport/Badge方式）

Symfony 8.1の認証システムは`Passport`オブジェクトを中心に設計されている。

1. `Authenticator::supports(Request)` でリクエストの対応可否を判定
2. `Authenticator::authenticate(Request)` で`Passport`を生成（`UserBadge`、`CredentialsInterface`、追加Badgeを含む）
3. `CheckPassportEvent`が発火し、各イベントリスナーがBadgeを検証
   - `CheckCredentialsListener` - パスワードハッシュの検証
   - `CsrfProtectionListener` - CSRFトークンの検証
   - `LoginThrottlingListener` - ログイン試行回数の検証
4. `Authenticator::createToken(Passport, firewallName)` で認証トークンを生成
5. 成功/失敗ハンドラーが応答を処理

#### 利用可能なBadge

| Badge | 用途 |
| --- | --- |
| `UserBadge` | ユーザー識別子とUserLoaderを保持 |
| `PasswordCredentials` | パスワード資格情報を保持 |
| `CustomCredentials` | カスタム検証ロジックを保持 |
| `CsrfTokenBadge` | CSRFトークンIDと値を保持 |
| `RememberMeBadge` | Remember Me機能の有効化 |
| `PasswordUpgradeBadge` | パスワードの自動リハッシュ |
| `PreAuthenticatedUserBadge` | 事前認証済みユーザーのマーク |

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション固定化対策 | `migrate`（デフォルト推奨） | `SessionAuthenticationStrategy`がサポートする戦略: `none`（変更なし）、`migrate`（セッションID再生成、属性保持）、`invalidate`（セッションID再生成、属性破棄） |
| CSRFトークン連携 | `ClearableTokenStorageInterface`による自動クリア | `migrate`戦略時にCSRFトークンストレージをクリアし、セッション固定化攻撃とCSRFトークン再利用を防止 |
| セッション有効期限 | フレームワーク設定（`framework.session`）で定義 | PHP標準の`session.gc_maxlifetime`等をSymfony設定から制御 |

#### Remember Meクッキー設定

| 項目 | デフォルト値 | 備考 |
| --- | --- | --- |
| クッキー名 | `REMEMBERME` | 設定変更可能 |
| 有効期間 | 31536000秒（1年） | `lifetime`オプションで変更可能 |
| HttpOnly | `true` | JavaScriptからのアクセスを防止 |
| Secure | `false`（リクエストのisSecure()にフォールバック） | HTTPS環境ではtrueが推奨 |
| SameSite | `null`（未設定） | `strict`または`lax`の設定が推奨 |

---

## 認可設計

### 権限体系

Symfony 8.1はロールベースアクセス制御（RBAC）とVoterパターンを組み合わせた柔軟な認可システムを提供する。

| ロール/属性 | 権限 | 説明 |
| --- | --- | --- |
| `ROLE_*` | ロールベースの権限 | `RoleVoter`が`ROLE_`プレフィックスの属性を評価。`RoleHierarchyVoter`によるロール階層の定義が可能 |
| `IS_AUTHENTICATED_FULLY` | 完全認証 | 現セッションでユーザー名/パスワード等により認証されたユーザー |
| `IS_AUTHENTICATED_REMEMBERED` | Remember Me含む認証 | 完全認証またはRemember Meによる認証 |
| `IS_AUTHENTICATED` | 認証済み | いずれかの方法で認証されたユーザー |
| `IS_IMPERSONATOR` | なりすまし中 | `SwitchUserToken`によるユーザー切り替え中のユーザー |
| `IS_REMEMBERED` | Remember Me認証 | Remember Meクッキーによる認証のみ |
| `PUBLIC_ACCESS` | 公開アクセス | 認証不要のリソース |

#### Voter一覧

| Voter | 役割 |
| --- | --- |
| `RoleVoter` | `ROLE_`プレフィックスの属性をユーザーのロールと照合 |
| `RoleHierarchyVoter` | ロール階層を考慮したロール評価 |
| `AuthenticatedVoter` | 認証レベル（FULLY/REMEMBERED/AUTHENTICATED等）の評価 |
| `ExpressionVoter` | ExpressionLanguageによる複合条件の評価 |
| `ClosureVoter` | クロージャベースの動的権限評価 |

### アクセス制御

#### AccessDecisionManager

`AccessDecisionManager`が全てのアクセス決定を統括する。投票結果の集約に以下の戦略（Strategy）が利用可能。

| 戦略 | クラス | 動作 |
| --- | --- | --- |
| Affirmative（デフォルト） | `AffirmativeStrategy` | 1つでもACCESS_GRANTEDがあれば許可 |
| Consensus | `ConsensusStrategy` | 多数決で決定 |
| Unanimous | `UnanimousStrategy` | 全てのVoterがACCESS_GRANTEDの場合のみ許可 |
| Priority | `PriorityStrategy` | 最初にACCESS_ABSTAINでない結果を返したVoterの結果を採用 |

#### Firewall / AccessMap

- `Firewall`がHTTPカーネルイベントをリッスンし、リクエストに対応するセキュリティリスナーを登録する
- `AccessMap`がURLパターンとHTTPメソッドに基づくアクセス制御ルールを定義し、必要なロールやチャネル（HTTP/HTTPS）を指定する
- `#[IsGranted]`アトリビュートにより、コントローラーメソッド単位でロールや式による認可チェックを宣言的に適用可能。HTTPメソッドフィルタリングもサポート

#### ユーザー切り替え（Impersonation）

`SwitchUserToken`と`ImpersonateUrlGenerator`により、管理者が他のユーザーとしてシステムを操作する機能を提供。`IS_IMPERSONATOR`属性で制御される。

---

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | `AccessMap`でチャネル制約（`https`）を設定可能。CSRFトークンマネージャーはHTTPS接続を自動検出し、名前空間に`https-`プレフィックスを付与 |
| HSTS | フレームワーク設定またはWebサーバー設定で対応。Symfonyはレスポンスヘッダーのカスタマイズをサポート |
| Clear-Site-Data | `ClearSiteDataLogoutListener`がログアウト時に`Clear-Site-Data`ヘッダーを設定。`cache`、`cookies`、`storage`、`clientHints`、`executionContexts`、`prefetchCache`、`prerenderCache`、`*`を指定可能 |
| CSRFクッキーのHTTPSプレフィックス | `SameOriginCsrfTokenManager`がHTTPS接続時にクッキー名に`__Host-`プレフィックスを自動付与し、HTTP経由での偽造を防止 |

---

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| パスワードハッシュ（推奨） | `NativePasswordHasher` - PHPネイティブの`password_hash()`を使用。bcrypt（デフォルト、cost=13）、Argon2i、Argon2idをサポート。bcryptの72バイト制限はSHA-512のBase64エンコードで回避 |
| パスワードハッシュ（Sodium） | `SodiumPasswordHasher` - libsodiumの`sodium_crypto_pwhash_str()`を使用。Argon2idアルゴリズム。opsLimit >= 3、memLimit >= 10KBの制約 |
| パスワードハッシュ（PBKDF2） | `Pbkdf2PasswordHasher` - PBKDF2アルゴリズム。SHA-512デフォルト、iterations=1000 |
| パスワードハッシュ（自動） | `MigratingPasswordHasher` - 複数ハッシャーのチェーン。`auto`アルゴリズム設定で最適なハッシャーを自動選択し、旧ハッシュからの自動マイグレーションを実現 |
| 署名（LoginLink/RememberMe） | `SignatureHasher` - HMAC-SHA256を使用。ユーザー属性、有効期限、秘密鍵に基づくハッシュを生成。タイミング攻撃対策として`hash_equals()`を使用 |
| CSRFトークン | `CsrfTokenManager` - トークンのランダム化（XOR + 32バイトランダムキー）とタイミングセーフな比較（`hash_equals()`） |
| OIDCトークン | JWS署名検証（`JWSVerifier`）とオプションのJWE復号（`JWEDecrypter`）。アルゴリズムチェッカーによる許可アルゴリズムの制限 |

### パスワード長制御

`CheckPasswordLengthTrait`により全パスワードハッシャーに最大パスワード長（4096バイト）の制限が適用される。空パスワードも明示的に拒否される。

### パスワード自動アップグレード

`PasswordUpgradeBadge`と`PasswordMigratingListener`により、ログイン時にパスワードハッシュを最新のアルゴリズムに自動アップグレードする仕組みを提供。`UserProvider`が`PasswordUpgraderInterface`を実装している場合に動作する。

### 機密情報管理

- `#[\SensitiveParameter]`アトリビュートにより、スタックトレースからパスワードや秘密鍵の値を隠蔽
- `DefaultLoginRateLimiter`はIPアドレスとユーザー名をHMAC-SHA256でハッシュ化してからレートリミッターのキーとして使用し、個人情報の保存を回避
- `JsonLoginAuthenticator`はパスワード取得後、リクエストデータ上のパスワードフィールドを`null`に設定して上書き
- 署名ハッシャーは空の秘密鍵を明示的に拒否（`InvalidArgumentException`をスロー）

---

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | Twig テンプレートエンジンによる自動出力エスケープ（デフォルト有効）。`html`、`js`、`css`、`url`、`html_attr`コンテキストに対応。`{{ variable\|e('html') }}`による明示的エスケープも可能 |
| SQLインジェクション | Doctrine ORMのパラメータバインディング（プリペアドステートメント）による保護。`Symfony\Bridge\Doctrine`がDoctrineとの統合を提供 |
| CSRF | 二重の防御戦略: (1) `CsrfTokenManager`によるトークンベース保護（セッション保存、XORランダム化、`hash_equals`比較）、(2) `SameOriginCsrfTokenManager`によるステートレス保護（Same-Origin検証 + ダブルサブミットクッキー）。`#[IsCsrfTokenValid]`アトリビュートによる宣言的CSRF保護も提供 |
| ブルートフォース攻撃 | `LoginThrottlingListener`と`DefaultLoginRateLimiter`による二層レート制限: グローバル（IPベース）とローカル（ユーザー名+IPベース）。`PeekableRequestRateLimiterInterface`による消費前チェックをサポート |
| 入力検証 | `Validator`コンポーネントによる包括的な入力検証。Email、URL、IP、CIDR、Charset、CssColor、CardScheme、BIC等の多数の組み込みバリデーション制約。カスタムバリデーション制約の定義も可能 |
| 型検証 | `FormLoginAuthenticator`、`JsonLoginAuthenticator`がユーザー名・パスワードの型（文字列）と空文字列を厳密にチェック。不正な型は`BadRequestHttpException`、空文字列は`BadCredentialsException`をスロー |

### CSRF保護の詳細

#### トークンベース CSRF保護（CsrfTokenManager）

- セッションストレージ（`NativeSessionTokenStorage`）またはカスタムストレージにトークンを保存
- HTTPS/HTTPプロトコルに応じた名前空間分離
- トークン値のXORランダム化により、ネットワーク上での再利用を防止
- `hash_equals()`によるタイミングセーフな比較

#### Same-Origin CSRF保護（SameOriginCsrfTokenManager）

ステートレスでHTTPキャッシュ互換のCSRF保護を提供。

1. `Sec-Fetch-Site: same-origin`ヘッダーによる検証
2. `Origin`/`Referer`ヘッダーのオリジン一致検証
3. ダブルサブミットクッキー（リクエストペイロードとクッキーの値照合）
4. オプションでヘッダーによるダブルサブミット（`CHECK_HEADER`/`CHECK_ONLY_HEADER`モード）
5. セッション存在時は検証方式のダウングレード防止（ビットフラグによる履歴管理）
6. HTTPS時は`__Host-`プレフィックス付きクッキーによるHTTPチャネル偽造防止
7. レスポンス時のクッキー自動クリア

---

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| 認証失敗ログ | `HttpBasicAuthenticator`がユーザー名と例外を`info`レベルで記録。`LoginThrottlingListener`がレート制限超過を検出 | PSR-3ロガー設定に依存 |
| Remember Meログ | ユーザー未検出（`info`）、非対応ユーザークラス（`warning`）、認証失敗（`debug`）、クッキー受理（`info`）、クッキークリア（`debug`）を記録 | PSR-3ロガー設定に依存 |
| CSRF検証ログ | `SameOriginCsrfTokenManager`がCSRF検証の成功/失敗を`debug`/`warning`/`error`レベルで詳細に記録（オリジン不一致、ダブルサブミット失敗、検証方式ダウングレード検出等） | PSR-3ロガー設定に依存 |
| OIDCトークン検証ログ | トークンのデコード/検証エラーを`error`レベルで記録。復号スキップを`debug`レベルで記録 | PSR-3ロガー設定に依存 |
| アクセス決定ログ | `TraceableAccessDecisionManager`と`Vote`オブジェクトにより、各Voterの投票結果と理由を記録（デバッグ/プロファイラー用） | WebProfiler設定に依存 |

---

## 追加セキュリティ機能

### Content Security Policy（CSP）

`ContentSecurityPolicyHandler`（WebProfilerBundle）がCSPヘッダーの解析・更新を提供。
- `Content-Security-Policy`、`Content-Security-Policy-Report-Only`、`X-Content-Security-Policy`ヘッダーをサポート
- `script-src`、`script-src-elem`、`style-src`、`style-src-elem`ディレクティブにnonce値を自動挿入
- `NonceGenerator`による暗号学的に安全なnonce生成

### ユーザーチェック

- `UserCheckerInterface`により認証成功後のユーザー状態チェック（アカウント無効化、期限切れ等）を実施
- `ChainUserChecker`による複数チェッカーの連鎖実行が可能
- `InMemoryUserChecker`がデフォルト実装

### ログアウトセキュリティ

- `ClearSiteDataLogoutListener` - `Clear-Site-Data`ヘッダーによるクライアントデータクリア
- `CookieClearingLogoutListener` - 指定クッキーのクリア
- `CsrfTokenClearingLogoutListener` - CSRFトークンストレージのクリア
- `SessionLogoutListener` - セッションの無効化
- `DefaultLogoutListener` - ログアウト後のリダイレクト

### ExpiredSignatureStorage

`SignatureHasher`と連携し、署名付きURL（ログインリンク等）の使用回数を制限する。`maxUses`パラメータで最大使用回数を設定可能。

---

## 備考

- 本プロジェクトはSymfony 8.1フレームワークのソースコードであり、PHP 8.4以上が必須要件である
- セキュリティ設定の多くはアプリケーション側の`security.yaml`で定義される。本書はフレームワークが提供するセキュリティ機構の設計を記載している
- `SameOriginCsrfTokenManager`の`clearCookies()`、`persistStrategy()`、`onKernelResponse()`メソッドはSymfony 8.1で非推奨となり、`SameOriginCsrfListener`への移行が推奨されている
- LDAP認証は`symfony/ldap`コンポーネントにより別途サポートされている
- `web-token/jwt-library`パッケージ（dev依存）によりJWT/JWS/JWE操作が提供される
