# バッチ設計書 8-secrets:set

## 概要

本ドキュメントは、Symfonyフレームワークが提供する `secrets:set` コンソールコマンドのバッチ設計書である。暗号化されたVaultにシークレット（機密情報）を設定・保存する処理の仕様を定義する。

### 本バッチの処理概要

`secrets:set` コマンドは、Symfonyのシークレット管理システム（Vault）に暗号化されたシークレット値を保存する。対話的入力、ファイルからの読み取り、STDIN、ランダム値の自動生成など複数の入力方式をサポートする。

**業務上の目的・背景**：アプリケーションで使用するデータベースパスワード、APIキー、暗号化キーなどの機密情報を安全に管理するために使用する。環境変数ファイル（.env）にプレーンテキストで保存する代わりに、暗号化されたVaultに格納することで、ソースコードリポジトリへの機密情報漏洩リスクを大幅に低減できる。ローカルVaultによるオーバーライド機能により、開発環境と本番環境で異なるシークレット値を安全に管理できる。

**バッチの実行タイミング**：新しいシークレットの追加時、既存シークレットの値変更時に随時実行される。初期セットアップ時やデプロイパイプラインの一環として使用されることもある。

**主要な処理内容**：
1. シークレット名と対象Vault（メインまたはローカル）を決定
2. ローカルVault使用時、メインVaultにシークレットが存在するか確認
3. シークレット値の取得（対話入力/ファイル/STDIN/ランダム生成）
4. 暗号化キーが存在しない場合は自動生成
5. Vaultにシークレットを暗号化して保存（seal）
6. ローカルVaultでオーバーライドされている場合は注意メッセージを表示

**前後の処理との関連**：`secrets:generate-keys` で暗号化キーを事前生成しておくことが推奨されるが、本コマンドが自動的にキーを生成することも可能。`secrets:list` で設定済みシークレットを確認でき、`secrets:remove` でシークレットを削除できる。

**影響範囲**：指定されたVault（メインまたはローカル）内の指定シークレットのみが影響を受ける。暗号化キーが新規生成された場合は、キーファイルも作成される。

## バッチ種別

シークレット管理 / セキュリティ設定コマンド

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時 |
| 実行時刻 | 任意 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 手動 |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Vault設定 | シークレット管理（Vault）が設定済みであること |
| ローカルVault使用時 | ローカルVaultが有効であること。メインVaultに対象シークレットが存在すること |

### 実行可否判定

- `--local` オプション指定時にローカルVaultが無効（null）の場合、エラーメッセージを出力して終了コード1を返す。
- ローカルVaultへの設定時、メインVaultに該当シークレットが存在しない場合、エラーメッセージを出力して終了コード1を返す（ローカルVaultはメインVaultのオーバーライドのみ可能）。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| name | string | Yes | なし | シークレット名 |
| file | string | No | なし | シークレット値を読み取るファイルパス。"-"でSTDIN |
| --local / -l | bool | No | false | ローカルVaultに保存する |
| --random / -r | int\|null | No | false（無効） | ランダム値を生成する。値を指定すると長さを指定（デフォルト16） |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| AbstractVault (vault) | サービス | メインVault |
| AbstractVault (localVault) | サービス | ローカルVault（オプション） |
| 対話入力 | 標準入力 | ファイル/ランダム未指定時、対話的にシークレット値を入力 |
| ファイル | テキスト | 指定されたファイルからシークレット値を読み取り |
| STDIN | テキスト | "-"指定時、標準入力からシークレット値を読み取り |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| Vault | 暗号化ファイル | 暗号化されたシークレットデータ |
| stderr | テキスト | 処理メッセージ（エラー出力に送信） |
| stdout | テキスト | ランダム生成値の出力 |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | Vault実装に依存（SodiumVaultの場合: {name}.{hash}.php） |
| 出力先 | Vault設定ディレクトリ |
| 文字コード | バイナリ（暗号化データ） |
| 区切り文字 | 該当なし |

## 処理フロー

### 処理シーケンス

```
1. 引数・オプション解析
   └─ シークレット名、ファイル、--local、--randomオプションを取得
2. 対象Vault決定
   └─ --local指定時はローカルVault、未指定時はメインVault
3. Vault有効性確認
   └─ ローカルVault指定時にnullならエラー終了
4. ローカルVault制約確認
   └─ ローカルVault指定時、メインVaultにシークレットが存在するか確認
5. シークレット値取得
   └─ --random指定時: ランダム値生成（base64エンコード、URL安全文字に変換）
   └─ ファイル未指定時: 対話入力（askHidden）
   └─ "-"指定時: STDIN読み取り
   └─ ファイル指定時: file_get_contents()
6. 暗号化キー確認・生成
   └─ generateKeys()で必要に応じてキーを生成
7. シークレット保存
   └─ vault.seal()で暗号化して保存
8. 結果出力
   └─ 成功メッセージ、ランダム値表示、ローカルオーバーライド注意メッセージ
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[引数・オプション取得]
    B --> C{--local指定?}
    C -->|Yes| D{ローカルVault有効?}
    C -->|No| E[メインVaultを使用]
    D -->|No| F[エラー: ローカルVault無効]
    D -->|Yes| G{メインVaultにシークレット存在?}
    G -->|No| H[エラー: シークレット未登録]
    G -->|Yes| I[ローカルVaultを使用]
    E --> J[シークレット値取得]
    I --> J
    J --> K{--random指定?}
    K -->|Yes| L[ランダム値生成]
    K -->|No| M{file引数あり?}
    M -->|No| N[対話入力 askHidden]
    M -->|"-"| O[STDIN読み取り]
    M -->|ファイルパス| P[ファイル読み取り]
    L --> Q[generateKeys確認・生成]
    N --> Q
    O --> Q
    P --> Q
    Q --> R[vault.seal で暗号化保存]
    R --> S[成功メッセージ出力]
    S --> T{ランダム値?}
    T -->|Yes| U[ランダム値を出力]
    T -->|No| V{ローカルオーバーライド存在?}
    U --> V
    V -->|Yes| W[注意メッセージ出力]
    V -->|No| X[バッチ終了 コード0]
    W --> X
```

## データベース操作仕様

### 操作別データベース影響一覧

| 処理 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| 該当なし | - | - | データベース操作なし（ファイルベースのVault） |

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 終了コード1 | Vault無効 | ローカルVaultが無効（null） | Vault設定を確認する |
| 終了コード1 | シークレット未登録 | ローカルVaultへの設定時、メインVaultに未登録 | 先にメインVaultにシークレットを登録する |
| InvalidArgumentException | ファイル不在 | 指定ファイルが存在しない | ファイルパスを確認する |
| InvalidArgumentException | ファイル読み取り不可 | 指定ファイルが読み取り不可 | ファイル権限を確認する |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

1. Vault設定に問題がある場合は、`framework.secrets` 設定を確認する。
2. 暗号化キーが紛失した場合は、`secrets:generate-keys` で再生成が必要（既存シークレットは復号不可）。
3. ファイル読み取りに失敗した場合は、ファイルの存在とパーミッションを確認する。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 単一シークレットの保存操作 |
| コミットタイミング | seal()完了時 |
| ロールバック条件 | 該当なし |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1件（単一シークレット） |
| 目標処理時間 | ミリ秒〜秒単位 |
| メモリ使用量上限 | 最小限 |

## 排他制御

Vaultファイルへの同時書き込みに対する明示的な排他制御はコマンド内には実装されていない。同時に複数プロセスから同一シークレットへの書き込みを行った場合、最後の書き込みが有効となる。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 成功ログ | キー生成時 | Vaultからのメッセージ |
| 警告ログ | キー生成時（prod） | `DO NOT COMMIT THE DECRYPTION KEY FOR THE PROD ENVIRONMENT` |
| 成功ログ | シークレット保存時 | `Secret was successfully stored in the vault.` |
| 情報ログ | ランダム値生成時 | `The generated random value is: {value}` |
| 注意ログ | ローカルオーバーライド時 | `Note that this secret is overridden in the local vault.` |
| 警告ログ | 空値入力時 | `No value provided: using empty string` |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 終了コード | 0以外 | 運用監視システム |

## 備考

- シェル補完機能をサポートしており、シークレット名の入力時に既存シークレット名が候補として表示される（`vault.list(false)` による）。
- ランダム値生成は `random_bytes()` を使用し、base64エンコード後に `+/` を `-_` に置換したURL安全な文字列を生成する。
- メッセージ出力はstderrに送信される（`ConsoleOutputInterface::getErrorOutput()` 使用）が、ランダム値はstdoutに出力される。これにより、パイプラインでランダム値のみを取得可能である。
- 本コマンドは `@internal` マーク付きであり、フレームワーク内部使用を前提としている。
- ソースコード: `src/Symfony/Bundle/FrameworkBundle/Command/SecretsSetCommand.php`
