# 機能設計書 15-OptionsResolver

## 概要

本ドキュメントは、Symfony OptionsResolverコンポーネントの機能設計を記述する。OptionsResolverは、PHPのarray_replace関数の改良版として、オプション解決機能を提供し、設定オプションの定義、デフォルト値、バリデーションを統合管理する。

### 本機能の処理概要

**業務上の目的・背景**：PHPにおいてオプション配列を引数として受け取るパターンは広く使われるが、型チェック、必須チェック、デフォルト値の適用、正規化処理を個々に実装するのは冗長である。OptionsResolverはこれらを宣言的に定義し、一貫したオプション処理を提供する。Formコンポーネントのオプション管理を始め、多くのSymfonyコンポーネントで利用される。

**機能の利用シーン**：フォーム型（FormType）のオプション定義、Bundle設定のオプション処理、カスタムサービスの設定管理に利用される。

**主要な処理内容**：
1. オプションの定義と必須設定（setDefined、setRequired）
2. デフォルト値の設定（setDefault、遅延評価対応）
3. 許容値と許容型の設定（setAllowedValues、setAllowedTypes）
4. 正規化処理（setNormalizer）
5. ネストされたオプションの解決（setDefault + OptionsResolver引数のクロージャ）
6. オプションの解決と検証（resolve）
7. 情報メッセージの設定（setInfo）

**関連システム・外部連携**：Formコンポーネント（AbstractType::configureOptions）、各種Symfonyコンポーネントの内部設定管理で利用される。

**権限による制御**：本コンポーネントは権限制御を行わない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | （直接的な関連画面なし） | - | フォームパネル等を通じて間接的に影響 |

## 機能種別

バリデーション / 設定管理 / デフォルト値解決

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| option | string | Yes | オプション名 | 定義済みオプションであること |
| value | mixed | No | オプション値 | setAllowedTypes/setAllowedValuesで定義された制約 |
| default | mixed/\Closure | No | デフォルト値（Closureの場合は遅延評価） | - |
| allowedValues | array | No | 許容値のリスト | - |
| allowedTypes | string[] | No | 許容型名のリスト | VALIDATION_FUNCTIONS定数に対応する型名 |
| normalizer | \Closure | No | 正規化クロージャ | - |

### 入力データソース

アプリケーションコード（フォーム型定義、Bundle設定等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| resolvedOptions | array | 解決済みオプション配列（デフォルト値適用、正規化済み、バリデーション済み） |

### 出力先

呼び出し元への返却値

## 処理フロー

### 処理シーケンス

```
1. オプション定義フェーズ
   └─ setDefined、setRequired、setDefault、setAllowedTypes、setAllowedValues、setNormalizer
2. 解決フェーズ（resolve呼び出し）
   └─ 2a. 未定義オプションのチェック（UndefinedOptionsException）
   └─ 2b. 必須オプションの欠落チェック（MissingOptionsException）
   └─ 2c. デフォルト値の適用（遅延評価の実行を含む）
   └─ 2d. ネストオプションの再帰的解決
   └─ 2e. 型チェック（InvalidOptionsException）
   └─ 2f. 許容値チェック（InvalidOptionsException）
   └─ 2g. 正規化処理の実行
3. 結果返却
   └─ 解決済みオプション配列の返却
```

### フローチャート

```mermaid
flowchart TD
    A[resolve呼び出し] --> B{未定義オプションあり?}
    B -->|Yes| C[UndefinedOptionsException]
    B -->|No| D{必須オプション欠落?}
    D -->|Yes| E[MissingOptionsException]
    D -->|No| F[デフォルト値適用]
    F --> G{遅延評価?}
    G -->|Yes| H[Closure実行]
    G -->|No| I[値をそのまま使用]
    H --> J{ネストオプション?}
    I --> J
    J -->|Yes| K[再帰的にresolve]
    J -->|No| L[型チェック]
    K --> L
    L --> M{型が許容?}
    M -->|No| N[InvalidOptionsException]
    M -->|Yes| O[許容値チェック]
    O --> P{値が許容?}
    P -->|No| N
    P -->|Yes| Q[正規化処理]
    Q --> R[解決済み配列返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-15-01 | 遅延評価 | Closureがデフォルト値として設定された場合、resolve時に初めて実行される | setDefault + resolve |
| BR-15-02 | ネストオプション | OptionsResolverを引数に持つClosureは、ネストされたオプション定義として処理 | setDefault + Closure(OptionsResolver) |
| BR-15-03 | 正規化チェーン | 同一オプションに複数のnormalizerを設定した場合、登録順に実行される | setNormalizer（force=false） |
| BR-15-04 | 型検証 | VALIDATION_FUNCTIONS定数に定義されたPHP組み込み型とクラス名/インターフェース名をサポート | setAllowedTypes + resolve |
| BR-15-05 | 解決後のロック | resolve実行後はオプション定義の変更が禁止される（AccessException） | resolve後のset系メソッド呼び出し |

### 計算ロジック

型検証: VALIDATION_FUNCTIONS定数（bool、int、float、string、array等）の組み込み型チェック関数と、class_exists/is_a/interface_existsによるクラス/インターフェース型チェックを組み合わせる。

## データベース操作仕様

### 操作別データベース影響一覧

本コンポーネントはデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | UndefinedOptionsException | 未定義のオプションが渡された | setDefinedでオプションを定義するか、渡すオプション名を修正 |
| - | MissingOptionsException | 必須オプションが未設定 | 必須オプションに値を設定 |
| - | InvalidOptionsException | 型チェックまたは許容値チェックに失敗 | setAllowedTypes/setAllowedValuesの定義に従った値を設定 |
| - | AccessException | 解決後にオプション定義を変更しようとした | resolve前にすべての定義を完了する |
| - | OptionDefinitionException | 不正なオプション定義 | オプション定義の整合性を確認 |

### リトライ仕様

リトライは行わない。

## トランザクション仕様

トランザクション管理は行わない。

## パフォーマンス要件

- resolve処理は各オプションに対して線形時間で実行
- 遅延評価により、実際に使用されるオプションのみ計算コストが発生
- ネストオプションは再帰的に解決されるため、深いネストではスタック使用量に注意

## セキュリティ考慮事項

- 型チェック機能により不正な型の値が混入することを防止
- 許容値チェックにより不正な値の注入を防止

## 備考

- OptionConfiguratorクラスにより、fluent APIでオプションを定義可能
- Optionsインターフェースを実装し、ArrayAccess/Countableに対応
- VALIDATION_FUNCTIONS定数でPHPの型名とis_*関数の対応を管理

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Options.php | `src/Symfony/Component/OptionsResolver/Options.php` | Optionsインターフェース（ArrayAccess/Countable） |
| 1-2 | OptionsResolver.php（プロパティ） | `src/Symfony/Component/OptionsResolver/OptionsResolver.php` | 内部状態プロパティ（defined、defaults、required、normalizers、allowedValues、allowedTypes等） |

**読解のコツ**: OptionsResolverの内部状態は多数の配列で管理される。defined（定義済み名）、defaults（デフォルト値）、required（必須名）、nested（ネスト定義）、normalizers（正規化関数）、allowedValues（許容値）、allowedTypes（許容型）の関係を理解する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | OptionsResolver.php（resolve） | `src/Symfony/Component/OptionsResolver/OptionsResolver.php` | resolveメソッドが主要エントリーポイント。未定義チェック、必須チェック、デフォルト適用、型チェック、許容値チェック、正規化の順で処理 |

**主要処理フロー**:
1. **30-48行目**: VALIDATION_FUNCTIONS定数。PHP組み込み型名とis_*関数の対応表
2. **53-100行目**: 内部プロパティ定義。defined、defaults、nested、required、resolved、normalizers、allowedValues、allowedTypes、info等

#### Step 3: オプション定義APIを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | OptionConfigurator.php | `src/Symfony/Component/OptionsResolver/OptionConfigurator.php` | Fluent APIによるオプション設定。setRequired、setDefault等のラッパー |

### プログラム呼び出し階層図

```
OptionsResolver::resolve()
    │
    ├─ 未定義オプションチェック
    ├─ 必須オプションチェック
    │
    ├─ offsetGet()（各オプションの解決）
    │      ├─ デフォルト値のClosure実行（遅延評価）
    │      ├─ ネストオプションの再帰的resolve
    │      ├─ 型チェック（VALIDATION_FUNCTIONS / class_exists）
    │      ├─ 許容値チェック
    │      └─ 正規化処理
    │
    └─ 解決済み配列の返却

OptionsResolver（定義API）
    ├─ setDefined()
    ├─ setRequired()
    ├─ setDefault()
    ├─ setAllowedTypes()
    ├─ setAllowedValues()
    ├─ setNormalizer()
    └─ setInfo()
```

### データフロー図

```
[入力]                    [処理]                           [出力]

オプション定義       ───▶ setDefined/setRequired等    ───▶ 内部状態に蓄積
(FormType等)

実際の値配列         ───▶ resolve()                    ───▶ 解決済み配列
                           │
                           ├─ 未定義/必須チェック
                           ├─ デフォルト値適用
                           ├─ 型・許容値チェック
                           └─ 正規化処理
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| OptionsResolver.php | `src/Symfony/Component/OptionsResolver/OptionsResolver.php` | ソース | オプション解決の中核クラス |
| Options.php | `src/Symfony/Component/OptionsResolver/Options.php` | ソース | Optionsインターフェース |
| OptionConfigurator.php | `src/Symfony/Component/OptionsResolver/OptionConfigurator.php` | ソース | Fluent API設定ヘルパー |
| Debug/ | `src/Symfony/Component/OptionsResolver/Debug/` | ソース | デバッグ用ユーティリティ |
| Exception/ | `src/Symfony/Component/OptionsResolver/Exception/` | ソース | 例外クラス群 |
