# 機能設計書 24-PasswordHasher

## 概要

本ドキュメントは、Symfony PasswordHasherコンポーネントの機能設計を記述する。このコンポーネントはパスワードハッシュ化ユーティリティを提供し、bcrypt、argon2等のアルゴリズムに対応する。

### 本機能の処理概要

**業務上の目的・背景**：パスワードの安全な保存は認証システムの基本要件である。パスワードを平文で保存することは重大なセキュリティリスクであり、一方向ハッシュ関数を使ってハッシュ化する必要がある。本コンポーネントは、複数のハッシュアルゴリズムを統一的なインターフェースで提供し、アルゴリズムの移行（マイグレーション）も安全に行える仕組みを提供する。

**機能の利用シーン**：ユーザー登録時のパスワードハッシュ化、ログイン時のパスワード検証、パスワード変更時の再ハッシュ化、ハッシュアルゴリズムの移行（例：bcryptからargon2idへ）、CLIコマンドによるパスワードハッシュ化。

**主要な処理内容**：
1. PasswordHasherInterfaceによるパスワードハッシュ化（hash）、検証（verify）、再ハッシュ必要性判定（needsRehash）
2. PasswordHasherFactoryによるユーザークラスに基づくハッシャーの自動選択
3. UserPasswordHasherによるユーザーオブジェクトと連携したハッシュ化・検証
4. MigratingPasswordHasherによる複数アルゴリズム間の自動移行
5. NativePasswordHasher（PHPネイティブ）、SodiumPasswordHasher（libsodium）、Pbkdf2PasswordHasher等の具体的実装
6. 'auto'アルゴリズムによる最適なハッシャーの自動選択

**関連システム・外部連携**：Security Core（PasswordAuthenticatedUserInterface）、Security HTTP（PasswordMigratingListener）、SecurityBundle（設定統合）と連携する。

**権限による制御**：パスワードハッシュ化自体には権限制御はない。パスワード変更操作に対する権限制御はアプリケーション層で行う。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | （直接的な画面関連なし） | - | バックエンドでのパスワードハッシュ化・検証処理 |

## 機能種別

セキュリティ / パスワードハッシュ化

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| plainPassword | string | Yes | ハッシュ化対象の平文パスワード | 最大4096文字（MAX_PASSWORD_LENGTH） |
| hashedPassword | string | Yes（検証時） | 検証対象のハッシュ値 | 有効なハッシュ文字列 |
| user | PasswordAuthenticatedUserInterface | Yes（UserPasswordHasher使用時） | ユーザーオブジェクト | インターフェース実装 |
| algorithm | string | No | ハッシュアルゴリズム名 | auto/native/sodium/bcrypt/argon2i/argon2id/pbkdf2/plaintext |

### 入力データソース

ログインフォーム、ユーザー登録フォーム、パスワード変更フォーム、CLIコマンド

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| hashedPassword | string | ハッシュ化されたパスワード文字列 |
| isValid | bool | パスワード検証結果 |
| needsRehash | bool | 再ハッシュが必要かどうか |

### 出力先

データベース（ユーザーテーブルのパスワードカラム）、認証結果（bool）

## 処理フロー

### 処理シーケンス

```
1. ハッシャー取得
   └─ PasswordHasherFactory::getPasswordHasher()でユーザークラスに基づくハッシャー選択
2. パスワードハッシュ化
   └─ hash()でアルゴリズム固有のハッシュ処理を実行
3. パスワード検証
   └─ verify()でハッシュ値と平文パスワードを比較
4. 再ハッシュ判定
   └─ needsRehash()でアルゴリズムやコストが変更されたか確認
5. マイグレーション（MigratingPasswordHasher使用時）
   └─ 複数ハッシャーで検証を試み、最初のハッシャーで再ハッシュ
```

### フローチャート

```mermaid
flowchart TD
    A[パスワードハッシュ化要求] --> B[PasswordHasherFactory::getPasswordHasher]
    B --> C{ユーザークラスに一致するハッシャー?}
    C -->|Yes| D[ハッシャー取得]
    C -->|No| E[RuntimeException]
    D --> F{algorithm = auto?}
    F -->|Yes| G[MigratingPasswordHasher構築]
    F -->|No| H[指定アルゴリズムのハッシャー構築]
    G --> I[hash/verify/needsRehash実行]
    H --> I

    J[パスワード検証] --> K[MigratingPasswordHasher::verify]
    K --> L[メインハッシャーで検証]
    L --> M{成功?}
    M -->|Yes| N[true返却]
    M -->|No| O[フォールバックハッシャーで検証]
    O --> P{成功?}
    P -->|Yes| N
    P -->|No| Q[false返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-24-01 | パスワード長制限 | 平文パスワードは最大4096文字まで | hash()呼び出し時 |
| BR-24-02 | auto選択 | algorithm='auto'の場合、native/sodium/pbkdf2のMigratingHasherを構築 | ハッシャー構成時 |
| BR-24-03 | マイグレーション | MigratingPasswordHasherはメインハッシャーで検証失敗時にフォールバックで試行 | verify()呼び出し時 |
| BR-24-04 | ユーザークラスマッチング | PasswordHasherFactoryはユーザーオブジェクトのクラス階層を辿ってハッシャーを検索 | getPasswordHasher()呼び出し時 |
| BR-24-05 | レガシーsalt対応 | LegacyPasswordAuthenticatedUserInterfaceを実装したユーザーからsaltを取得 | レガシーユーザー対応時 |

### 計算ロジック

各アルゴリズムの計算はPHPネイティブ関数に委任される。NativePasswordHasher: password_hash() / password_verify()、SodiumPasswordHasher: sodium_crypto_pwhash_str() / sodium_crypto_pwhash_str_verify()。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| パスワード保存 | usersテーブル等 | UPDATE | ハッシュ化されたパスワードの保存（アプリケーション層で実行） |

### テーブル別操作詳細

コンポーネント自体はデータベース操作を行わない。アプリケーション層で実装する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | InvalidPasswordException | パスワードが最大長を超過 | 4096文字以下のパスワードを使用 |
| - | RuntimeException | ユーザークラスに対するハッシャーが未設定 | security.yaml等でパスワードハッシャーを設定 |
| - | LogicException | argon2iが利用不可 | sodium拡張またはPHP argon2i対応が必要 |

### リトライ仕様

リトライは不要。

## トランザクション仕様

直接的なトランザクション管理は行わない。

## パフォーマンス要件

パスワードハッシュ化は意図的に計算コストが高い処理であり、time_cost・memory_cost等のパラメータで調整可能。bcryptのデフォルトコストは13、argon2のtime_costとmemory_costはPHP/libsodiumのデフォルト値に従う。

## セキュリティ考慮事項

- #[SensitiveParameter]アトリビュートによる平文パスワードのスタックトレースからの除外
- MAX_PASSWORD_LENGTH (4096)によるDoS攻撃（長大パスワードによるCPU消費）の防止
- MigratingPasswordHasherによる安全なアルゴリズム移行
- password_verify()による内部的なタイミングセーフ比較
- PlaintextPasswordHasherは開発・テスト環境専用

## 備考

'auto'アルゴリズムを使用することで、環境に最適なハッシュアルゴリズムが自動選択される。本番環境ではsodium（argon2id）またはnative（bcrypt）が推奨される。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | PasswordHasherInterface.php | `src/Symfony/Component/PasswordHasher/PasswordHasherInterface.php` | hash, verify, needsRehashの3メソッド。MAX_PASSWORD_LENGTH=4096定数 |
| 1-2 | PasswordHasherFactoryInterface.php | `src/Symfony/Component/PasswordHasher/Hasher/PasswordHasherFactoryInterface.php` | ファクトリインターフェース |
| 1-3 | UserPasswordHasherInterface.php | `src/Symfony/Component/PasswordHasher/Hasher/UserPasswordHasherInterface.php` | ユーザーレベルのハッシャーインターフェース |

**読解のコツ**: PasswordHasherInterfaceは3メソッドのシンプルなインターフェース。#[SensitiveParameter]アトリビュートがplainPasswordパラメータに付与されている点に注意。

#### Step 2: ファクトリを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | PasswordHasherFactory.php | `src/Symfony/Component/PasswordHasher/Hasher/PasswordHasherFactory.php` | ハッシャーファクトリの中核実装 |

**主要処理フロー**:
1. **34-62行目**: getPasswordHasher()でユーザークラスに基づくハッシャー検索。PasswordHasherAwareInterfaceまたはクラス階層マッチング
2. **69-107行目**: createHasher()で設定配列からハッシャーインスタンスを構築
3. **109-215行目**: getHasherConfigFromAlgorithm()でアルゴリズム名から設定を解決。'auto'はMigratingPasswordHasherを構築
4. **217-237行目**: getMigratingPasswordConfig()でmigrate_fromオプションの処理

#### Step 3: ユーザーレベルハッシャーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | UserPasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/UserPasswordHasher.php` | ユーザーオブジェクトと連携するハッシャー |

**主要処理フロー**:
- **31-41行目**: hashPassword()でLegacyPasswordAuthenticatedUserInterfaceのsalt対応後、ファクトリからハッシャーを取得してhash
- **43-57行目**: isPasswordValid()でユーザーのパスワードがnullでないことを確認後、verify
- **59-68行目**: needsRehash()でユーザーのパスワードの再ハッシュ必要性判定

#### Step 4: 個別ハッシャーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | NativePasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/NativePasswordHasher.php` | PHPネイティブのpassword_hash/verify使用 |
| 4-2 | SodiumPasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/SodiumPasswordHasher.php` | libsodiumのargon2使用 |
| 4-3 | MigratingPasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/MigratingPasswordHasher.php` | 複数ハッシャーのマイグレーション対応 |

### プログラム呼び出し階層図

```
UserPasswordHasher::hashPassword()
    |
    +-- LegacyPasswordAuthenticatedUserInterface::getSalt()
    +-- PasswordHasherFactory::getPasswordHasher()
    |       |
    |       +-- PasswordHasherAwareInterface::getPasswordHasherName()
    |       +-- createHasher()
    |              +-- getHasherConfigFromAlgorithm()
    |              +-- new MigratingPasswordHasher(...)
    |
    +-- PasswordHasherInterface::hash()
           |
           +-- NativePasswordHasher::hash()     [password_hash()]
           +-- SodiumPasswordHasher::hash()     [sodium_crypto_pwhash_str()]
           +-- Pbkdf2PasswordHasher::hash()     [hash_pbkdf2()]
```

### データフロー図

```
[入力]                    [処理]                          [出力]

plainPassword ---------> UserPasswordHasher -----------> hashedPassword (string)
UserInterface ---------> PasswordHasherFactory
                              |
                         getPasswordHasher()
                              |
                         NativePasswordHasher
                         /SodiumPasswordHasher
                         /MigratingPasswordHasher
                              |
                         hash() / verify() / needsRehash()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| PasswordHasherInterface.php | `src/Symfony/Component/PasswordHasher/PasswordHasherInterface.php` | ソース | ハッシャーインターフェース |
| PasswordHasherFactory.php | `src/Symfony/Component/PasswordHasher/Hasher/PasswordHasherFactory.php` | ソース | ハッシャーファクトリ |
| UserPasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/UserPasswordHasher.php` | ソース | ユーザーレベルハッシャー |
| NativePasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/NativePasswordHasher.php` | ソース | PHPネイティブハッシャー |
| SodiumPasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/SodiumPasswordHasher.php` | ソース | Sodiumハッシャー |
| MigratingPasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/MigratingPasswordHasher.php` | ソース | マイグレーション対応ハッシャー |
| Pbkdf2PasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/Pbkdf2PasswordHasher.php` | ソース | PBKDF2ハッシャー |
| MessageDigestPasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/MessageDigestPasswordHasher.php` | ソース | メッセージダイジェストハッシャー |
| PlaintextPasswordHasher.php | `src/Symfony/Component/PasswordHasher/Hasher/PlaintextPasswordHasher.php` | ソース | 平文ハッシャー（テスト用） |
| CheckPasswordLengthTrait.php | `src/Symfony/Component/PasswordHasher/Hasher/CheckPasswordLengthTrait.php` | ソース | パスワード長検証トレイト |
