# 通知設計書 11-FailedMessageEvent（Notifier）

## 概要

本ドキュメントは、Symfony Notifierコンポーネントにおける `FailedMessageEvent` イベントの設計仕様を定義する。通知メッセージの送信が失敗した際にディスパッチされるイベントであり、エラーハンドリングやログ記録、リトライ処理の起点となる。

### 本通知の処理概要

FailedMessageEvent（Notifier）は、Notifierコンポーネントのトランスポート層で通知メッセージの送信が失敗した際に自動的にディスパッチされるイベントオブジェクトである。SMS、チャット、プッシュ通知など、Notifierが扱う全チャネルの送信失敗を一元的に捕捉するための仕組みを提供する。

**業務上の目的・背景**：通知送信はネットワーク障害やサービスプロバイダの障害等により失敗する可能性がある。FailedMessageEventにより、送信失敗をアプリケーション側で検知し、管理者への通知、ログ記録、代替チャネルへのフォールバック等のエラーハンドリングを実装することが可能になる。これにより、通知送信の信頼性と可観測性が向上する。

**通知の送信タイミング**：Notifierコンポーネントの `AbstractTransport::send()` メソッド内で `doSend()` が例外をスローした際に、即座にディスパッチされる。具体的には、SMS送信、チャットメッセージ送信、プッシュ通知送信等の実行時にトランスポート層で例外が発生した場合である。

**通知の受信者**：EventDispatcherに登録されたイベントリスナーおよびイベントサブスクライバーが受信者となる。開発者がアプリケーション内で `FailedMessageEvent::class` に対するリスナーを登録することで、任意のエラーハンドリングロジックを実装できる。

**通知内容の概要**：送信に失敗したメッセージオブジェクト（MessageInterface実装）と、失敗の原因となった例外オブジェクト（Throwable）の2つの情報を保持する。

**期待されるアクション**：リスナー側で失敗内容をログに記録する、管理者に障害を通知する、リトライキューに再投入する、代替チャネルで再送信する等の対応を行う。

## 通知種別

イベント通知（Symfonyイベントディスパッチャーベース）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（EventDispatcherによるディスパッチ） |
| 優先度 | 高（送信失敗の直後にディスパッチ） |
| リトライ | イベント自体のリトライはなし（リスナー側で実装） |

### 送信先決定ロジック

EventDispatcherに登録された全ての `FailedMessageEvent::class` リスナーに対してイベントが配信される。送信先の決定はSymfonyのEventDispatcherコンポーネントの仕組みに準拠し、優先度順にリスナーが呼び出される。

## 通知テンプレート

### メール通知の場合

本イベントはメール通知ではなく、イベントオブジェクトとしてディスパッチされるため、メールテンプレートは適用されない。

### 本文テンプレート

```
該当なし（イベントオブジェクトとしてデータを保持するのみ）
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| 該当なし | - | - | イベントオブジェクトのため添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| message | 送信に失敗したメッセージ | MessageInterface実装オブジェクト | Yes |
| error | 失敗原因の例外 | \Throwable | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| トランスポート送信失敗 | AbstractTransport::doSend()が例外をスロー | EventDispatcherが設定されていること | send()メソッド内のtry-catchブロックで捕捉された例外発生時 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| EventDispatcherが未設定 | AbstractTransportのコンストラクタでdispatcherがnullの場合、イベントはディスパッチされない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[Transport::send呼び出し] --> B[MessageEventディスパッチ]
    B --> C[doSend実行]
    C -->|例外発生| D[FailedMessageEventディスパッチ]
    C -->|成功| E[SentMessageEventディスパッチ]
    D --> F[例外を再スロー]
    F --> G[呼び出し元へ例外伝播]
    E --> H[SentMessage返却]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| 該当なし | - | Notifierコンポーネント自体はDB参照を行わない |

### テーブル別参照項目詳細

該当なし。FailedMessageEventはイベントオブジェクトであり、データベースアクセスは行わない。

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 該当なし | - | イベントリスナー側で必要に応じてDB更新を実装 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| トランスポート送信失敗 | doSend()内で\Throwableが発生 | FailedMessageEventをディスパッチ後、例外を再スロー |
| リスナー内例外 | イベントリスナー内で例外が発生 | EventDispatcherの標準動作に従い伝播 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | イベント自体にリトライ機構はなし |
| リトライ間隔 | - |
| リトライ対象エラー | リスナー側で実装が必要 |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（イベントディスパッチのため制限なし） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし。送信失敗が発生した時点で即座にディスパッチされる。

## セキュリティ考慮事項

- FailedMessageEventにはメッセージオブジェクトと例外オブジェクトが含まれるため、リスナー内でログ出力する際は個人情報（電話番号、メールアドレス等）のマスキングを考慮すること
- 例外のスタックトレースにはシステム内部のパス情報等が含まれる可能性があるため、外部への露出に注意すること

## 備考

- FailedMessageEvent（Notifier）は `Symfony\Component\Notifier\Event\FailedMessageEvent` に定義されている
- 同名の `FailedMessageEvent` がMailerコンポーネントにも存在するが（No.14参照）、名前空間が異なり別のクラスである
- Symfony 6.3で追加されたイベントである

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

FailedMessageEventが保持するデータの構造を理解することが出発点となる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | MessageInterface.php | `src/Symfony/Component/Notifier/Message/MessageInterface.php` | 送信失敗したメッセージの共通インターフェース。getRecipientId(), getSubject(), getOptions(), getTransport()の4メソッドを定義（行17-26） |
| 1-2 | FailedMessageEvent.php | `src/Symfony/Component/Notifier/Event/FailedMessageEvent.php` | イベント本体。MessageInterfaceと\Throwableの2プロパティをコンストラクタで受け取り、それぞれのgetterを提供（行20-37） |

**読解のコツ**: FailedMessageEventはfinalクラスであり、継承による拡張は不可。Eventクラス（`Symfony\Contracts\EventDispatcher\Event`）を継承しており、stopPropagation()等の基本機能を持つ。

#### Step 2: エントリーポイントを理解する

FailedMessageEventがどこでディスパッチされるかを特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AbstractTransport.php | `src/Symfony/Component/Notifier/Transport/AbstractTransport.php` | send()メソッド内でdoSend()を呼び出し、例外発生時にFailedMessageEventをディスパッチする（行67-86） |

**主要処理フロー**:
1. **行73**: `MessageEvent` をディスパッチ（送信前イベント）
2. **行76**: `doSend($message)` を実行（実際の送信処理）
3. **行77-78**: 例外キャッチ時に `new FailedMessageEvent($message, $error)` を生成してディスパッチ
4. **行80**: 例外を再スロー

#### Step 3: イベントリスナーの実例を理解する

NotificationLoggerListenerがMessageEventを購読する例を見ることで、同様にFailedMessageEventにリスナーを登録するパターンを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | NotificationLoggerListener.php | `src/Symfony/Component/Notifier/EventListener/NotificationLoggerListener.php` | EventSubscriberInterfaceの実装例。getSubscribedEvents()で購読イベントと優先度を定義（行46-51） |

**主要処理フロー**:
- **行48-49**: `MessageEvent::class => ['onNotification', -255]` で最低優先度で購読

### プログラム呼び出し階層図

```
AbstractTransport::send($message)
    |
    +-- EventDispatcher::dispatch(new MessageEvent($message))
    |
    +-- AbstractTransport::doSend($message)  [抽象メソッド - 各トランスポートが実装]
    |       |
    |       +-- [例外発生時]
    |               |
    |               +-- EventDispatcher::dispatch(new FailedMessageEvent($message, $error))
    |               |
    |               +-- throw $error
    |
    +-- [成功時] EventDispatcher::dispatch(new SentMessageEvent($sentMessage))
```

### データフロー図

```
[入力]                     [処理]                           [出力]

MessageInterface -----> AbstractTransport::send()
                            |
                            +---> doSend() で例外発生
                            |
\Throwable ----------> FailedMessageEvent生成
                            |
                            +---> EventDispatcher::dispatch()
                            |         |
                            |         +---> リスナー1: ログ記録
                            |         +---> リスナー2: 管理者通知
                            |         +---> リスナーN: カスタム処理
                            |
                            +---> 例外再スロー --> 呼び出し元
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| FailedMessageEvent.php | `src/Symfony/Component/Notifier/Event/FailedMessageEvent.php` | ソース | イベントクラス本体 |
| AbstractTransport.php | `src/Symfony/Component/Notifier/Transport/AbstractTransport.php` | ソース | イベントのディスパッチ元 |
| MessageInterface.php | `src/Symfony/Component/Notifier/Message/MessageInterface.php` | ソース | メッセージの共通インターフェース |
| MessageEvent.php | `src/Symfony/Component/Notifier/Event/MessageEvent.php` | ソース | 送信前イベント（関連イベント） |
| SentMessageEvent.php | `src/Symfony/Component/Notifier/Event/SentMessageEvent.php` | ソース | 送信成功イベント（関連イベント） |
| NotificationLoggerListener.php | `src/Symfony/Component/Notifier/EventListener/NotificationLoggerListener.php` | ソース | イベントリスナーの実装例 |
