# 通知設計書 12-MessageEvent（Mailer）

## 概要

本ドキュメントは、Symfony Mailerコンポーネントにおける `MessageEvent` イベントの設計仕様を定義する。メール送信前にディスパッチされるイベントであり、送信するメッセージやエンベロープの変換・加工、送信の拒否（reject）機能を提供する。

### 本通知の処理概要

MessageEvent（Mailer）は、Mailerコンポーネントのトランスポート層でメール送信が実行される直前にディスパッチされるイベントオブジェクトである。イベントリスナーを通じて、送信前のメッセージ内容やエンベロープの変更、Messengerスタンプの追加、さらには送信そのものの拒否が可能である。

**業務上の目的・背景**：メール送信処理において、送信前にメッセージの内容を動的に変更したい場面（例：全メールにフッターを追加、特定条件でCC追加、テスト環境での送信先書き換え等）が存在する。MessageEventはこのようなクロスカッティングな関心事をイベントリスナーとして分離し、送信ロジック本体を変更せずに柔軟なカスタマイズを可能にする。また、reject()メソッドにより特定条件下でメール送信を完全に抑止することもできる。

**通知の送信タイミング**：`AbstractTransport::send()` メソッド内で、実際のメール送信処理（`doSend()`）が実行される前にディスパッチされる。EventDispatcherが設定されている場合に限り発火する。

**通知の受信者**：EventDispatcherに登録された `MessageEvent::class` のイベントリスナーおよびイベントサブスクライバーが受信者となる。

**通知内容の概要**：送信予定のRawMessageオブジェクト、Envelopeオブジェクト、トランスポート名、キューイング状態（queued）の情報を保持する。リスナーはこれらの値を読み取り、messageとenvelopeについてはsetterで変更も可能。

**期待されるアクション**：リスナー側でメッセージ内容の加工（ヘッダー追加、本文変更等）、エンベロープの変更（送信先変更等）、送信拒否（reject()）、Messengerスタンプの追加（キューイング時）等を行う。

## 通知種別

イベント通知（Symfonyイベントディスパッチャーベース）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（EventDispatcherによるディスパッチ） |
| 優先度 | 高（メール送信前の必須フック） |
| リトライ | なし（イベントディスパッチ自体のリトライはなし） |

### 送信先決定ロジック

EventDispatcherに登録された全ての `Symfony\Component\Mailer\Event\MessageEvent::class` リスナーに対してイベントが配信される。リスナーの呼び出し順序はEventDispatcherの優先度設定に従う。reject()が呼ばれた場合、stopPropagation()も同時に実行されるため、後続のリスナーは呼び出されない。

## 通知テンプレート

### メール通知の場合

本イベントはメール通知そのものではなく、メール送信前に発火するイベントオブジェクトであるため、メールテンプレートは適用されない。

### 本文テンプレート

```
該当なし（イベントオブジェクトとしてデータを保持・変更するのみ）
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| 該当なし | - | - | イベントオブジェクトのため添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| message | 送信予定のメッセージ | RawMessage（getMessage()/setMessage()） | Yes |
| envelope | 送信エンベロープ | Envelope（getEnvelope()/setEnvelope()） | Yes |
| transport | トランスポート名 | string（getTransport()） | Yes |
| queued | キューイング状態 | bool（isQueued()） | Yes |
| rejected | 送信拒否フラグ | bool（isRejected()） | No（デフォルトfalse） |
| stamps | Messengerスタンプ配列 | StampInterface[]（getStamps()） | No（queued時のみ） |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| メール送信処理 | AbstractTransport::send()呼び出し | EventDispatcherが設定されていること | doSend()実行前にディスパッチ |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| EventDispatcherが未設定 | AbstractTransportのコンストラクタでdispatcherがnullの場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[AbstractTransport::send呼び出し] --> B[メッセージとエンベロープのクローン]
    B --> C{EventDispatcherあり?}
    C -->|なし| D[SentMessage生成 → doSend → 返却]
    C -->|あり| E[MessageEventディスパッチ]
    E --> F{rejected?}
    F -->|Yes| G[null返却]
    F -->|No| H[エンベロープ・メッセージ取得]
    H --> I[TemplatedEmailレンダリングチェック]
    I --> J[SentMessage生成]
    J --> K[doSend実行]
    K -->|成功| L[SentMessageEventディスパッチ]
    K -->|失敗| M[FailedMessageEventディスパッチ → 例外再スロー]
    L --> N[SentMessage返却]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| 該当なし | - | Mailerコンポーネント自体はDB参照を行わない |

### テーブル別参照項目詳細

該当なし。MessageEventはイベントオブジェクトであり、データベースアクセスは行わない。

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 該当なし | - | イベントリスナー側で必要に応じてDB更新を実装 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| LogicException | キューイングされていないメッセージに対してaddStamp()/getStamps()を呼び出した場合 | queued=trueの場合のみスタンプ操作を行うよう実装 |
| LogicException | TemplatedEmailが未レンダリングの状態でdoSend()に到達した場合 | BodyRendererInterfaceを設定してテンプレートをレンダリングする |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | イベント自体にリトライ機構はなし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（イベントディスパッチのため制限なし） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし。メール送信処理が実行されるたびにディスパッチされる。

## セキュリティ考慮事項

- MessageEventのリスナーはメッセージ内容とエンベロープを変更可能なため、不正なリスナーが登録されると送信先の改ざんやメッセージの改変が可能になる。信頼できるリスナーのみを登録すること
- reject()によるメール送信抑止は、stopPropagation()も同時に呼び出すため、後続リスナーの実行も停止される点に注意
- キューイング時のスタンプ追加は、Messengerコンポーネントとの連携で使用されるため、スタンプの内容にセンシティブ情報を含めないこと

## 備考

- MessageEvent（Mailer）は `Symfony\Component\Mailer\Event\MessageEvent` に定義されている
- 同名の `MessageEvent` がNotifierコンポーネントにも存在するが（No.9参照）、名前空間が異なり別のクラスである
- reject()呼び出し時にsend()メソッドはnullを返す（行76-78）
- queuedフラグはコンストラクタで設定され、Messengerを通じて非同期送信される場合にtrueとなる

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

MessageEventが保持・操作するデータの構造を理解することが出発点となる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | MessageEvent.php | `src/Symfony/Component/Mailer/Event/MessageEvent.php` | イベント本体。4つのコンストラクタ引数（message, envelope, transport, queued）と、rejected/stampsの内部状態を管理（行25-101） |
| 1-2 | Envelope.php | `src/Symfony/Component/Mailer/Envelope.php` | メール送信のエンベロープ（送信元・送信先）情報 |
| 1-3 | RawMessage.php | `src/Symfony/Component/Mime/RawMessage.php` | メールメッセージの基底クラス |

**読解のコツ**: MessageEventはfinalクラスであり、メッセージとエンベロープに対してはgetter/setterの両方を提供する点が特徴的。reject()はstopPropagation()も呼ぶ点に注目。

#### Step 2: エントリーポイントを理解する

MessageEventがどこで生成・ディスパッチされるかを特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AbstractTransport.php | `src/Symfony/Component/Mailer/Transport/AbstractTransport.php` | send()メソッド全体がエントリーポイント。行61-104でMessageEventの生成、ディスパッチ、rejected判定、メッセージ取得の一連の流れを実装 |

**主要処理フロー**:
1. **行63-64**: メッセージとエンベロープのクローン
2. **行74**: `new MessageEvent($message, $envelope, (string) $this)` でイベント生成
3. **行75**: `$this->dispatcher->dispatch($event)` でディスパッチ
4. **行76-78**: `$event->isRejected()` で拒否判定、trueならnull返却
5. **行80-81**: リスナーによる変更後のenvelope/messageを取得
6. **行83-85**: TemplatedEmailの未レンダリングチェック
7. **行90-96**: doSend()実行、失敗時にFailedMessageEventディスパッチ
8. **行98**: 成功時にSentMessageEventディスパッチ

#### Step 3: reject機能を理解する

MessageEvent固有のreject機能の実装を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | MessageEvent.php | `src/Symfony/Component/Mailer/Event/MessageEvent.php` | reject()メソッド（行75-79）がrejectedフラグ設定とstopPropagation()を同時実行 |

**主要処理フロー**:
- **行75-79**: reject()がrejected=trueに設定し、stopPropagation()で後続リスナーを停止

#### Step 4: スタンプ機能を理解する

Messengerとの連携で使用されるスタンプ追加機能を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | MessageEvent.php | `src/Symfony/Component/Mailer/Event/MessageEvent.php` | addStamp()/getStamps()（行81-100）はqueued=trueの場合のみ利用可能。falseの場合LogicException |

**主要処理フロー**:
- **行81-88**: addStamp()はqueuedチェック後にスタンプ配列に追加
- **行93-100**: getStamps()はqueuedチェック後にスタンプ配列を返却

### プログラム呼び出し階層図

```
AbstractTransport::send($message, $envelope)
    |
    +-- clone $message / clone $envelope / Envelope::create()
    |
    +-- [dispatcher あり]
    |       |
    |       +-- new MessageEvent($message, $envelope, $transport)
    |       |
    |       +-- EventDispatcher::dispatch($event)
    |       |       |
    |       |       +-- リスナー: setMessage() / setEnvelope() / reject() / addStamp()
    |       |
    |       +-- [rejected] --> return null
    |       |
    |       +-- $event->getEnvelope() / $event->getMessage()
    |       |
    |       +-- TemplatedEmailレンダリングチェック
    |       |
    |       +-- new SentMessage($message, $envelope)
    |       |
    |       +-- doSend($sentMessage)
    |               |
    |               +-- [成功] --> dispatch(SentMessageEvent)
    |               +-- [失敗] --> dispatch(FailedMessageEvent) --> throw
    |
    +-- [dispatcher なし]
            |
            +-- new SentMessage($message, $envelope)
            +-- doSend($sentMessage)
```

### データフロー図

```
[入力]                        [処理]                              [出力]

RawMessage --------+
                   |
Envelope ----------+----> MessageEvent生成
                   |          |
Transport名 -------+          +----> EventDispatcher::dispatch()
                              |          |
                              |          +----> リスナー: message変更
                              |          +----> リスナー: envelope変更
                              |          +----> リスナー: reject()
                              |          +----> リスナー: addStamp()
                              |
                              +----> [rejected] --> null
                              |
                              +----> [変更後message/envelope] --> SentMessage --> doSend()
                                                                     |
                                                                     +----> SentMessageEvent / FailedMessageEvent
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| MessageEvent.php | `src/Symfony/Component/Mailer/Event/MessageEvent.php` | ソース | イベントクラス本体 |
| AbstractTransport.php | `src/Symfony/Component/Mailer/Transport/AbstractTransport.php` | ソース | イベントのディスパッチ元 |
| Envelope.php | `src/Symfony/Component/Mailer/Envelope.php` | ソース | メール送信エンベロープ |
| RawMessage.php | `src/Symfony/Component/Mime/RawMessage.php` | ソース | メールメッセージ基底クラス |
| SentMessage.php | `src/Symfony/Component/Mailer/SentMessage.php` | ソース | 送信済みメッセージ |
| SentMessageEvent.php | `src/Symfony/Component/Mailer/Event/SentMessageEvent.php` | ソース | 送信成功イベント |
| FailedMessageEvent.php | `src/Symfony/Component/Mailer/Event/FailedMessageEvent.php` | ソース | 送信失敗イベント |
| TemplatedEmail.php | `src/Symfony/Bridge/Twig/Mime/TemplatedEmail.php` | ソース | テンプレートメール（レンダリングチェック対象） |
