# 通知設計書 13-SentMessageEvent（Mailer）

## 概要

本ドキュメントは、Symfony Mailerコンポーネントにおける `SentMessageEvent` イベントの設計仕様を定義する。メール送信が正常に完了した後にディスパッチされるイベントであり、送信済みメッセージの情報を保持する。

### 本通知の処理概要

SentMessageEvent（Mailer）は、Mailerコンポーネントのトランスポート層でメール送信が正常に完了した直後にディスパッチされるイベントオブジェクトである。送信済みメッセージ（SentMessage）の情報を保持しており、送信完了後の後処理をイベントリスナーとして実装することを可能にする。

**業務上の目的・背景**：メール送信完了を検知して後続処理を行いたい場面（例：送信ログの記録、送信統計の集計、送信完了通知の発行、ステータス更新等）が存在する。SentMessageEventにより、これらの後処理をメール送信ロジック本体から分離し、イベント駆動で実装できる。これによりメール送信の可観測性（Observability）が向上する。

**通知の送信タイミング**：`AbstractTransport::send()` メソッド内で `doSend()` が正常に完了した直後にディスパッチされる。EventDispatcherが設定されている場合に限り発火する。

**通知の受信者**：EventDispatcherに登録された `SentMessageEvent::class` のイベントリスナーおよびイベントサブスクライバーが受信者となる。

**通知内容の概要**：送信済みのSentMessageオブジェクトを保持する。SentMessageにはオリジナルメッセージ、変換後のRawMessage、エンベロープ、メッセージID、デバッグ情報が含まれる。

**期待されるアクション**：リスナー側で送信ログを記録する、送信成功をダッシュボードに反映する、メッセージIDを用いた送信追跡を開始する、後続のビジネスロジックをトリガーする等の対応を行う。

## 通知種別

イベント通知（Symfonyイベントディスパッチャーベース）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（EventDispatcherによるディスパッチ） |
| 優先度 | 中（送信後の後処理フック） |
| リトライ | なし（イベントディスパッチ自体のリトライはなし） |

### 送信先決定ロジック

EventDispatcherに登録された全ての `Symfony\Component\Mailer\Event\SentMessageEvent::class` リスナーに対してイベントが配信される。リスナーの呼び出し順序はEventDispatcherの優先度設定に従う。

## 通知テンプレート

### メール通知の場合

本イベントはメール通知ではなく、メール送信完了後に発火するイベントオブジェクトであるため、メールテンプレートは適用されない。

### 本文テンプレート

```
該当なし（イベントオブジェクトとしてデータを保持するのみ）
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| 該当なし | - | - | イベントオブジェクトのため添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| message | 送信済みメッセージ | SentMessage（getMessage()） | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| メール送信成功 | AbstractTransport::doSend()が正常完了 | EventDispatcherが設定されていること | doSend()成功後にディスパッチ |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| EventDispatcherが未設定 | AbstractTransportのコンストラクタでdispatcherがnullの場合 |
| doSend()が例外をスロー | 送信失敗時はSentMessageEventではなくFailedMessageEventがディスパッチされる |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[AbstractTransport::send呼び出し] --> B[MessageEventディスパッチ]
    B --> C[doSend実行]
    C -->|成功| D[SentMessageEventディスパッチ]
    C -->|失敗| E[FailedMessageEventディスパッチ]
    D --> F[SentMessage返却]
    E --> G[例外再スロー]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| 該当なし | - | Mailerコンポーネント自体はDB参照を行わない |

### テーブル別参照項目詳細

該当なし。SentMessageEventはイベントオブジェクトであり、データベースアクセスは行わない。

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 該当なし | - | イベントリスナー側で必要に応じてDB更新を実装 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー内例外 | イベントリスナー内で例外が発生 | EventDispatcherの標準動作に従い伝播 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | イベント自体にリトライ機構はなし |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（イベントディスパッチのため制限なし） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし。メール送信成功時に即座にディスパッチされる。

## セキュリティ考慮事項

- SentMessageオブジェクトにはメール本文やヘッダー情報が含まれるため、リスナー内でログ出力する際はセンシティブ情報のマスキングを考慮すること
- SentMessage::getOriginalMessage()は元のメッセージオブジェクトを返すため、個人情報（送信先メールアドレス等）が含まれる可能性がある

## 備考

- SentMessageEvent（Mailer）は `Symfony\Component\Mailer\Event\SentMessageEvent` に定義されている
- 同名の `SentMessageEvent` がNotifierコンポーネントにも存在するが（No.10参照）、名前空間が異なり別のクラスである
- SentMessageオブジェクトはgetMessageId()でMessage-IDヘッダーの値を取得可能
- AbstractTransport::send()内でdoSend()成功後にスロットリングチェック（checkThrottling）がfinallyブロックで実行される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

SentMessageEventが保持するSentMessageの構造を理解することが出発点となる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | SentMessage.php | `src/Symfony/Component/Mailer/SentMessage.php` | 送信済みメッセージ。original（元メッセージ）、raw（変換後）、envelope、messageId、debugの5プロパティを保持（行20-95） |
| 1-2 | SentMessageEvent.php | `src/Symfony/Component/Mailer/Event/SentMessageEvent.php` | イベント本体。SentMessageを1つだけ保持するシンプルなイベント（行20-30） |

**読解のコツ**: SentMessageEventは非常にシンプルで、SentMessageのgetterのみを提供する。データの本体はSentMessageクラスにある。

#### Step 2: エントリーポイントを理解する

SentMessageEventがどこでディスパッチされるかを特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AbstractTransport.php | `src/Symfony/Component/Mailer/Transport/AbstractTransport.php` | send()メソッド内、doSend()成功後の行98でディスパッチ |

**主要処理フロー**:
1. **行87**: `new SentMessage($message, $envelope)` でSentMessage生成
2. **行90**: `$this->doSend($sentMessage)` で実際の送信実行
3. **行98**: `$this->dispatcher->dispatch(new SentMessageEvent($sentMessage))` でイベントディスパッチ
4. **行100**: `return $sentMessage` で送信済みメッセージを返却

#### Step 3: SentMessageの生成過程を理解する

SentMessageのコンストラクタ内での処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SentMessage.php | `src/Symfony/Component/Mailer/SentMessage.php` | コンストラクタ（行30-49）でメッセージのバリデーション、Message-IDヘッダーの自動生成、RawMessageへの変換を実行 |

**主要処理フロー**:
- **行34**: `$message->ensureValidity()` でメッセージの妥当性検証
- **行36**: オリジナルメッセージの保存
- **行38-48**: Message型の場合、Message-IDヘッダーの確認・自動付与、RawMessageへの変換

### プログラム呼び出し階層図

```
AbstractTransport::send($message, $envelope)
    |
    +-- new MessageEvent($message, $envelope, $transport)
    |
    +-- EventDispatcher::dispatch($messageEvent)
    |
    +-- new SentMessage($message, $envelope)
    |       |
    |       +-- $message->ensureValidity()
    |       +-- Message-ID生成（未設定の場合）
    |       +-- RawMessage変換
    |
    +-- doSend($sentMessage)  [実際のメール送信]
    |
    +-- new SentMessageEvent($sentMessage)
    |
    +-- EventDispatcher::dispatch($sentMessageEvent)
    |       |
    |       +-- リスナー1: ログ記録
    |       +-- リスナー2: 統計更新
    |       +-- リスナーN: カスタム後処理
    |
    +-- return $sentMessage
```

### データフロー図

```
[入力]                     [処理]                           [出力]

RawMessage ------+
                 |
Envelope --------+----> SentMessage生成
                            |
                            +----> doSend()で送信実行
                            |
                            +----> SentMessageEvent生成
                            |          |
                            |          +----> EventDispatcher::dispatch()
                            |                     |
                            |                     +----> リスナーへ配信
                            |
SentMessage ----> 呼び出し元へ返却
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| SentMessageEvent.php | `src/Symfony/Component/Mailer/Event/SentMessageEvent.php` | ソース | イベントクラス本体 |
| SentMessage.php | `src/Symfony/Component/Mailer/SentMessage.php` | ソース | 送信済みメッセージデータ |
| AbstractTransport.php | `src/Symfony/Component/Mailer/Transport/AbstractTransport.php` | ソース | イベントのディスパッチ元 |
| MessageEvent.php | `src/Symfony/Component/Mailer/Event/MessageEvent.php` | ソース | 送信前イベント（関連イベント） |
| FailedMessageEvent.php | `src/Symfony/Component/Mailer/Event/FailedMessageEvent.php` | ソース | 送信失敗イベント（関連イベント） |
| Envelope.php | `src/Symfony/Component/Mailer/Envelope.php` | ソース | メール送信エンベロープ |
