# 通知設計書 14-FailedMessageEvent（Mailer）

## 概要

本ドキュメントは、Symfony Mailerコンポーネントにおける `FailedMessageEvent` イベントの設計仕様を定義する。メール送信が失敗した際にディスパッチされるイベントであり、失敗したメッセージとエラー情報を保持する。

### 本通知の処理概要

FailedMessageEvent（Mailer）は、Mailerコンポーネントのトランスポート層でメール送信が失敗した際に自動的にディスパッチされるイベントオブジェクトである。送信に失敗したRawMessageオブジェクトと、失敗原因の例外オブジェクト（Throwable）を保持する。

**業務上の目的・背景**：メール送信はSMTPサーバーの障害、ネットワーク問題、認証エラー等により失敗する可能性がある。FailedMessageEventにより、送信失敗をアプリケーション側で検知し、管理者への障害通知、送信ログへのエラー記録、代替送信方法の試行等のエラーハンドリングを実装できる。これにより、メール送信の信頼性と障害の可視化が向上する。

**通知の送信タイミング**：`AbstractTransport::send()` メソッド内で `doSend()` が例外をスローした際に、例外を再スローする前にディスパッチされる。EventDispatcherが設定されている場合に限り発火する。

**通知の受信者**：EventDispatcherに登録された `FailedMessageEvent::class` のイベントリスナーおよびイベントサブスクライバーが受信者となる。

**通知内容の概要**：送信に失敗したメッセージオブジェクト（RawMessage）と、失敗の原因となった例外オブジェクト（Throwable）の2つの情報を保持する。

**期待されるアクション**：リスナー側で失敗内容をログに記録する、管理者に障害を通知する、メール送信キューに再投入する、代替SMTPサーバーでの再送信を試みる等の対応を行う。

## 通知種別

イベント通知（Symfonyイベントディスパッチャーベース）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（EventDispatcherによるディスパッチ） |
| 優先度 | 高（送信失敗の直後にディスパッチ） |
| リトライ | イベント自体のリトライはなし（リスナー側で実装） |

### 送信先決定ロジック

EventDispatcherに登録された全ての `Symfony\Component\Mailer\Event\FailedMessageEvent::class` リスナーに対してイベントが配信される。送信先の決定はSymfonyのEventDispatcherコンポーネントの仕組みに準拠し、優先度順にリスナーが呼び出される。

## 通知テンプレート

### メール通知の場合

本イベントはメール通知ではなく、メール送信失敗時に発火するイベントオブジェクトであるため、メールテンプレートは適用されない。

### 本文テンプレート

```
該当なし（イベントオブジェクトとしてデータを保持するのみ）
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| 該当なし | - | - | イベントオブジェクトのため添付ファイルなし |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| message | 送信に失敗したメッセージ | RawMessage（getMessage()） | Yes |
| error | 失敗原因の例外 | \Throwable（getError()） | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| メール送信失敗 | AbstractTransport::doSend()が例外をスロー | EventDispatcherが設定されていること | doSend()のcatchブロック内でディスパッチ |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| EventDispatcherが未設定 | AbstractTransportのコンストラクタでdispatcherがnullの場合 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[AbstractTransport::send呼び出し] --> B[MessageEventディスパッチ]
    B --> C[doSend実行]
    C -->|例外発生| D[FailedMessageEventディスパッチ]
    C -->|成功| E[SentMessageEventディスパッチ]
    D --> F[checkThrottling実行]
    F --> G[例外再スロー]
    E --> H[SentMessage返却]
```

## データベース参照・更新仕様

### 参照テーブル一覧

| テーブル名 | 用途 | 備考 |
|-----------|------|------|
| 該当なし | - | Mailerコンポーネント自体はDB参照を行わない |

### テーブル別参照項目詳細

該当なし。FailedMessageEventはイベントオブジェクトであり、データベースアクセスは行わない。

### 更新テーブル一覧

| テーブル名 | 操作 | 概要 |
|-----------|------|------|
| 該当なし | - | イベントリスナー側で必要に応じてDB更新を実装 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| トランスポート送信失敗 | doSend()内で\Throwableが発生 | FailedMessageEventをディスパッチ後、checkThrottling()を実行し例外を再スロー |
| リスナー内例外 | イベントリスナー内で例外が発生 | EventDispatcherの標準動作に従い伝播 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | イベント自体にリトライ機構はなし |
| リトライ間隔 | - |
| リトライ対象エラー | リスナー側で実装が必要 |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（イベントディスパッチのため制限なし） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし。メール送信失敗が発生した時点で即座にディスパッチされる。

## セキュリティ考慮事項

- FailedMessageEventにはRawMessageオブジェクトと例外オブジェクトが含まれるため、リスナー内でログ出力する際はメール本文やヘッダーに含まれる個人情報のマスキングを考慮すること
- 例外のスタックトレースにはシステム内部のパス情報やSMTPサーバーの認証情報等が含まれる可能性があるため、外部への露出に注意すること

## 備考

- FailedMessageEvent（Mailer）は `Symfony\Component\Mailer\Event\FailedMessageEvent` に定義されている
- 同名の `FailedMessageEvent` がNotifierコンポーネントにも存在するが（No.11参照）、名前空間が異なり別のクラスである
- Mailer版はRawMessageを保持するのに対し、Notifier版はMessageInterface（Notifier独自）を保持する点が構造上の違いである
- FailedMessageEventディスパッチ後にcheckThrottling()が実行される（行93）。これにより送信失敗時もレート制限が適用される

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

FailedMessageEventが保持するデータの構造を理解することが出発点となる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | FailedMessageEvent.php | `src/Symfony/Component/Mailer/Event/FailedMessageEvent.php` | イベント本体。RawMessageと\Throwableの2プロパティをコンストラクタで受け取り、それぞれのgetterを提供（行20-37） |
| 1-2 | RawMessage.php | `src/Symfony/Component/Mime/RawMessage.php` | メールメッセージの基底クラス |

**読解のコツ**: FailedMessageEvent（Mailer）はNotifier版と構造は類似しているが、メッセージの型がRawMessage（Mime）である点が異なる。Notifier版はMessageInterface（Notifier独自）を使用する。

#### Step 2: エントリーポイントを理解する

FailedMessageEventがどこでディスパッチされるかを特定する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AbstractTransport.php | `src/Symfony/Component/Mailer/Transport/AbstractTransport.php` | send()メソッド内のtry-catchブロック（行89-96）でdoSend()の例外をキャッチし、FailedMessageEventをディスパッチ |

**主要処理フロー**:
1. **行90**: `$this->doSend($sentMessage)` で実際の送信実行
2. **行91**: `catch (\Throwable $error)` で例外キャッチ
3. **行92**: `$this->dispatcher->dispatch(new FailedMessageEvent($message, $error))` でイベントディスパッチ
4. **行93**: `$this->checkThrottling()` でレート制限チェック
5. **行95**: `throw $error` で例外再スロー

#### Step 3: Notifier版との比較

NotifierコンポーネントのFailedMessageEventとの違いを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FailedMessageEvent.php（Notifier） | `src/Symfony/Component/Notifier/Event/FailedMessageEvent.php` | Notifier版はMessageInterface（Notifier）を使用。Mailer版はRawMessage（Mime）を使用 |
| 3-2 | AbstractTransport.php（Notifier） | `src/Symfony/Component/Notifier/Transport/AbstractTransport.php` | Notifier版のディスパッチ箇所（行77-78）。checkThrottling()がない点がMailer版との違い |

### プログラム呼び出し階層図

```
AbstractTransport::send($message, $envelope)
    |
    +-- EventDispatcher::dispatch(new MessageEvent($message, $envelope, $transport))
    |
    +-- new SentMessage($message, $envelope)
    |
    +-- try: doSend($sentMessage)  [実際のメール送信]
    |       |
    |       +-- [例外発生時]
    |               |
    |               +-- EventDispatcher::dispatch(new FailedMessageEvent($message, $error))
    |               |
    |               +-- checkThrottling()
    |               |
    |               +-- throw $error
    |
    +-- [成功時] EventDispatcher::dispatch(new SentMessageEvent($sentMessage))
    |
    +-- finally: checkThrottling()
```

### データフロー図

```
[入力]                     [処理]                              [出力]

RawMessage --------+
                   |
Envelope ----------+----> AbstractTransport::send()
                            |
                            +----> doSend() で例外発生
                            |
\Throwable --------+----> FailedMessageEvent生成
                            |
                            +----> EventDispatcher::dispatch()
                            |          |
                            |          +----> リスナー1: エラーログ記録
                            |          +----> リスナー2: 管理者通知
                            |          +----> リスナーN: カスタム処理
                            |
                            +----> checkThrottling()
                            |
                            +----> 例外再スロー --> 呼び出し元
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| FailedMessageEvent.php | `src/Symfony/Component/Mailer/Event/FailedMessageEvent.php` | ソース | イベントクラス本体 |
| AbstractTransport.php | `src/Symfony/Component/Mailer/Transport/AbstractTransport.php` | ソース | イベントのディスパッチ元 |
| RawMessage.php | `src/Symfony/Component/Mime/RawMessage.php` | ソース | メールメッセージ基底クラス |
| MessageEvent.php | `src/Symfony/Component/Mailer/Event/MessageEvent.php` | ソース | 送信前イベント（関連イベント） |
| SentMessageEvent.php | `src/Symfony/Component/Mailer/Event/SentMessageEvent.php` | ソース | 送信成功イベント（関連イベント） |
| FailedMessageEvent.php（Notifier） | `src/Symfony/Component/Notifier/Event/FailedMessageEvent.php` | ソース | Notifier版の同名イベント（比較参照） |
