# API設計書

## 概要

本ドキュメントは、TensorFlowプロジェクトが提供する分散実行基盤のgRPC APIを設計・整理したものである。TensorFlowはHTTP REST APIではなく、Protocol Buffers (proto3) で定義されたgRPCサービスを通じて、分散トレーニングやEager実行、データパイプライン、デバッグなどの機能を提供する。本書では、各gRPCサービスのRPCメソッド、リクエスト/レスポンスのメッセージ構造、および用途を記載する。

## 共通仕様

### プロトコル

```
gRPC (HTTP/2ベース)
```

TensorFlowの分散ランタイムは全てgRPCプロトコルで通信する。各サービスはProtocol Buffers (proto3) で定義されている。

### シリアライゼーション形式

Protocol Buffers (proto3)。全てのリクエスト/レスポンスはProtocol Buffersメッセージとしてシリアライズされる。

### 共通パッケージ

| サービス | パッケージ | 定義ファイル |
| --- | --- | --- |
| MasterService | `tensorflow.grpc` | `tensorflow/core/protobuf/master_service.proto` |
| WorkerService | `tensorflow.grpc` | `tensorflow/core/protobuf/worker_service.proto` |
| EagerService | `tensorflow.eager` | `tensorflow/core/protobuf/eager_service.proto` |
| EventListener (Debug) | `tensorflow` | `tensorflow/core/debug/debug_service.proto` |
| RpcService | `tensorflow.rpc` | `tensorflow/distribute/experimental/rpc/proto/tf_rpc_service.proto` |
| DispatcherService (tf.data) | `tensorflow.data` | `tensorflow/core/data/service/dispatcher.proto` |
| WorkerService (tf.data) | `tensorflow.data` | `tensorflow/core/data/service/worker.proto` |

### 共通エラーハンドリング

gRPCの標準ステータスコードが使用される。一部のRPC（RunStep, RunGraph等）では、長いエラーメッセージを扱うために `store_errors_in_response_body` フラグをサポートし、レスポンスボディ内に `status_code` / `status_error_message` を返却する仕組みを備える。

### リトライ検出

多くのRPCメソッドは `request_id` フィールドをサポートする。リトライ時は同一の `request_id` を使用し、ワーカー側でリトライ検出を行う。`request_id` が0の場合、リトライ検出は無効となる。

## API一覧

| カテゴリ | サービス名 | RPCメソッド | 説明 |
| --- | --- | --- | --- |
| Master | MasterService | CreateSession | セッションを作成する |
| Master | MasterService | ExtendSession | セッションのグラフを拡張する |
| Master | MasterService | PartialRunSetup | 部分実行の準備を行う |
| Master | MasterService | RunStep | グラフの計算ステップを実行する |
| Master | MasterService | CloseSession | セッションを閉じる |
| Master | MasterService | ListDevices | 利用可能なデバイスを一覧取得する |
| Master | MasterService | Reset | 全セッションを破棄しリソースを解放する |
| Master | MasterService | MakeCallable | Callableを登録する |
| Master | MasterService | RunCallable | 登録済みCallableを実行する |
| Master | MasterService | ReleaseCallable | Callableのリソースを解放する |
| Worker | WorkerService | GetStatus | ワーカーのデバイス状態を取得する |
| Worker | WorkerService | CreateWorkerSession | ワーカーセッションを作成する |
| Worker | WorkerService | DeleteWorkerSession | ワーカーセッションを削除する |
| Worker | WorkerService | RegisterGraph | サブグラフを登録する |
| Worker | WorkerService | DeregisterGraph | サブグラフの登録を解除する |
| Worker | WorkerService | RunGraph | 登録済みサブグラフを実行する |
| Worker | WorkerService | CleanupGraph | ステップのステートをクリーンアップする |
| Worker | WorkerService | CleanupAll | 全コンテナリソースをクリーンアップする |
| Worker | WorkerService | RecvTensor | テンソルを受信する |
| Worker | WorkerService | MarkRecvFinished | テンソル受信の完了を通知する |
| Worker | WorkerService | Logging | ロギングの制御と取得を行う |
| Worker | WorkerService | Tracing | トレーシングの設定を行う |
| Worker | WorkerService | RecvBuf | Collective Ops用のバッファを受信する |
| Worker | WorkerService | GetStepSequence | ステップシーケンスを取得する |
| Worker | WorkerService | CompleteGroup | Collectiveグループの完了を報告する |
| Worker | WorkerService | CompleteInstance | Collectiveインスタンスの完了を報告する |
| Eager | EagerService | CreateContext | Eagerコンテキストを初期化する |
| Eager | EagerService | UpdateContext | Eagerコンテキストを更新する |
| Eager | EagerService | Enqueue | オペレーションをエンキューする |
| Eager | EagerService | StreamingEnqueue | ストリーミングでオペレーションをエンキューする |
| Eager | EagerService | WaitQueueDone | キュー内オペレーションの完了を待機する |
| Eager | EagerService | RunComponentFunction | コンポーネント関数を実行する |
| Eager | EagerService | KeepAlive | コンテキストの生存を延長する |
| Eager | EagerService | CloseContext | Eagerコンテキストを閉じる |
| Debug | EventListener | SendEvents | デバッグイベントを送信する（ストリーミング） |
| Debug | EventListener | SendTracebacks | トレースバック情報を送信する |
| Debug | EventListener | SendSourceFiles | ソースファイル情報を送信する |
| RPC | RpcService | Call | リモート登録関数を呼び出す |
| RPC | RpcService | List | 利用可能なメソッド一覧を取得する |
| tf.data Dispatcher | DispatcherService | WorkerHeartbeat | ワーカーのハートビートを処理する |
| tf.data Dispatcher | DispatcherService | WorkerUpdate | ワーカーの状態更新を受け取る |
| tf.data Dispatcher | DispatcherService | GetDatasetDef | データセット定義を取得する |
| tf.data Dispatcher | DispatcherService | GetSplit | 次のスプリットを取得する |
| tf.data Dispatcher | DispatcherService | GetVersion | APIバージョンを取得する |
| tf.data Dispatcher | DispatcherService | GetOrRegisterDataset | データセットを登録または取得する |
| tf.data Dispatcher | DispatcherService | GetOrCreateJob | ジョブを作成または取得する |
| tf.data Dispatcher | DispatcherService | GetOrCreateIteration | イテレーションを作成または取得する |
| tf.data Dispatcher | DispatcherService | MaybeRemoveTask | ラウンドロビンタスクの削除を試みる |
| tf.data Dispatcher | DispatcherService | ReleaseIterationClient | イテレーションクライアントを解放する |
| tf.data Dispatcher | DispatcherService | ClientHeartbeat | クライアントのハートビートを処理する |
| tf.data Dispatcher | DispatcherService | GetWorkers | 登録済みワーカー一覧を取得する |
| tf.data Dispatcher | DispatcherService | GetDataServiceMetadata | データサービスメタデータを取得する |
| tf.data Dispatcher | DispatcherService | GetDataServiceConfig | データサービス設定を取得する |
| tf.data Dispatcher | DispatcherService | Snapshot | データセットのスナップショットを開始する |
| tf.data Dispatcher | DispatcherService | GetSnapshotSplit | スナップショットのスプリットを取得する |
| tf.data Dispatcher | DispatcherService | GetSnapshotStreams | スナップショットストリーム情報を取得する |
| tf.data Dispatcher | DispatcherService | DisableCompressionAtRuntime | ランタイム圧縮無効化を制御する |
| tf.data Worker | WorkerService (data) | ProcessTask | データセットタスクを処理する |
| tf.data Worker | WorkerService (data) | GetElement | 次のデータセット要素を取得する |
| tf.data Worker | WorkerService (data) | GetWorkerTasks | 実行中のタスク一覧を取得する |
| tf.data Worker | WorkerService (data) | GetSnapshotTaskProgresses | スナップショットタスクの進捗を取得する |

## 各APIエンドポイント定義

### MasterService

MasterServiceは、クライアントが分散TensorFlow計算を実行するためのサービスである。マスターはセッションの管理、ノードのデバイスへの配置、ワーカーへのサブグラフ実行命令の発行を担う。

#### 1. CreateSession

セッションを新規作成し、グラフ定義と設定を登録する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc CreateSession(CreateSessionRequest) returns (CreateSessionResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: CreateSessionRequest**

```protobuf
{
  "graph_def": GraphDef,       // 初期グラフ定義
  "config": ConfigProto,       // 設定オプション
  "target": "string"           // クライアント側ターゲット文字列
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| graph_def | GraphDef | - | 初期グラフ定義 |
| config | ConfigProto | - | セッション設定オプション |
| target | string | - | クライアント側から見たターゲット文字列 |

**レスポンス: CreateSessionResponse**

```protobuf
{
  "session_handle": "string",  // セッションハンドル
  "graph_version": 0           // グラフの初期バージョン番号
}
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| session_handle | string | 後続の呼び出しで使用するセッションハンドル |
| graph_version | int64 | グラフの初期バージョン番号（ExtendSessionで使用） |

---

#### 2. ExtendSession

既存セッションのグラフに新しいノードを追加する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc ExtendSession(ExtendSessionRequest) returns (ExtendSessionResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: ExtendSessionRequest**

```protobuf
{
  "session_handle": "string",       // セッションハンドル（必須）
  "graph_def": GraphDef,            // 追加するノード（必須）
  "current_graph_version": 0        // 現在のグラフバージョン（必須）
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | CreateSessionで返却されたセッションハンドル |
| graph_def | GraphDef | ○ | 追加するノード群。既存ノードと同名の場合はILLEGAL_ARGUMENTエラー |
| current_graph_version | int64 | ○ | サーバー側バージョンと一致しない場合はFAILED_PRECONDITIONエラー |

**レスポンス: ExtendSessionResponse**

```protobuf
{
  "new_graph_version": 0    // 新しいグラフバージョン番号
}
```

---

#### 3. PartialRunSetup

将来の部分実行（Partial Run）のためのフィード、フェッチ、ターゲットを事前設定する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc PartialRunSetup(PartialRunSetupRequest) returns (PartialRunSetupResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: PartialRunSetupRequest**

```protobuf
{
  "session_handle": "string",   // セッションハンドル（必須）
  "feed": ["string"],           // 将来のステップでフィードするテンソル名
  "fetch": ["string"],          // フェッチするテンソル名
  "target": ["string"],         // 実行するがフェッチしないノード名
  "request_id": 0               // リトライ検出用リクエストID
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | CreateSessionで返却されたセッションハンドル |
| feed | repeated string | - | 将来のステップでフィードするテンソル名 |
| fetch | repeated string | - | フェッチするテンソル名 |
| target | repeated string | - | 実行ターゲットノード名（出力はフェッチしない） |
| request_id | int64 | - | リクエスト一意識別子。0の場合リトライ検出無効 |

**レスポンス: PartialRunSetupResponse**

```protobuf
{
  "partial_run_handle": "string"  // 部分実行ハンドル
}
```

---

#### 4. RunStep

グラフの計算ステップを実行する。Session.Run() APIの実装に相当する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc RunStep(RunStepRequest) returns (RunStepResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: RunStepRequest**

```protobuf
{
  "session_handle": "string",
  "feed": [NamedTensorProto],
  "fetch": ["string"],
  "target": ["string"],
  "options": RunOptions,
  "partial_run_handle": "string",
  "store_errors_in_response_body": false,
  "request_id": 0
}
```

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | セッションハンドル |
| feed | repeated NamedTensorProto | - | フィードするテンソル群 |
| fetch | repeated string | - | フェッチするテンソル名のリスト |
| target | repeated string | - | 実行ターゲットノード名 |
| options | RunOptions | - | 実行オプション |
| partial_run_handle | string | - | 部分実行ハンドル（指定時は部分実行） |
| store_errors_in_response_body | bool | - | trueの場合、長いエラーメッセージをレスポンスボディに格納 |
| request_id | int64 | - | リトライ検出用一意識別子 |

**レスポンス: RunStepResponse**

```protobuf
{
  "tensor": [NamedTensorProto],
  "metadata": RunMetadata,
  "status_code": 0,
  "status_error_message": "string"
}
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| tensor | repeated NamedTensorProto | フェッチされたテンソル群 |
| metadata | RunMetadata | 実行メタデータ（オプション指定時に返却） |
| status_code | error.Code | store_errors_in_response_body時の実際のステータスコード |
| status_error_message | string | store_errors_in_response_body時のエラーメッセージ |

---

#### 5. CloseSession

セッションを閉じ、関連リソースを解放する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc CloseSession(CloseSessionRequest) returns (CloseSessionResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: CloseSessionRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | 閉じるセッションのハンドル |

**レスポンス: CloseSessionResponse**

空メッセージ。

---

#### 6. ListDevices

マスターが利用可能なデバイスの一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc ListDevices(ListDevicesRequest) returns (ListDevicesResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: ListDevicesRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | - | セッションハンドル（省略時はマスター起動時のClusterSpecを使用） |

**レスポンス: ListDevicesResponse**

```protobuf
{
  "local_device": [DeviceAttributes],
  "remote_device": [DeviceAttributes]
}
```

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| local_device | repeated DeviceAttributes | ローカルデバイスの属性一覧 |
| remote_device | repeated DeviceAttributes | リモートデバイスの属性一覧 |

---

#### 7. Reset

全既存セッションを破棄し、指定コンテナ内のリソースを解放する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc Reset(ResetRequest) returns (ResetResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: ResetRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| container | repeated string | - | リソースを解放するコンテナ名リスト。空の場合はデフォルトコンテナ |
| device_filters | repeated string | - | デバイスフィルタ（例: "/job:ps", "/job:worker/replica:3"） |

**レスポンス: ResetResponse**

空メッセージ。

---

#### 8. MakeCallable

Callableを登録し、RunCallableで効率的に実行できるようにする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc MakeCallable(MakeCallableRequest) returns (MakeCallableResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: MakeCallableRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | セッションハンドル |
| options | CallableOptions | - | Callableの挙動を定義するオプション |
| request_id | int64 | - | リトライ検出用リクエストID |

**レスポンス: MakeCallableResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| handle | int64 | 作成されたCallableのハンドル |

---

#### 9. RunCallable

MakeCallableで登録したCallableを実行する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc RunCallable(RunCallableRequest) returns (RunCallableResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: RunCallableRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | セッションハンドル |
| handle | int64 | ○ | MakeCallableで返却されたハンドル |
| feed | repeated TensorProto | - | CallableOptions.feedの順序でのテンソル値 |
| request_id | int64 | - | リトライ検出用リクエストID |

**レスポンス: RunCallableResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| fetch | repeated TensorProto | CallableOptions.fetchの順序での出力テンソル |
| metadata | RunMetadata | 実行メタデータ |

---

#### 10. ReleaseCallable

MakeCallableで登録したCallableのリソースを解放する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.MasterService` |
| RPCメソッド | `rpc ReleaseCallable(ReleaseCallableRequest) returns (ReleaseCallableResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/master_service.proto` |

**リクエスト: ReleaseCallableRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | セッションハンドル |
| handle | int64 | ○ | 解放するCallableのハンドル |

**レスポンス: ReleaseCallableResponse**

空メッセージ。

---

### WorkerService

WorkerServiceは、MasterServiceの代わりにローカルデバイス上でデータフローグラフを実行するサービスである。ワーカーは複数の「登録済みグラフ」を管理し、それぞれがクライアントグラフのサブグラフに対応する。

#### 11. GetStatus

ワーカーのデバイス属性を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc GetStatus(GetStatusRequest) returns (GetStatusResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: GetStatusRequest**

空メッセージ。

**レスポンス: GetStatusResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| device_attributes | repeated DeviceAttributes | ワーカーのデバイス属性一覧 |

---

#### 12. CreateWorkerSession

ワーカーセッションを作成し、クラスタ構成を登録する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc CreateWorkerSession(CreateWorkerSessionRequest) returns (CreateWorkerSessionResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: CreateWorkerSessionRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | セッションハンドル |
| server_def | ServerDef | - | TensorFlowワーカーの構成定義 |
| isolate_session_state | bool | - | trueの場合、変数等のリソースを他セッションと共有しない |
| cluster_device_attributes | repeated DeviceAttributes | - | クラスタ内全デバイスの属性 |
| master_task | string | - | リクエスト元のマスタータスク名 |
| master_incarnation | int64 | - | マスタータスクのローカルCPUデバイスのincarnation ID |

**レスポンス: CreateWorkerSessionResponse**

空メッセージ。

---

#### 13. DeleteWorkerSession

ワーカーセッションを削除し、関連する全てのワーカー側ステートを解放する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc DeleteWorkerSession(DeleteWorkerSessionRequest) returns (DeleteWorkerSessionResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: DeleteWorkerSessionRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | 削除するセッションのハンドル |

**レスポンス: DeleteWorkerSessionResponse**

空メッセージ。

---

#### 14. RegisterGraph

マスターが配置したサブグラフをワーカーに登録する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc RegisterGraph(RegisterGraphRequest) returns (RegisterGraphResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: RegisterGraphRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | ○ | セッションハンドル |
| create_worker_session_called | bool | - | CreateWorkerSessionが呼ばれたかどうか |
| graph_def | GraphDef | ○ | サブグラフ定義（各ノードにdevice_name設定済み） |
| graph_options | GraphOptions | - | グラフオプション |
| debug_options | DebugOptions | - | tfdbg用デバッグオプション |
| collective_graph_key | int64 | - | Collective Ops用の調整キー |
| config_proto | ConfigProto | - | セッションのConfigProto |

**レスポンス: RegisterGraphResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| graph_handle | string | 登録成功時に返却されるグラフハンドル |

---

#### 15. DeregisterGraph

登録済みサブグラフの登録を解除する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc DeregisterGraph(DeregisterGraphRequest) returns (DeregisterGraphResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: DeregisterGraphRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | - | セッションハンドル（空の場合グローバル名前空間） |
| create_worker_session_called | bool | - | CreateWorkerSessionが呼ばれたかどうか |
| graph_handle | string | ○ | RegisterGraphで返却されたグラフハンドル |

**レスポンス: DeregisterGraphResponse**

空メッセージ。

---

#### 16. RunGraph

登録済みサブグラフを実行する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc RunGraph(RunGraphRequest) returns (RunGraphResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: RunGraphRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| session_handle | string | - | セッションハンドル |
| create_worker_session_called | bool | - | CreateWorkerSessionが呼ばれたかどうか |
| graph_handle | string | ○ | RegisterGraphで返却されたグラフハンドル |
| step_id | int64 | ○ | グローバルユニークなステップID |
| exec_opts | ExecutorOpts | - | 実行オプション（コスト記録、タイムライン等） |
| send | repeated NamedTensorProto | - | 実行前にグラフに送信するテンソル |
| recv_key | repeated string | - | 実行後にフェッチするテンソルキー |
| is_partial | bool | - | 部分実行かどうか |
| is_last_partial_run | bool | - | 部分実行シーケンスの最後かどうか |
| store_errors_in_response_body | bool | - | エラーをレスポンスボディに格納するか |
| request_id | int64 | - | リトライ検出用リクエストID |

**レスポンス: RunGraphResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| recv | repeated NamedTensorProto | recv_keyに対応するテンソル |
| step_stats | StepStats | ステップ統計情報 |
| cost_graph | CostGraphDef | コストグラフ |
| partition_graph | repeated GraphDef | パーティショングラフ |
| status_code | error.Code | 実際のステータスコード |
| status_error_message | string | エラーメッセージ |

---

#### 17. CleanupGraph

ステップ実行後の残存ステートをクリーンアップする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc CleanupGraph(CleanupGraphRequest) returns (CleanupGraphResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: CleanupGraphRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| step_id | int64 | ○ | クリーンアップ対象のステップID |

**レスポンス: CleanupGraphResponse**

空メッセージ。

---

#### 18. CleanupAll

指定コンテナ内の全リソースを解放する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc CleanupAll(CleanupAllRequest) returns (CleanupAllResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: CleanupAllRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| container | repeated string | - | コンテナ名リスト（空の場合デフォルトコンテナ） |

**レスポンス: CleanupAllResponse**

空メッセージ。

---

#### 19. RecvTensor

他のワーカーからテンソルを受信する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc RecvTensor(RecvTensorRequest) returns (RecvTensorResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: RecvTensorRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| step_id | int64 | ○ | テンソルが生成されるステップID |
| rendezvous_key | string | ○ | テンソル受信チャネルの識別キー |
| dma_ok | bool | - | 帯域外DMA転送を使用するか |
| client_locality | DeviceLocality | - | クライアント側デバイスのローカリティ情報 |
| server_locality | DeviceLocality | - | サーバー側デバイスのローカリティ情報 |
| transport_options | google.protobuf.Any | - | RPCサブシステム固有の情報 |
| request_id | int64 | - | リトライ検出用リクエストID |

**レスポンス: RecvTensorResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| tensor | TensorProto | 受信テンソル |
| is_dead | bool | deadノードの出力の場合true（内容は無効） |
| send_start_micros | int64 | テンソルが利用可能になった時刻（マイクロ秒） |
| transport_options | google.protobuf.Any | DMA等の追加情報 |
| require_ack | bool | 受信側がMarkRecvFinishedRequestを送信すべきかどうか |

---

#### 20. MarkRecvFinished

テンソル受信の完了を送信者に通知する（レスポンスキャッシュ管理用）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc MarkRecvFinished(MarkRecvFinishedRequest) returns (MarkRecvFinishedResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: MarkRecvFinishedRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| request_id | int64 | ○ | 完了したRecvTensorリクエストのID |

**レスポンス: MarkRecvFinishedResponse**

空メッセージ。

---

#### 21. Logging

RPCロギングの有効化/無効化、およびログデータの取得を行う。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc Logging(LoggingRequest) returns (LoggingResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: LoggingRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| enable_rpc_logging | bool | - | trueでRPCロギングを有効化 |
| disable_rpc_logging | bool | - | trueでRPCロギングを無効化 |
| clear | bool | - | trueで保存済みログデータを全て破棄 |
| fetch_step_id | repeated int64 | - | 取得するステップIDリスト（取得後はストアから削除） |

**レスポンス: LoggingResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| step | repeated LabeledStepStats | ステップIDとステップ統計のペア |

**注意**: この機能はオープンソース版では完全にはサポートされておらず、メッセージ形式は変更される可能性がある。

---

#### 22. Tracing

分散トレーシングの設定を行う。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc Tracing(TracingRequest) returns (TracingResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: TracingRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| options | TraceOpts | - | トレースオプション（duration, use_kernel_profiler等） |

**レスポンス: TracingResponse**

空メッセージ。

**注意**: この機能はオープンソース版では完全にはサポートされていない。

---

#### 23. RecvBuf

Collective Ops用のバッファデータを受信する（RDMA等の効率的転送をサポート）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc RecvBuf(RecvBufRequest) returns (RecvBufResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: RecvBufRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| step_id | int64 | ○ | BufRendezvousを見つけるためのステップID |
| buf_rendezvous_key | string | ○ | BufRendezvousエントリの識別文字列 |
| num_bytes | int64 | ○ | 期待するバイトサイズ |
| buf_ptr | fixed64 | - | RDMA使用時のクライアント側宛先アドレス |
| client_locality | DeviceLocality | - | クライアント側デバイスのローカリティ |
| server_locality | DeviceLocality | - | サーバー側デバイスのローカリティ |
| transport_options | google.protobuf.Any | - | 実装固有データ |
| src_device | string | - | ソースデバイス名 |
| dst_device | string | - | 宛先デバイス名 |
| request_id | int64 | - | リトライ検出用ID |
| src_incarnation | uint64 | - | ソースデバイスのincarnation番号 |

**レスポンス: RecvBufResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| buf_ptr | fixed64 | サーバー側ソースフィールドのアドレス |
| num_bytes | int64 | buf_ptrフィールドのバイト長 |
| is_dead | bool | 値がdeadの場合true |
| transport_options | google.protobuf.Any | 実装固有データ |
| send_start_micros | int64 | 送信開始時刻 |
| require_ack | bool | 受信者がackを送信すべきかどうか |

---

#### 24. GetStepSequence

Collective Ops間で同一step_idを調整するためのステップシーケンスを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc GetStepSequence(GetStepSequenceRequest) returns (GetStepSequenceResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: GetStepSequenceRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| graph_key | repeated int64 | - | ステップシーケンスを取得するグラフキー |

**レスポンス: GetStepSequenceResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| step_sequence | repeated StepSequence | グラフキーと次のstep_idのペア |

---

#### 25. CompleteGroup

Collective Opsのグループメンバーシップを完了するために、デバイス名を提供する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc CompleteGroup(CompleteGroupRequest) returns (CompleteGroupResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: CompleteGroupRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| group_key | int32 | ○ | グループ識別キー |
| group_size | int32 | ○ | グループサイズ |
| device_type | string | ○ | デバイスタイプ |
| collective_type | int32 | - | Collectiveタイプ |
| device_attributes | DeviceAttributes | - | デバイス属性 |

**レスポンス: CompleteGroupResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| group_key | int32 | グループキー |
| group_size | int32 | グループサイズ |
| device_type | string | デバイスタイプ |
| num_tasks | int32 | デバイスをホストする個別タスク数 |
| communicator_key | bytes | コミュニケーターキー |
| device_attributes | repeated DeviceAttributes | グループ内全デバイスの属性 |

---

#### 26. CompleteInstance

Collective Opsのインスタンスが一貫して宣言されていることを確認する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.grpc.WorkerService` |
| RPCメソッド | `rpc CompleteInstance(CompleteInstanceRequest) returns (CompleteInstanceResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/worker_service.proto` |

**リクエスト: CompleteInstanceRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| name | string | ○ | オペレーション名 |
| type | int32 | ○ | Collectiveタイプ |
| data_type | DataType | ○ | データ型 |
| shape | TensorShapeProto | ○ | テンソル形状 |
| group_key | int32 | ○ | グループキー |
| group_size | int32 | ○ | グループサイズ |
| instance_key | int32 | ○ | インスタンスキー |
| device_type | string | ○ | デバイスタイプ |
| subdiv_offset | repeated int32 | - | サブディビジョンオフセット |
| device | string | - | デバイス名 |
| is_source | bool | - | ソースかどうか（ブロードキャスト時） |
| step_id | int64 | - | ステップID |

**レスポンス: CompleteInstanceResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| instance_key | int32 | インスタンスキー |
| source_rank | int32 | ブロードキャスト時のソースランク |

---

### EagerService

EagerServiceは、リモートEagerエクゼキューターの代わりにローカルデバイス上でオペレーションを即時実行するサービスである。

#### 27. CreateContext

ワーカーを初期化し、クラスタ内の他のワーカー情報と認証トークンを交換する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.eager.EagerService` |
| RPCメソッド | `rpc CreateContext(CreateContextRequest) returns (CreateContextResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/eager_service.proto` |

**リクエスト: CreateContextRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| server_def | ServerDef | ○ | クラスタ構成とワーカーの位置 |
| async | bool | - | trueで非同期実行モード |
| keep_alive_secs | int64 | - | コンテキストのキープアライブ秒数 |
| version_def | VersionDef | - | エンキューするOpsのバージョン定義 |
| cluster_device_attributes | repeated DeviceAttributes | - | クラスタ内デバイス属性 |
| context_id | fixed64 | ○ | 作成するコンテキストのID（ランダム生成） |
| context_view_id | fixed64 | - | コンテキストのビューID |
| lazy_copy_remote_function_inputs | bool | - | trueでリモート入力の遅延コピーを有効化 |
| clear_existing_contexts | bool | - | trueでワーカー環境のリソースマネージャーをクリア |

**レスポンス: CreateContextResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| device_attributes | repeated DeviceAttributes | ワーカーがローカルにアクセス可能なデバイス一覧 |

---

#### 28. UpdateContext

既存ワーカーのEagerコンテキストを更新する（分散Eagerクラスタのサーバー集合を更新する際に使用）。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.eager.EagerService` |
| RPCメソッド | `rpc UpdateContext(UpdateContextRequest) returns (UpdateContextResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/eager_service.proto` |

**リクエスト: UpdateContextRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| server_def | ServerDef | ○ | クラスタ構成とワーカーの位置 |
| cluster_device_attributes | repeated DeviceAttributes | - | クラスタ内デバイス属性（空の場合はビューIDのインクリメントのみ） |
| context_id | fixed64 | ○ | 更新対象のコンテキストID |
| context_view_id | fixed64 | ○ | コンテキストのビューID（連続的にインクリメント） |

**レスポンス: UpdateContextResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| device_attributes | repeated DeviceAttributes | ワーカーがローカルにアクセス可能なデバイス一覧 |

---

#### 29. Enqueue

オペレーション（Execute, DeleteTensorHandle等）のリストを受け取り、同期モードでは実行、非同期モードではエンキューする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.eager.EagerService` |
| RPCメソッド | `rpc Enqueue(EnqueueRequest) returns (EnqueueResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/eager_service.proto` |

**リクエスト: EnqueueRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| context_id | fixed64 | ○ | コンテキストID |
| queue | repeated QueueItem | ○ | 実行するオペレーションのリスト |

**QueueItemの種類**:
- `handle_to_decref`: テンソルハンドルの参照カウント減少
- `operation`: Eagerオペレーションの実行
- `send_tensor`: テンソルの送信
- `register_function`: 関数の登録
- `cleanup_function`: 関数のクリーンアップ
- `sync_remote_executor_for_stream`: リモートエクゼキューターの同期
- `send_packed_handle`: パックされたテンソルハンドルの送信
- `remove_function`: 関数の削除

**レスポンス: EnqueueResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| queue_response | repeated QueueResponse | 各QueueItemに対するレスポンス |

---

#### 30. StreamingEnqueue

Enqueueのストリーミング版。前のリクエストのレスポンスを待たずに後続のリクエストを送信できる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.eager.EagerService` |
| RPCメソッド | `rpc StreamingEnqueue(stream EnqueueRequest) returns (stream EnqueueResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/eager_service.proto` |

双方向ストリーミングRPC。リクエスト/レスポンスの型はEnqueue/EnqueueResponseと同一。

---

#### 31. WaitQueueDone

指定されたオペレーションIDの完了を待機する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.eager.EagerService` |
| RPCメソッド | `rpc WaitQueueDone(WaitQueueDoneRequest) returns (WaitQueueDoneResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/eager_service.proto` |

**リクエスト: WaitQueueDoneRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| context_id | fixed64 | ○ | コンテキストID |
| op_id | repeated int64 | - | 待機するオペレーションID（空の場合は全保留中のオペレーション） |

**レスポンス: WaitQueueDoneResponse**

空メッセージ。

---

#### 32. RunComponentFunction

コンポーネント関数を非同期モードで実行する。メイン関数から分離された並列実行が可能。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.eager.EagerService` |
| RPCメソッド | `rpc RunComponentFunction(RunComponentFunctionRequest) returns (RunComponentFunctionResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/eager_service.proto` |

**リクエスト: RunComponentFunctionRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| context_id | fixed64 | ○ | コンテキストID |
| operation | Operation | ○ | 実行するオペレーション |
| output_num | repeated int32 | - | 親関数の出力インデックス |

**レスポンス: RunComponentFunctionResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| shape | repeated TensorShapeProto | 出力テンソルの形状 |
| tensor | repeated TensorProto | 出力テンソル |

---

#### 33. KeepAlive

コンテキストのガベージコレクションを遅延させ、コンテキストの存在を検証する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.eager.EagerService` |
| RPCメソッド | `rpc KeepAlive(KeepAliveRequest) returns (KeepAliveResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/eager_service.proto` |

**リクエスト: KeepAliveRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| context_id | fixed64 | ○ | 維持するコンテキストのID |

**レスポンス: KeepAliveResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| context_view_id | fixed64 | コンテキストのビューID |

---

#### 34. CloseContext

コンテキストを閉じ、以降そのコンテキストIDを使用した呼び出しを無効にする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.eager.EagerService` |
| RPCメソッド | `rpc CloseContext(CloseContextRequest) returns (CloseContextResponse)` |
| 定義ファイル | `tensorflow/core/protobuf/eager_service.proto` |

**リクエスト: CloseContextRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| context_id | fixed64 | ○ | 閉じるコンテキストのID |
| context_view_id | fixed64 | - | コンテキストのビューID |

**レスポンス: CloseContextResponse**

空メッセージ。

---

### EventListener (Debug Service)

EventListenerは、デバッグ対象のTensorFlowランタイムからEventプロトコルバッファを受信するサービスである。

#### 35. SendEvents

デバッグイベント（中間テンソル、グラフ定義等）をストリーミングで送信する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.EventListener` |
| RPCメソッド | `rpc SendEvents(stream Event) returns (stream EventReply)` |
| 定義ファイル | `tensorflow/core/debug/debug_service.proto` |

双方向ストリーミングRPC。

**リクエスト**: `stream Event` - イベントのストリーム

**レスポンス**: `stream EventReply` - デバッグオペレーション状態変更やテンソルオーバーライド情報

---

#### 36. SendTracebacks

TensorFlow実行呼び出しのトレースバック情報を送信する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.EventListener` |
| RPCメソッド | `rpc SendTracebacks(CallTraceback) returns (EventReply)` |
| 定義ファイル | `tensorflow/core/debug/debug_service.proto` |

**リクエスト: CallTraceback**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| call_type | CallType | - | 呼び出しタイプ（GRAPH_EXECUTION / EAGER_EXECUTION） |
| call_key | string | - | 呼び出しキー |
| origin_stack | tfprof.CodeDef | - | 呼び出し元のスタックトレース |
| origin_id_to_string | map<int64, string> | - | IDから文字列値へのマッピング |
| graph_traceback | tfprof.OpLogProto | - | グラフのトレースバック |
| graph_version | int64 | - | グラフのバージョン |

**レスポンス: EventReply**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| debug_op_state_changes | repeated DebugOpStateChange | デバッグオペレーション状態変更 |
| tensor | TensorProto | 現在のテンソル値をオーバーライドするための新テンソル値 |

---

#### 37. SendSourceFiles

デバッグ対象のソースコードファイル群を送信する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.EventListener` |
| RPCメソッド | `rpc SendSourceFiles(DebuggedSourceFiles) returns (EventReply)` |
| 定義ファイル | `tensorflow/core/debug/debug_service.proto` |

---

### RpcService

RpcServiceは、リモートサーバーに登録された関数を呼び出すための実験的サービスである。

#### 38. Call

リモートサーバーに登録された関数を呼び出す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.rpc.RpcService` |
| RPCメソッド | `rpc Call(CallRequest) returns (CallResponse)` |
| 定義ファイル | `tensorflow/distribute/experimental/rpc/proto/tf_rpc_service.proto` |

**リクエスト: CallRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| method | string | ○ | 呼び出すメソッド名 |
| input_tensors | repeated TensorProto | - | 入力テンソル |

**レスポンス: CallResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| output_tensors | repeated TensorProto | 出力テンソル |

---

#### 39. List

リモートサーバーで利用可能なメソッド一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.rpc.RpcService` |
| RPCメソッド | `rpc List(ListRequest) returns (ListResponse)` |
| 定義ファイル | `tensorflow/distribute/experimental/rpc/proto/tf_rpc_service.proto` |

**リクエスト: ListRequest**

空メッセージ。

**レスポンス: ListResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| registered_methods | repeated RegisteredMethod | 登録済みメソッドのリスト（名前、入力/出力仕様） |

---

### DispatcherService (tf.data)

DispatcherServiceは、tf.dataサービスのディスパッチャーが提供するサービスであり、データセットの登録、ジョブ管理、スプリット配布、ワーカー/クライアント管理を担う。

#### 40. WorkerHeartbeat

ワーカーからの定期的なハートビートを処理し、新しいタスクやスナップショットタスクを返却する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc WorkerHeartbeat(WorkerHeartbeatRequest) returns (WorkerHeartbeatResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: WorkerHeartbeatRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| worker_address | string | ○ | ワーカーアドレス |
| transfer_servers | repeated DataTransferServerInfo | - | データ転送サーバー情報 |
| worker_tags | repeated string | - | ワーカータグ |
| worker_uid | int64 | - | ワーカーのUID |
| current_tasks | repeated int64 | - | 現在のタスクIDリスト |
| snapshot_task_progress | map<string, SnapshotTaskProgress> | - | スナップショットタスクの進捗 |
| active_tasks | repeated ActiveTask | - | アクティブタスク情報（処理時間含む） |

**レスポンス: WorkerHeartbeatResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| new_tasks | repeated TaskDef | 新しいタスク定義 |
| tasks_to_delete | repeated int64 | 削除すべきタスクID |
| snapshot_tasks | repeated SnapshotTaskDef | スナップショットタスク |

---

#### 41. WorkerUpdate

ワーカーの状態更新をディスパッチャーに報告する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc WorkerUpdate(WorkerUpdateRequest) returns (WorkerUpdateResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: WorkerUpdateRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| worker_address | string | ○ | ワーカーアドレス |
| updates | repeated TaskProgress | - | タスク進捗のリスト |

**レスポンス: WorkerUpdateResponse**

空メッセージ。

---

#### 42. GetDatasetDef

データセット定義を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc GetDatasetDef(GetDatasetDefRequest) returns (GetDatasetDefResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: GetDatasetDefRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| dataset_id | string | ○ | データセットID |

**レスポンス: GetDatasetDefResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| dataset_def | DatasetDef | データセット定義 |

---

#### 43. GetSplit

イテレーションの次のスプリットを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc GetSplit(GetSplitRequest) returns (GetSplitResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: GetSplitRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| iteration_id | int64 | ○ | イテレーションID |
| repetition | int64 | - | リピティション番号 |
| split_provider_index | int64 | - | スプリットプロバイダーインデックス |

**レスポンス: GetSplitResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| split | TensorProto | スプリットデータ |
| end_of_splits | bool | スプリットの終了フラグ |

---

#### 44. GetVersion

tf.dataサービスのAPIバージョンを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc GetVersion(GetVersionRequest) returns (GetVersionResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: GetVersionRequest**

空メッセージ。

**レスポンス: GetVersionResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| version | int64 | APIバージョン番号 |

---

#### 45. GetOrRegisterDataset

データセットを登録するか、既に登録済みの場合はそのIDを返す。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc GetOrRegisterDataset(GetOrRegisterDatasetRequest) returns (GetOrRegisterDatasetResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: GetOrRegisterDatasetRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| dataset | DatasetDef | ○ | 登録するデータセット |
| metadata | DataServiceMetadata | - | tf.dataサービスのメタデータ |
| dataset_id | string | - | 指定時はこのIDで登録（省略時は自動生成） |

**レスポンス: GetOrRegisterDatasetResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| dataset_id | string | 登録されたデータセットのID |

---

#### 46. GetOrCreateJob

ジョブが存在すれば取得し、存在しなければ作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc GetOrCreateJob(GetOrCreateJobRequest) returns (GetOrCreateJobResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: GetOrCreateJobRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| dataset_id | string | ○ | データセットID |
| processing_mode_def | ProcessingModeDef | ○ | データ処理モード（シャーディングポリシー等） |
| job_name | string | - | 共有ジョブ名（省略時は新規ジョブを常に作成） |
| num_consumers | int64 | - | コンシューマー数（ラウンドロビン配信時） |
| use_cross_trainer_cache | bool | - | クロストレーナーキャッシュの有効化 |
| target_workers | TargetWorkers | - | 読み取り元ワーカーの指定 |

**レスポンス: GetOrCreateJobResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| job_id | int64 | ジョブID |

---

#### 47. GetOrCreateIteration

イテレーションが存在すれば取得し、存在しなければ作成する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc GetOrCreateIteration(GetOrCreateIterationRequest) returns (GetOrCreateIterationResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: GetOrCreateIterationRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| job_id | int64 | ○ | ジョブID |
| repetition | int64 | - | リピティション番号 |

**レスポンス: GetOrCreateIterationResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| iteration_client_id | int64 | イテレーションクライアントID |

---

#### 48. MaybeRemoveTask

ラウンドロビン読み取りイテレーションからタスクの削除を試みる。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc MaybeRemoveTask(MaybeRemoveTaskRequest) returns (MaybeRemoveTaskResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: MaybeRemoveTaskRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| task_id | int64 | ○ | タスクID |
| consumer_index | int64 | ○ | コンシューマーインデックス |
| round | int64 | ○ | ラウンド番号 |

**レスポンス: MaybeRemoveTaskResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| removed | bool | タスクが削除されたかどうか |

---

#### 49. ReleaseIterationClient

イテレーションクライアントを解放し、イテレーションのクリーンアップを可能にする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc ReleaseIterationClient(ReleaseIterationClientRequest) returns (ReleaseIterationClientResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: ReleaseIterationClientRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| iteration_client_id | int64 | ○ | 解放するイテレーションクライアントID |

**レスポンス: ReleaseIterationClientResponse**

空メッセージ。

---

#### 50. ClientHeartbeat

クライアントの生存を通知し、タスク情報の更新を受け取る。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc ClientHeartbeat(ClientHeartbeatRequest) returns (ClientHeartbeatResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: ClientHeartbeatRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| iteration_client_id | int64 | ○ | イテレーションクライアントID |
| current_round | int64 | - | 現在のラウンド（ラウンドロビン時） |
| blocked_round | int64 | - | ブロックに成功したラウンド |
| target_processing_time_nsec | double | - | クライアントが観測したターゲット処理時間（ナノ秒） |

**レスポンス: ClientHeartbeatResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| task_info | repeated TaskInfo | クライアントが読み取るべき全タスクのリスト |
| block_round | int64 | クライアントが開始すべきでないラウンド |
| iteration_finished | bool | イテレーションが完了したかどうか |
| deployment_mode | DeploymentMode | デプロイメントモード |

---

#### 51. GetWorkers

ディスパッチャーに登録された全ワーカーのリストを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc GetWorkers(GetWorkersRequest) returns (GetWorkersResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: GetWorkersRequest**

空メッセージ。

**レスポンス: GetWorkersResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| workers | repeated WorkerInfo | ワーカー情報（アドレス等） |

---

#### 52. GetDataServiceMetadata

登録済みデータセットのメタデータを取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc GetDataServiceMetadata(GetDataServiceMetadataRequest) returns (GetDataServiceMetadataResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

---

#### 53. GetDataServiceConfig

データサービスクラスタの設定を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc GetDataServiceConfig(GetDataServiceConfigRequest) returns (GetDataServiceConfigResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

---

#### 54. Snapshot

データセットの出力をディスクに永続化するプロセスを開始する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.DispatcherService` |
| RPCメソッド | `rpc Snapshot(SnapshotRequest) returns (SnapshotResponse)` |
| 定義ファイル | `tensorflow/core/data/service/dispatcher.proto` |

**リクエスト: SnapshotRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| dataset | DatasetDef | ○ | スナップショット対象のデータセット |
| path | string | ○ | スナップショットの保存先パス |
| metadata | DistributedSnapshotMetadata | - | スナップショットのメタデータ |

**レスポンス: SnapshotResponse**

空メッセージ。

---

#### 55. GetSnapshotSplit / GetSnapshotStreams / DisableCompressionAtRuntime

スナップショットのスプリット取得、ストリーム情報取得、ランタイム圧縮無効化の制御を行うRPCメソッド群。詳細は `tensorflow/core/data/service/dispatcher.proto` を参照。

---

### WorkerService (tf.data)

tf.dataサービスのワーカーが提供するサービスであり、データセットタスクの処理とデータ要素の配信を担う。

#### 56. ProcessTask

データセットタスクを処理し、クライアントがデータ要素を取得できるようにする。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.WorkerService` |
| RPCメソッド | `rpc ProcessTask(ProcessTaskRequest) returns (ProcessTaskResponse)` |
| 定義ファイル | `tensorflow/core/data/service/worker.proto` |

**リクエスト: ProcessTaskRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| task | TaskDef | ○ | 処理するタスク定義 |

**レスポンス: ProcessTaskResponse**

空メッセージ。

---

#### 57. GetElement

次のデータセット要素を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.WorkerService` |
| RPCメソッド | `rpc GetElement(GetElementRequest) returns (GetElementResponse)` |
| 定義ファイル | `tensorflow/core/data/service/worker.proto` |

**リクエスト: GetElementRequest**

| フィールド名 | 型 | 必須 | 説明 |
| --- | --- | --- | --- |
| task_id | int64 | ○ | 要素を取得するタスクID |
| consumer_index | int64 | - | コンシューマーインデックス（ラウンドロビン時） |
| round_index | int64 | - | ラウンドインデックス（同期読み取り時） |
| skipped_previous_round | bool | - | 前のラウンドをスキップしたかどうか |
| allow_skip | bool | - | trueの場合、データがない時にスキップ可能 |
| trainer_id | string | - | マルチトレーナーキャッシュ用トレーナーID |

**レスポンス: GetElementResponse**

| フィールド名 | 型 | 説明 |
| --- | --- | --- |
| compressed / uncompressed | oneof | データ要素（圧縮または非圧縮） |
| element_index | int64 | タスク内での要素インデックス |
| end_of_sequence | bool | イテレーター終了フラグ |
| skip_task | bool | ラウンドがスキップされたかどうか |

---

#### 58. GetWorkerTasks

ワーカーが現在実行中のタスク一覧を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.WorkerService` |
| RPCメソッド | `rpc GetWorkerTasks(GetWorkerTasksRequest) returns (GetWorkerTasksResponse)` |
| 定義ファイル | `tensorflow/core/data/service/worker.proto` |

---

#### 59. GetSnapshotTaskProgresses

ワーカーが実行中のスナップショットタスクの進捗を取得する。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| サービス | `tensorflow.data.WorkerService` |
| RPCメソッド | `rpc GetSnapshotTaskProgresses(GetSnapshotTaskProgressesRequest) returns (GetSnapshotTaskProgressesResponse)` |
| 定義ファイル | `tensorflow/core/data/service/worker.proto` |

---

## 備考

- 本プロジェクト（TensorFlow）はHTTP REST APIを提供するWebアプリケーションではなく、機械学習フレームワークである。本書で記載しているAPIは全てgRPC（Protocol Buffers）ベースの内部分散通信APIである。
- Logging / Tracing のRPCはオープンソース版では完全にサポートされておらず、メッセージ形式が変更される可能性がある。
- RecvBuf等のCollective Ops関連RPCは実験的な機能であり、変更の可能性がある。
- RpcService（`tensorflow.rpc`パッケージ）は実験的な分散RPC機能であり、`tensorflow/distribute/experimental/` 配下に定義されている。
- tf.dataサービスのDispatcherServiceとWorkerServiceは、分散データパイプラインの管理に使用され、フォールトトレラントモードをサポートする。
