# アーキテクチャ設計書

## 概要

本ドキュメントは、TensorFlow v2.21.0 のアプリケーションアーキテクチャを記述する。TensorFlow は Google が開発したオープンソースの機械学習フレームワークであり、データフローグラフを用いた数値計算を基盤として、ニューラルネットワークの構築・訓練・推論を実現する。本書では、TensorFlow のレイヤー構成、モジュール構成、ミドルウェア構成、データフロー、および横断的関心事について整理する。

## システム全体構成

### アーキテクチャ概要図

[アーキテクチャ構成図](./アーキテクチャ構成図.md) を参照。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| ハードウェアアクセラレータ (GPU/TPU) | C API / デバイスドライバ (CUDA, ROCm, TPU Runtime) | 計算カーネルの実行・高速化 |
| XLA (Accelerated Linear Algebra) | コンパイラ連携 (tf2xla, MLIR) | 計算グラフの最適化コンパイル |
| TensorRT | TF-TRT 統合 (tensorflow/compiler/tf2tensorrt) | NVIDIA GPU 向け推論最適化 |
| SavedModel / Checkpoint ストレージ | ファイルシステム I/O (ローカル, GCS, S3, HDFS) | モデルの保存・復元 |
| TensorBoard | イベントファイル / Summary API | トレーニングの可視化・モニタリング |
| tf.data 外部データソース | データパイプライン (TFRecord, CSV, 各種データベース) | 学習データの読み込み・前処理 |
| gRPC / Remote Worker | gRPC プロトコル (distributed_runtime) | 分散学習・リモート実行 |
| Android / iOS ランタイム | TensorFlow Lite (FlatBuffers, delegates) | モバイル・エッジデバイスでの推論 |

## レイヤー構成

### アーキテクチャスタイル

TensorFlow は **レイヤードアーキテクチャ** を採用している。最上位にユーザー向け Python API 層があり、その下に C/C++ で実装されたコアランタイム層、さらにハードウェア抽象化層（プラットフォーム層）が存在する。各レイヤーは下位レイヤーへの依存のみが許可される単方向依存構造となっている。加えて、コンパイラスタック（XLA/MLIR）が横断的にグラフ最適化を担う。

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Python API 層 | ユーザー向け高水準 API の提供、Eager Execution の実現 | `tensorflow/python/` (keras, eager, data, distribute, saved_model, framework) |
| C API 層 | 言語バインディング向け安定 ABI の提供 | `tensorflow/c/` (c_api.h, c_api_experimental.h, eager/) |
| C++ API 層 | C++ ユーザー向けの計算グラフ構築・実行 API | `tensorflow/cc/` (client, ops, gradients, training) |
| Core Runtime 層 | 計算グラフの構築・最適化・実行、Op カーネル管理、デバイス管理 | `tensorflow/core/` (common_runtime, framework, graph, kernels, ops) |
| コンパイラ層 | 計算グラフのコンパイル最適化、XLA/MLIR 変換 | `tensorflow/compiler/` (xla, mlir, jit, tf2xla, tf2tensorrt) |
| Platform 層 | OS・ハードウェア抽象化、基本ユーティリティ | `tensorflow/core/platform/`, `tensorflow/core/lib/` |
| TensorFlow Lite 層 | モバイル・組込みデバイス向け軽量推論ランタイム | `tensorflow/lite/` (core, delegates, kernels, c/) |

### レイヤー間の依存関係ルール

- **上位から下位への単方向依存**: Python API 層 -> C API 層 -> Core Runtime 層 -> Platform 層
- **Python API 層** は C API 層を pybind11 / SWIG 経由で呼び出す。Core Runtime を直接参照しない。
- **C API 層** は Core Runtime 層の安定したインターフェースのみを公開する。ABI の後方互換性を維持する。
- **Core Runtime 層** は Platform 層の抽象化を通じてハードウェアにアクセスする。
- **コンパイラ層** は Core Runtime 層のグラフ表現を入力として受け取り、最適化されたコードを生成する。Core Runtime からコンパイラ層への参照は JIT コンパイルパスで行われる。
- **TensorFlow Lite 層** は独立したランタイムであり、Core Runtime との直接依存はない。SavedModel からの変換ツール（TFLite Converter）を通じて間接的に連携する。
- **禁止**: 下位レイヤーから上位レイヤーへの参照。Platform 層が Python API 層を参照するなどは禁止。

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| グラフ構築・実行 | 計算グラフの定義、Eager/Graph 実行モード管理 | `core/graph/`, `core/common_runtime/`, `python/eager/`, `python/framework/` |
| Op カーネル | 数学演算、ニューラルネットワーク演算の実装 | `core/kernels/`, `core/ops/` |
| データパイプライン | 学習データの読み込み・変換・バッチ処理 | `python/data/`, `core/kernels/data/` |
| 分散学習 | マルチデバイス・マルチノードでの並列学習 | `core/distributed_runtime/`, `python/distribute/`, `dtensor/` |
| モデル保存・復元 | SavedModel / Checkpoint の永続化と復元 | `python/saved_model/`, `python/checkpoint/`, `cc/saved_model/` |
| コンパイラ最適化 | グラフ最適化、XLA コンパイル、MLIR 変換 | `compiler/jit/`, `compiler/mlir/`, `compiler/tf2xla/` |
| グラフ最適化 (Grappler) | グラフレベルの自動最適化パス | `core/grappler/` |
| 推論最適化 (TF Lite) | モバイル・エッジ向け推論 | `lite/` |
| TPU サポート | TPU デバイス固有の最適化・ランタイム | `core/tpu/`, `dtensor/` |
| プロファイリング | 実行時間・メモリ使用量の計測・分析 | `core/profiler/`, `python/profiler/` |

### パッケージ構造

```
tensorflow/
├── python/                    # Python API 層
│   ├── eager/                 #   Eager Execution エンジン
│   ├── framework/             #   テンソル・Op・グラフの Python ラッパー
│   ├── ops/                   #   Python Op 定義
│   ├── keras/                 #   Keras 高水準 API
│   ├── data/                  #   tf.data パイプライン
│   ├── distribute/            #   分散学習戦略
│   ├── saved_model/           #   SavedModel 保存・復元
│   ├── checkpoint/            #   チェックポイント管理
│   ├── compiler/              #   XLA JIT コンパイル (Python IF)
│   ├── profiler/              #   プロファイリング
│   ├── autograph/             #   Python -> Graph 自動変換
│   └── lite/                  #   TFLite コンバータ
├── c/                         # C API 層
│   ├── c_api.h                #   安定 C API
│   ├── c_api_experimental.h   #   実験的 C API
│   └── eager/                 #   Eager 用 C API
├── cc/                        # C++ API 層
│   ├── client/                #   セッション・スコープ
│   ├── ops/                   #   C++ Op ラッパー
│   ├── gradients/             #   勾配計算
│   └── training/              #   学習用ユーティリティ
├── core/                      # Core Runtime 層
│   ├── framework/             #   テンソル・Op・デバイスの基盤フレームワーク
│   ├── common_runtime/        #   セッション・エグゼキュータ・デバイス管理
│   ├── distributed_runtime/   #   分散実行ランタイム (gRPC)
│   ├── graph/                 #   グラフ構築・変換
│   ├── kernels/               #   Op カーネル実装 (CPU/GPU)
│   ├── ops/                   #   Op 登録・シグネチャ定義
│   ├── grappler/              #   グラフ最適化エンジン
│   ├── platform/              #   OS 抽象化・ユーティリティ
│   ├── lib/                   #   基本ライブラリ (文字列, IO, ハッシュ等)
│   ├── profiler/              #   プロファイラ
│   ├── tpu/                   #   TPU サポート
│   └── protobuf/              #   Protocol Buffers 定義
├── compiler/                  # コンパイラ層
│   ├── jit/                   #   XLA JIT コンパイルパス
│   ├── mlir/                  #   MLIR ベース変換パイプライン
│   ├── tf2xla/                #   TF Op -> XLA HLO 変換
│   ├── tf2tensorrt/           #   TensorRT 統合
│   └── aot/                   #   Ahead-of-Time コンパイル
├── lite/                      # TensorFlow Lite 層
│   ├── core/                  #   Lite ランタイムコア
│   ├── kernels/               #   Lite Op カーネル
│   ├── delegates/             #   ハードウェアデリゲート (GPU, NNAPI等)
│   ├── c/                     #   Lite C API
│   └── examples/              #   サンプルアプリケーション
├── dtensor/                   # DTensor (分散テンソル)
│   ├── cc/                    #   DTensor C++ 実装
│   ├── mlir/                  #   DTensor MLIR パス
│   └── python/                #   DTensor Python API
├── java/                      # Java バインディング
├── go/                        # Go バインディング
├── js/                        # JavaScript (TensorFlow.js) 関連
└── tools/                     # ビルド・開発ツール
```

### コンポーネント依存関係

主要コンポーネント間の依存関係は以下の通りである。

1. **Python API -> C API**: `tensorflow/python/` は pybind11 を通じて `tensorflow/c/` の C API を呼び出す。
2. **C API -> Core Runtime**: `tensorflow/c/` は `tensorflow/core/framework/` および `tensorflow/core/common_runtime/` を内部実装として利用する。
3. **Core Runtime -> Kernels**: `tensorflow/core/common_runtime/` のエグゼキュータが `tensorflow/core/kernels/` のカーネルをディスパッチする。
4. **Core Runtime -> Grappler**: グラフ最適化時に `tensorflow/core/grappler/` の最適化パスが適用される。
5. **JIT -> XLA**: `tensorflow/compiler/jit/` が XLA クラスタを識別し、`tensorflow/compiler/tf2xla/` を通じて XLA HLO に変換する。
6. **MLIR パイプライン**: `tensorflow/compiler/mlir/` が TF グラフを MLIR 表現に変換し、各種最適化パスを適用後、ターゲットコードを生成する。
7. **分散ランタイム -> gRPC**: `tensorflow/core/distributed_runtime/` が gRPC を用いてワーカー間通信を行う。

## ミドルウェア構成

### データベース

TensorFlow 自体はデータベースを内蔵しないが、以下のデータ永続化メカニズムを持つ。

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| ファイル形式 | TFRecord (Protocol Buffers) | - | 学習データの効率的な読み書き |
| ファイル形式 | SavedModel (Protocol Buffers + 変数ファイル) | - | モデルの永続化・配布 |
| ファイル形式 | Checkpoint (.ckpt, .index, .data) | - | 学習途中の状態保存・復元 |
| KV ストア | LevelDB / LMDB (tf.data 経由) | - | 一部データセット読み込みに利用 |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| tf.data キャッシュ (メモリ/ディスク) | - | データパイプラインのキャッシュ (`.cache()`) | エポック単位 / 明示的クリアまで |
| BFC (Best-Fit with Coalescing) アロケータ | - | GPU/CPU メモリのプール管理・再利用 | セッションライフタイム |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| gRPC | - | 分散学習におけるワーカー間通信、パラメータサーバー通信 |
| Rendezvous (core/distributed_runtime) | - | 分散テンソル送受信の抽象化レイヤー |
| Collective Operations | - | AllReduce, AllGather 等の集約通信 |

### 検索エンジン

該当なし。TensorFlow は検索エンジンを使用しない。

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Protocol Buffers (protobuf) | 3.x | グラフ定義、Op 定義、設定ファイルのシリアライズ |
| FlatBuffers | - | TensorFlow Lite モデルファイル形式 |
| Bazel | - | ビルドシステム |
| CUDA / cuDNN | 対応バージョンは configure 時に指定 | NVIDIA GPU カーネル実行 |
| ROCm | - | AMD GPU サポート |
| NCCL | - | マルチ GPU 集約通信 |
| Eigen | 3.x | CPU 上の線形代数演算 |
| oneDNN (MKL-DNN) | - | Intel CPU 向け最適化カーネル |

## データフロー

### リクエスト処理の流れ

TensorFlow 2.x における Eager Execution モードでの Op 実行フローを以下に示す。

1. **Python API 呼び出し**: ユーザーが `tf.matmul(a, b)` などの Python API を呼び出す。
2. **Eager ディスパッチ**: `tensorflow/python/eager/execute.py` が Op を即座に実行するために C API を呼び出す。
3. **C API 経由**: `tensorflow/c/eager/` の Eager API を通じて Core Runtime にディスパッチされる。
4. **デバイス選択**: `common_runtime` のデバイスマネージャが適切なデバイス (CPU/GPU/TPU) を選択する。
5. **カーネル実行**: 選択されたデバイスに対応する Op カーネル (`core/kernels/`) が実行される。
6. **結果返却**: 計算結果がテンソルとして Python 層に返却される。

Graph Execution モード (`tf.function` デコレータ使用時) では以下の追加ステップが加わる。

1. **トレーシング**: `tf.function` が Python コードをトレースし、`FuncGraph` を構築する。
2. **Grappler 最適化**: `core/grappler/` の最適化パス（定数畳み込み、演算融合、レイアウト最適化等）が適用される。
3. **XLA コンパイル（オプション）**: `compiler/jit/` により XLA クラスタが特定され、XLA でコンパイルされる。
4. **エグゼキュータ実行**: 最適化されたグラフが `common_runtime` のエグゼキュータにより実行される。

### 非同期処理の流れ

1. **tf.data パイプライン**: `dataset.prefetch()` によりデータ読み込みと前処理がバックグラウンドスレッドで非同期に実行される。
2. **非同期 Op 実行**: GPU Op は CUDA ストリームに非同期にエンキューされ、ホスト CPU は次の Op のディスパッチを続行する。
3. **分散学習の非同期通信**: `DistributedRuntime` が Rendezvous プロトコルを用いてテンソルの非同期送受信を行う。
4. **Collective 操作**: AllReduce 等の勾配集約が非同期にスケジューリングされる。

### データ永続化の流れ

1. **Checkpoint 保存**: `tf.train.Checkpoint.save()` が呼ばれると、追跡対象の変数がシリアライズされ、`.index` と `.data` ファイルに書き出される。
2. **SavedModel エクスポート**: `tf.saved_model.save()` により、グラフ定義 (GraphDef)、変数値、メタデータが `saved_model.pb` および `variables/` ディレクトリに保存される。
3. **TFLite 変換**: `tf.lite.TFLiteConverter` が SavedModel を FlatBuffers 形式の `.tflite` モデルに変換する。
4. **TFRecord 書き込み**: `tf.io.TFRecordWriter` が Protocol Buffers でエンコードされたデータを TFRecord ファイルに書き出す。

## 横断的関心事

### 認証・認可

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| gRPC TLS/SSL | `core/distributed_runtime/` | 分散学習時のワーカー間通信の暗号化 |
| クラウドストレージ認証 (GCS, S3) | `core/platform/cloud/` | リモートファイルシステムアクセス時の認証 |
| TF Serving ACL | (外部コンポーネント) | モデルサービング時のアクセス制御 |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | `tensorflow/core/platform/` の logging 機構 (VLOG, LOG) | 標準出力 / 標準エラー出力 |
| プロファイリングログ | `core/profiler/` + TensorBoard | イベントファイル (logdir) |
| Summary / メトリクス | `tf.summary` API | TensorBoard イベントファイル |
| Op 実行トレース | `tf.debugging.enable_traceback_filtering()` | メモリ内 |

### エラーハンドリング

TensorFlow は Status ベースのエラーハンドリングを C++ 層で採用し、Python 層では例外に変換する。

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| Op 実行エラー | C++ Status -> Python 例外変換 (`tensorflow/python/framework/errors.py`) | `tf.errors.OpError` 系例外 (InvalidArgumentError, ResourceExhaustedError 等) |
| グラフ構築エラー | Python 例外 | `ValueError`, `TypeError` 等 |
| デバイスエラー (OOM等) | Status コード `RESOURCE_EXHAUSTED` | `tf.errors.ResourceExhaustedError` |
| 分散通信エラー | gRPC Status コード | `tf.errors.UnavailableError`, リトライ機構 |
| 数値エラー (NaN, Inf) | `tf.debugging.check_numerics` / デバッグモード | 検出時に `InvalidArgumentError` |

### トランザクション管理

TensorFlow は従来のデータベーストランザクションを持たないが、以下の一貫性保証メカニズムがある。

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| Checkpoint 保存 | アトミック書き込み (一時ファイル -> リネーム) | ファイルレベルのアトミシティ |
| SavedModel 保存 | ディレクトリ単位のアトミック書き込み | ファイルシステム依存 |
| 変数更新 (分散学習) | Collective 同期 / パラメータサーバー | 同期 SGD: 強一貫性、非同期 SGD: 結果整合性 |
| tf.Variable 代入 | `assign()` 操作の排他制御 | Op レベルのアトミシティ |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| レイヤード分離 | 全体構成 | Python / C API / Core Runtime / Platform の明確な層分離 |
| デバイス抽象化 (Strategy パターン) | デバイス管理 | `Device` 基底クラスを CPU/GPU/TPU が継承、`tf.distribute.Strategy` |
| Op レジストリ (Registry パターン) | Op カーネル管理 | `REGISTER_OP`, `REGISTER_KERNEL_BUILDER` マクロによるカーネル登録 |
| データフローグラフ | 計算モデル | `Graph`, `Node`, `Edge` による有向非巡回グラフの計算表現 |
| Eager by Default | 実行モデル | TF2.x では Eager Execution がデフォルト、`tf.function` でグラフモードに移行 |
| Protocol Buffers によるスキーマ定義 | データ形式 | `GraphDef`, `OpDef`, `SavedModel` 等すべて protobuf で定義 |
| C API による ABI 安定性 | 言語バインディング | `c_api.h` が安定した公開 API を提供し、内部実装変更の影響を隔離 |

### コーディング規約

- **C++ コード**: Google C++ Style Guide に準拠。`tensorflow/core/` 以下のコードは `namespace tensorflow` で統一。
- **Python コード**: Google Python Style Guide に準拠。型ヒントの積極的利用。`tf_export` デコレータによる公開 API 管理。
- **ビルドシステム**: Bazel を使用。各ディレクトリに `BUILD` ファイルを配置し、依存関係を明示的に管理。
- **テスト**: `*_test.cc` (C++) / `*_test.py` (Python) の命名規則。`tf.test.TestCase` を基底クラスとした単体テスト。
- **API バージョニング**: `tf_export` デコレータと `compat.v1` / `compat.v2` モジュールによる API 互換性管理。

## 備考

- TensorFlow は非常に大規模なコードベースであり、本書では主要なアーキテクチャ構成要素に焦点を当てている。
- XLA の内部アーキテクチャ（HLO IR、各種バックエンド）は別途詳細設計が必要である。
- Keras API は TensorFlow 2.x で密結合されていたが、今後のバージョンでは独立パッケージ (keras 3.x) への移行が進んでいる。
- DTensor は比較的新しいコンポーネントであり、分散テンソル計算の統一的な抽象化を提供する。
