# TensorFlow コードリーディングガイドライン

## はじめに

このガイドラインは、TensorFlow（バージョン2.21.0）のコードベースを効率的に理解するための手引きです。
C++やPythonに精通していないエンジニアでも、段階的に学習できるよう構成されています。

TensorFlowは、Googleが開発したエンドツーエンドのオープンソース機械学習プラットフォームです。主にC++とPythonで実装されており、Bazelビルドシステムを採用しています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、TensorFlowで使用されるC++とPythonの基本構文と概念を解説します。

TensorFlowは主に以下の言語で構成されています。

| 言語 | 用途 | 主要ディレクトリ |
|------|------|-----------------|
| C++ | コアランタイム、カーネル実装、フレームワーク | `tensorflow/core/`, `tensorflow/cc/` |
| Python | 公開API、高レベルインターフェース、テスト | `tensorflow/python/` |
| Starlark (Bazel) | ビルド定義 | 各ディレクトリの`BUILD`ファイル、`.bzl`ファイル |
| Protocol Buffers | データシリアライゼーション定義 | `tensorflow/core/framework/*.proto` |

### 1.1 プログラム構造

#### C++（コアランタイム）

TensorFlowのC++コードは名前空間 `tensorflow` 内に配置されます。典型的なC++ファイルの構造は以下の通りです。

```cpp
// ファイル: tensorflow/core/kernels/aggregate_ops.cc:1-35
/* Copyright 2015 The TensorFlow Authors. All Rights Reserved.
...
==============================================================================*/

// See docs in ../ops/math_ops.cc.

#define EIGEN_USE_THREADS

#include "tensorflow/core/kernels/aggregate_ops.h"

#include "tensorflow/core/framework/op_kernel.h"
#include "tensorflow/core/framework/register_types.h"
#include "tensorflow/core/framework/tensor.h"

namespace tensorflow {

typedef Eigen::ThreadPoolDevice CPUDevice;
typedef Eigen::GpuDevice GPUDevice;

template <typename Device, typename T>
class AddNOp : public OpKernel {
 public:
  explicit AddNOp(OpKernelConstruction* context) : OpKernel(context) {}

  void Compute(OpKernelContext* ctx) override {
    // ...
  }
};
```

重要なポイント:
- すべてのコードは `namespace tensorflow` 内に配置される
- ライセンスヘッダーがファイル先頭に必須
- インクルードパスはリポジトリルートからの絶対パスを使用（例: `"tensorflow/core/framework/tensor.h"`）

#### Python（公開API）

Python側は、`tensorflow.python` パッケージ配下に構成されます。

```python
# ファイル: tensorflow/python/ops/array_ops.py:16-55
"""Support for manipulating tensors."""

import numbers
import numpy as np

from tensorflow.python.eager import context
from tensorflow.python.framework import ops
from tensorflow.python.framework import dtypes
from tensorflow.python.framework import tensor_shape
from tensorflow.python.util.tf_export import tf_export
```

重要なポイント:
- モジュールの先頭にdocstringが記述される
- `tf_export` デコレータで公開APIが定義される
- インポートは `tensorflow.python.*` の形式で階層的に行われる

### 1.2 データ型と変数

#### C++のデータ型

TensorFlowのC++コードでは、独自の型エイリアスとProtocol Buffersで定義された型が使用されます。

```cpp
// ファイル: tensorflow/core/framework/tensor.h:16-37
#include "tensorflow/core/framework/tensor_shape.h"
#include "tensorflow/core/framework/tensor_types.h"
#include "tensorflow/core/framework/types.h"
#include "tensorflow/core/framework/types.pb.h"
```

主要なデータ型:
- `Tensor` - 多次元配列を表現するコアデータ型
- `TensorShape` - テンソルの形状を表す型
- `DataType` (enum) - `DT_FLOAT`, `DT_INT32` などのデータ型列挙
- `OpKernel` - オペレーションのカーネル実装の基底クラス
- `absl::Status` - エラーハンドリング用のステータス型

#### Pythonのデータ型

```python
# ファイル: tensorflow/python/framework/ops.py:33-40
from tensorflow.core.framework import attr_value_pb2
from tensorflow.core.framework import function_pb2
from tensorflow.core.framework import graph_pb2
from tensorflow.core.framework import node_def_pb2
from tensorflow.core.framework import op_def_pb2
from tensorflow.core.framework import types_pb2
```

主要なPythonクラス:
- `tf.Tensor` - テンソルオブジェクト
- `tf.Operation` - 計算グラフ内のオペレーション
- `tf.Graph` - 計算グラフ全体
- `tf.dtypes.DType` - データ型

### 1.3 制御構造

#### C++のエラーハンドリングパターン

TensorFlowのC++コードでは、`absl::Status` を使ったエラーハンドリングが標準です。

```cpp
// ファイル: tensorflow/core/ops/array_ops.cc:46-56
absl::Status GetAxisForPackAndUnpack(InferenceContext* c,
                                     int32_t rank_after_pack, int32_t* axis) {
  TF_RETURN_IF_ERROR(c->GetAttr("axis", axis));
  if (*axis < -1 * rank_after_pack || *axis >= rank_after_pack) {
    return errors::InvalidArgument("Invalid axis: ", *axis, "; must be in [",
                                   -1 * rank_after_pack, ",", rank_after_pack,
                                   ")");
  }
  if (*axis < 0) *axis = (rank_after_pack + *axis);
  return absl::OkStatus();
}
```

重要なマクロ:
- `TF_RETURN_IF_ERROR(expr)` - エラーの場合は即座にリターン
- `OP_REQUIRES(ctx, condition, error)` - カーネル内のアサーション
- `OP_REQUIRES_OK(ctx, expr)` - Status式のチェック

#### Pythonの実行モード

TensorFlowにはEager実行モードとGraph実行モードの2つの実行モードがあります。

```python
# ファイル: tensorflow/python/eager/context.py:15
"""State management for eager execution."""
```

### 1.4 関数/メソッド定義

#### C++カーネルの定義パターン

```cpp
// ファイル: tensorflow/core/kernels/aggregate_ops.cc:37-42
template <typename Device, typename T>
class AddNOp : public OpKernel {
 public:
  explicit AddNOp(OpKernelConstruction* context) : OpKernel(context) {}

  void Compute(OpKernelContext* ctx) override {
    // カーネル実装
  }
};
```

カーネル定義の要素:
1. `OpKernel` を継承するクラスを定義
2. コンストラクタで `OpKernelConstruction` を受け取る
3. `Compute` メソッドをオーバーライドして実装

#### PythonのAPI定義パターン

```python
# tensorflow/python/ops/array_ops.py のパターン
from tensorflow.python.util.tf_export import tf_export

@tf_export("reshape")
def reshape(tensor, shape, name=None):
    """Reshapes a tensor."""
    # 実装
```

### 1.5 モジュール/インポート

#### C++のインクルード規則

```cpp
// ファイル: tensorflow/core/kernels/aggregate_ops.cc:20-29
#include "tensorflow/core/kernels/aggregate_ops.h"   // 自身のヘッダ

#include "tensorflow/core/framework/op_kernel.h"      // フレームワーク
#include "tensorflow/core/framework/register_types.h"
#include "tensorflow/core/framework/tensor.h"
#include "tensorflow/core/kernels/aggregate_ops_cpu.h" // 同パッケージ
#include "tensorflow/core/lib/gtl/inlined_vector.h"    // ライブラリ
```

インクルード順序の規則:
1. 自身のヘッダファイル
2. TensorFlowフレームワークのヘッダ
3. 同パッケージのヘッダ
4. ライブラリヘッダ

#### Pythonのインポート規則

```python
# ファイル: tensorflow/python/ops/array_ops.py:18-55
import numbers
import numpy as np                                        # 外部ライブラリ

from tensorflow.python.eager import context               # TF内部モジュール
from tensorflow.python.framework import ops
from tensorflow.python.util.tf_export import tf_export    # ユーティリティ
```

---

## 2. プロジェクト固有の概念

> このセクションでは、TensorFlow特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### Op（オペレーション）の定義と登録

TensorFlowの計算は「Op」として定義されます。Opの定義は、インターフェース定義（C++）とカーネル実装（C++）、Python バインディングの3層で構成されます。

**Op定義**（`tensorflow/core/ops/`）:

```cpp
// ファイル: tensorflow/core/ops/array_ops.cc:21-31
#include "tensorflow/core/framework/common_shape_fns.h"
#include "tensorflow/core/framework/op.h"
#include "tensorflow/core/framework/shape_inference.h"
```

Op定義ファイルでは、`REGISTER_OP` マクロを使用してオペレーションのインターフェース（入力・出力・属性・形状推論）を宣言します。

**カーネル実装**（`tensorflow/core/kernels/`）:

```cpp
// ファイル: tensorflow/core/kernels/aggregate_ops.cc:37-50
template <typename Device, typename T>
class AddNOp : public OpKernel {
 public:
  explicit AddNOp(OpKernelConstruction* context) : OpKernel(context) {}
  void Compute(OpKernelContext* ctx) override {
    if (!ctx->ValidateInputsAreSameShape(this)) return;
    const Tensor& input0 = ctx->input(0);
    const int num = ctx->num_inputs();
    if (num == 1) {
      ctx->set_output(0, input0);
      return;
    }
    // ...
  }
};
```

**Pythonバインディング**（`tensorflow/python/ops/`）:

```python
# ファイル: tensorflow/python/ops/array_ops.py:40-48
from tensorflow.python.ops import gen_array_ops
from tensorflow.python.ops.gen_array_ops import *
```

`gen_*` ファイルはC++ Opから自動生成されるPythonラッパーです。

#### Tensor（テンソル）

```cpp
// ファイル: tensorflow/core/framework/tensor.h:69-76
/// Interface to access the raw ref-counted data buffer.
class TensorBuffer : public core::RefCounted {
 public:
  explicit TensorBuffer(void* data_ptr) : data_(data_ptr) {}
  ~TensorBuffer() override {}
  /// \brief data() points to a memory region of size() bytes.
};
```

テンソルはTensorFlowの基本データ構造であり、参照カウント方式のメモリ管理が行われます。

#### Executor（実行器）

```cpp
// ファイル: tensorflow/core/common_runtime/executor.h:43-50
// Executor runs a graph computation.
// Example:
//   Graph* graph = ...;
//      ... construct graph ...
//   Executor* executor;
//   TF_CHECK_OK(NewSimpleExecutor(my_device, graph, &executor));
//   Rendezvous* rendezvous = NewNaiveRendezvous();
//   TF_CHECK_OK(rendezvous->Send("input", some_input_tensor));
```

Executorはグラフの実行を担当するコンポーネントです。

### 2.2 プロジェクト独自のパターン

#### 自動コード生成パターン

TensorFlowでは、C++のOp定義からPythonのラッパーコードが自動生成されます。

- `gen_*` プレフィックスのPythonファイル: C++ Opから自動生成
- `*_grad.py` ファイル: 勾配計算の定義
- `*.proto` ファイル: Protocol Buffers定義から `*_pb2.py` が自動生成

#### Bazelビルドシステム

```python
# ファイル: tensorflow/tensorflow.bzl:1-2
"""Provides build configuration for TensorFlow."""
load("@rules_cc//cc:cc_library.bzl", _cc_library = "cc_library")
```

すべてのビルドはBazelで管理され、`BUILD` ファイルと `.bzl` マクロファイルで定義されます。

#### デバイス抽象化パターン

```cpp
// ファイル: tensorflow/core/kernels/aggregate_ops.cc:34-35
typedef Eigen::ThreadPoolDevice CPUDevice;
typedef Eigen::GpuDevice GPUDevice;
```

カーネルはデバイス型（CPU/GPU）をテンプレートパラメータとして受け取り、同一コードで複数デバイスに対応します。

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `*_ops.cc` / `*_ops.py` | オペレーション定義/バインディング | `array_ops.cc`, `math_ops.py` |
| `*_op.cc` / `*_op.h` | 個別カーネル実装 | `aggregate_ops.cc`, `argmax_op.cc` |
| `*_test.cc` / `*_test.py` | テストファイル | `as_string_op_test.cc`, `array_ops_test.py` |
| `*.proto` | Protocol Buffers定義 | `tensor.proto`, `graph.proto` |
| `*.pb.h` / `*_pb2.py` | protobuf自動生成コード | `types.pb.h`, `attr_value_pb2` |
| `gen_*.py` | 自動生成Pythonバインディング | `gen_array_ops.py` |
| `*_grad.py` / `*_grad.cc` | 勾配計算の実装 | `array_grad.py`, `array_grad.cc` |
| `*.gpu.cu.cc` | CUDA GPU実装 | `aggregate_ops_gpu.cu.cc` |
| `*_cpu.h` | CPU特化実装 | `aggregate_ops_cpu.h` |
| `BUILD` | Bazelビルド定義 | 各ディレクトリに存在 |
| `*.bzl` | Bazelマクロ/ルール | `tensorflow.bzl` |
| `_pywrap_*.pyi` | C++バインディング型スタブ | `_pywrap_tfe.pyi` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `Op` (サフィックス, C++) | OpKernel実装クラス | `AddNOp`, `ArgMaxOp` |
| `OpKernel` | カーネル基底クラス | `OpKernel` |
| `tf_export` (Python デコレータ) | 公開API宣言 | `@tf_export("reshape")` |
| `TF_` (プレフィックス, C マクロ) | TensorFlowマクロ | `TF_RETURN_IF_ERROR` |
| `REGISTER_OP` | Op登録マクロ | `REGISTER_OP("Pack")` |
| `REGISTER_KERNEL_BUILDER` | カーネル登録マクロ | `REGISTER_KERNEL_BUILDER(...)` |
| `k` (プレフィックス, C++) | 定数 | `kDefaultCapacity` |
| `_` (プレフィックス, Python) | 内部関数/変数 | `_DEFAULT_CUDA_COMPUTE_CAPABILITIES` |
| `Handle` (サフィックス, C++) | ハンドル型 | `DimensionHandle`, `ShapeHandle` |

### 3.3 プログラム分類一覧

| 分類 | 説明 | ディレクトリ |
|------|------|-------------|
| Op定義 | オペレーションのインターフェース定義 | `tensorflow/core/ops/` |
| カーネル | Opの実際の計算実装 | `tensorflow/core/kernels/` |
| フレームワーク | コアデータ構造とインフラ | `tensorflow/core/framework/` |
| ランタイム | 実行エンジン | `tensorflow/core/common_runtime/` |
| Python API | ユーザー向けPythonインターフェース | `tensorflow/python/` |
| コンパイラ | XLA, MLIRベースの最適化 | `tensorflow/compiler/` |
| Lite | モバイル/組み込み向け軽量版 | `tensorflow/lite/` |
| 分散処理 | 分散学習サポート | `tensorflow/python/distribute/` |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
tensorflow-master/
├── tensorflow/               # メインソースコード
│   ├── core/                 # C++コアライブラリ
│   │   ├── framework/        #   コアデータ構造(Tensor, Op, Graph)
│   │   ├── kernels/          #   Op実装(CPU/GPU)
│   │   ├── ops/              #   Op定義(インターフェース)
│   │   ├── common_runtime/   #   実行エンジン
│   │   ├── distributed_runtime/ # 分散実行
│   │   ├── graph/            #   グラフ操作
│   │   ├── grappler/         #   グラフ最適化
│   │   ├── lib/              #   低レベルライブラリ
│   │   ├── platform/         #   プラットフォーム抽象化
│   │   ├── protobuf/         #   Protocol Buffers定義
│   │   └── public/           #   公開C++ API
│   ├── python/               # PythonフロントエンドAPI
│   │   ├── framework/        #   Python側のコアフレームワーク
│   │   ├── ops/              #   Python Op実装/バインディング
│   │   ├── eager/            #   Eager実行モード
│   │   ├── keras/            #   Keras高レベルAPI
│   │   ├── data/             #   tf.data入力パイプライン
│   │   ├── distribute/       #   分散戦略
│   │   ├── training/         #   トレーニングユーティリティ
│   │   ├── compiler/         #   XLAコンパイラバインディング
│   │   └── autograph/        #   PythonからGraphへの変換
│   ├── cc/                   # C++ クライアントAPI
│   │   ├── framework/        #   C++フレームワーク
│   │   ├── ops/              #   C++ Op
│   │   ├── gradients/        #   C++勾配
│   │   └── saved_model/      #   SavedModelサポート
│   ├── compiler/             # コンパイラスタック
│   │   ├── mlir/             #   MLIR方言・パス
│   │   ├── tf2xla/           #   TF->XLA変換
│   │   ├── jit/              #   JITコンパイル
│   │   └── aot/              #   Ahead-of-Timeコンパイル
│   ├── lite/                 # TensorFlow Lite
│   ├── dtensor/              # DTensor(分散テンソル)
│   ├── java/                 # Javaバインディング
│   ├── go/                   # Goバインディング
│   └── js/                   # JavaScriptバインディング
├── third_party/              # サードパーティ依存関係
├── tools/                    # ビルド・開発ツール
├── ci/                       # CI/CD設定
├── configure.py              # ビルド設定スクリプト
├── WORKSPACE                 # Bazelワークスペース定義
└── BUILD                     # ルートBUILDファイル
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `tensorflow/core/framework/` | テンソル、Op、グラフなどのコアデータ構造 | `tensor.h`, `op_kernel.h`, `types.h` |
| `tensorflow/core/kernels/` | 各Opのデバイス別計算実装 | `*_op.cc`, `*_op.h`, `*.cu.cc` |
| `tensorflow/core/ops/` | Opのインターフェース定義と形状推論 | `*_ops.cc` |
| `tensorflow/core/common_runtime/` | グラフ実行エンジン | `executor.h`, `device.h` |
| `tensorflow/core/graph/` | 計算グラフの構築と操作 | `graph.h` |
| `tensorflow/core/grappler/` | グラフレベルの最適化パス | `optimizers/` |
| `tensorflow/python/framework/` | PythonのGraph、Op、Tensor抽象 | `ops.py`, `dtypes.py` |
| `tensorflow/python/ops/` | PythonのOp関数群 | `array_ops.py`, `math_ops.py` |
| `tensorflow/python/eager/` | Eager実行管理 | `context.py`, `execute.py` |
| `tensorflow/compiler/mlir/` | MLIR方言とコンパイルパス | `README.md` |
| `tensorflow/compiler/tf2xla/` | TFグラフからXLAへの変換 | `*.cc`, `*.h` |
| `tensorflow/lite/` | モバイル/エッジデバイス向け | `interpreter.h` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

TensorFlowはレイヤードアーキテクチャを採用しており、上位層から下位層へ依存する形で構成されています。

```
┌──────────────────────────────────────────────────────┐
│         ユーザーAPI層 (Python / C++ / Java / Go)       │
│   tf.keras, tf.data, tf.distribute, tf.saved_model   │
├──────────────────────────────────────────────────────┤
│              Python フロントエンド層                     │
│   tensorflow/python/ops, framework, eager             │
├──────────────────────────────────────────────────────┤
│              コンパイラ層                               │
│   XLA, MLIR, Grappler (グラフ最適化)                    │
├──────────────────────────────────────────────────────┤
│              C++ ランタイム層                           │
│   Executor, Session, Device Management                │
├──────────────────────────────────────────────────────┤
│              コアフレームワーク層                        │
│   Tensor, OpKernel, Graph, Shape Inference            │
├──────────────────────────────────────────────────────┤
│              カーネル層                                 │
│   CPU Kernels, GPU Kernels (CUDA), TPU Kernels        │
├──────────────────────────────────────────────────────┤
│              プラットフォーム抽象化層                     │
│   OS, Threading, Memory, Eigen, cuDNN, oneDNN         │
└──────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| ユーザーAPI | 高レベルAPIの提供 | `tensorflow/python/keras/`, `tensorflow/python/data/` |
| Pythonフロントエンド | Op呼び出し、テンソル操作 | `tensorflow/python/ops/array_ops.py`, `tensorflow/python/framework/ops.py` |
| コンパイラ | グラフ最適化、XLA変換 | `tensorflow/compiler/mlir/`, `tensorflow/core/grappler/` |
| ランタイム | グラフ実行、デバイス管理 | `tensorflow/core/common_runtime/executor.h` |
| コアフレームワーク | データ構造、Op定義 | `tensorflow/core/framework/tensor.h`, `tensorflow/core/framework/op_kernel.h` |
| カーネル | 計算の実装 | `tensorflow/core/kernels/aggregate_ops.cc` |
| プラットフォーム | OS・ハードウェア抽象化 | `tensorflow/core/platform/`, `tensorflow/core/lib/` |

### 5.3 データフロー

TensorFlowにおけるデータフローは以下の流れで進行します。

1. **Python API呼び出し**: ユーザーが `tf.matmul(a, b)` などを呼び出す
2. **Op解決**: Eager実行の場合は即座に、Graph実行の場合はグラフに追加
3. **ディスパッチ**: Python -> C++バインディング経由でカーネル選択
4. **カーネル実行**: デバイス（CPU/GPU/TPU）に応じたカーネルが実行
5. **結果返却**: Tensorオブジェクトとして結果がPythonに返される

```
ユーザーコード (Python)
    │
    ▼
tf.matmul(a, b)  [tensorflow/python/ops/math_ops.py]
    │
    ▼
gen_math_ops.mat_mul(...)  [自動生成コード]
    │
    ▼
C++ Op Dispatch  [tensorflow/core/common_runtime/]
    │
    ▼
MatMulOp::Compute()  [tensorflow/core/kernels/matmul_op.cc]
    │
    ▼
Eigen/cuBLAS/oneDNN  [プラットフォームライブラリ]
```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

#### Pythonパッケージのエントリーポイント

```python
# ファイル: tensorflow/__init__.py:20-24
from tensorflow.python import pywrap_tensorflow  # pylint: disable=unused-import

from tensorflow.python.platform import flags
from tensorflow.python.platform import app
app.flags = flags
```

`tensorflow/__init__.py` がパッケージのエントリーポイントです。ここから `pywrap_tensorflow`（C++バインディング）がロードされ、TensorFlowランタイムが初期化されます。

#### ビルド設定のエントリーポイント

```python
# ファイル: configure.py:15
"""configure script to get build parameters from user."""
```

ビルド設定は `configure.py` から開始され、CUDA、cuDNN、NCCL等のパスを設定します。

### 6.2 ビジネスロジック

TensorFlowでは「ビジネスロジック」はOp（オペレーション）として実装されます。

**Op定義（インターフェース）**: `tensorflow/core/ops/` 内の `*_ops.cc` ファイル

```cpp
// ファイル: tensorflow/core/ops/array_ops.cc:37-43
namespace tensorflow {
using shape_inference::DimensionHandle;
using shape_inference::InferenceContext;
using shape_inference::ShapeHandle;
using shape_inference::UnchangedShape;
```

**カーネル実装（計算処理）**: `tensorflow/core/kernels/` 内の `*_op.cc` ファイル

```cpp
// ファイル: tensorflow/core/kernels/aggregate_ops.cc:38-42
class AddNOp : public OpKernel {
 public:
  explicit AddNOp(OpKernelConstruction* context) : OpKernel(context) {}
  void Compute(OpKernelContext* ctx) override {
    // 計算実装
  }
};
```

### 6.3 データアクセス

TensorFlowにおけるデータアクセスは主に以下のコンポーネントで管理されます。

**tf.data パイプライン**: `tensorflow/python/data/`

データの読み込み、変換、バッチ処理を行う入力パイプラインフレームワークです。

**Protocol Buffers**: `tensorflow/core/framework/*.proto`

```
tensorflow/core/framework/
├── tensor.proto              # テンソルのシリアライズ形式
├── graph.proto               # 計算グラフの形式
├── op_def.proto              # Op定義の形式
├── attr_value.proto          # 属性値の形式
└── types.proto               # データ型定義
```

**SavedModel**: `tensorflow/cc/saved_model/`, `tensorflow/python/saved_model/`

モデルの永続化フォーマットです。

### 6.4 ユーティリティ/共通機能

| ユーティリティ | 場所 | 用途 |
|--------------|------|------|
| `tensorflow/core/lib/` | C++基盤ライブラリ | GTL(Google Template Library)、文字列処理 |
| `tensorflow/core/platform/` | プラットフォーム抽象 | ロギング、ファイルI/O、スレッド |
| `tensorflow/python/util/` | Pythonユーティリティ | デコレータ、互換性、deprecation |
| `tensorflow/core/util/` | C++ユーティリティ | テンソルフォーマット、パディング |

```python
# ファイル: tensorflow/python/ops/array_ops.py:50-55
from tensorflow.python.util import deprecation
from tensorflow.python.util import dispatch
from tensorflow.python.util import nest
from tensorflow.python.util import tf_decorator
from tensorflow.python.util.tf_export import tf_export
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| Op定義-カーネル分離 | インターフェースと実装の分離 | 高 | `core/ops/`, `core/kernels/` |
| テンプレートによるデバイス抽象 | CPU/GPUの統一実装 | 高 | `core/kernels/*.cc` |
| REGISTER マクロ | Opやカーネルの登録 | 高 | `core/ops/`, `core/kernels/` |
| tf_export デコレータ | 公開APIの宣言 | 高 | `python/ops/*.py` |
| 自動コード生成 | C++からPythonラッパー生成 | 高 | `python/ops/gen_*.py` |
| Status ベースエラー処理 | エラーの伝播 | 高 | `core/**/*.cc` |
| Shape Inference | 出力形状の推論 | 中 | `core/ops/*.cc` |
| Protocol Buffers | データシリアライゼーション | 中 | `core/framework/*.proto` |

### 各パターンの詳細

#### パターン1: Op定義-カーネル分離

**目的:** Opのインターフェース（型シグネチャ）と実装（計算ロジック）を分離し、同一Opに対して複数のカーネル（CPU/GPU/TPU）を登録可能にする。

**Op定義側:**
```cpp
// ファイル: tensorflow/core/ops/array_ops.cc
// REGISTER_OP("Reshape")
//   .Input("tensor: T")
//   .Input("shape: Tshape")
//   .Output("output: T")
//   .Attr("T: type")
//   .Attr("Tshape: {int32, int64} = DT_INT32")
//   .SetShapeFn(...);
```

**カーネル実装側:**
```cpp
// ファイル: tensorflow/core/kernels/aggregate_ops.cc:37-42
template <typename Device, typename T>
class AddNOp : public OpKernel {
 public:
  explicit AddNOp(OpKernelConstruction* context) : OpKernel(context) {}
  void Compute(OpKernelContext* ctx) override { /* 実装 */ }
};
```

**解説:** Op定義ファイルでインターフェースを宣言し、kernelsディレクトリ内の別ファイルでデバイス別の実装を提供します。`REGISTER_KERNEL_BUILDER` マクロでOpとカーネルを関連付けます。

#### パターン2: テンプレートによるデバイス抽象化

**目的:** 一つのカーネルクラスで複数デバイスをサポートする。

**実装例:**
```cpp
// ファイル: tensorflow/core/kernels/aggregate_ops.cc:34-35
typedef Eigen::ThreadPoolDevice CPUDevice;
typedef Eigen::GpuDevice GPUDevice;

template <typename Device, typename T>
class AddNOp : public OpKernel { /* ... */ };
```

**解説:** C++テンプレートでデバイス型をパラメータ化し、CPU版とGPU版を同一コードから生成します。GPU固有の実装は `*.gpu.cu.cc` ファイルに分離されます。

#### パターン3: Status ベースのエラー処理

**目的:** エラーを例外ではなく戻り値で伝播する。

**実装例:**
```cpp
// ファイル: tensorflow/core/ops/array_ops.cc:46-56
absl::Status GetAxisForPackAndUnpack(InferenceContext* c,
                                     int32_t rank_after_pack, int32_t* axis) {
  TF_RETURN_IF_ERROR(c->GetAttr("axis", axis));
  if (*axis < -1 * rank_after_pack || *axis >= rank_after_pack) {
    return errors::InvalidArgument("Invalid axis: ", *axis);
  }
  return absl::OkStatus();
}
```

**解説:** `absl::Status` を返り値にし、`TF_RETURN_IF_ERROR` マクロでエラーを自動伝播します。C++の例外を使わないのはパフォーマンス上の理由です。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定（Python API呼び出し）
2. 処理の流れを追跡（Python -> C++バインディング -> カーネル）
3. データの変換を確認（Tensor入力 -> 計算 -> Tensor出力）
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: テンソル演算の実行フロー（AddN Op）

**概要:** 複数のテンソルを要素ごとに加算する `tf.add_n()` の処理フローを追跡します。

**処理フロー:**
```
tf.add_n() → gen_math_ops.add_n() → C++ Dispatch → AddNOp::Compute() → Eigen
```

**詳細な追跡:**

1. **Python API呼び出し** (`tensorflow/python/ops/math_ops.py`)
   ```python
   # ユーザーが tf.add_n([tensor1, tensor2, tensor3]) を呼び出す
   ```

2. **自動生成コードへの委譲** (`tensorflow/python/ops/gen_math_ops.py`)
   ```python
   # gen_math_ops.add_n() が呼ばれる（自動生成コード）
   ```

3. **C++カーネル実行** (`tensorflow/core/kernels/aggregate_ops.cc:38-50`)
   ```cpp
   template <typename Device, typename T>
   class AddNOp : public OpKernel {
    public:
     explicit AddNOp(OpKernelConstruction* context) : OpKernel(context) {}
     void Compute(OpKernelContext* ctx) override {
       if (!ctx->ValidateInputsAreSameShape(this)) return;
       const Tensor& input0 = ctx->input(0);
       const int num = ctx->num_inputs();
       if (num == 1) {
         ctx->set_output(0, input0);
         return;
       }
       // 複数テンソルを加算
     }
   };
   ```

4. **Op定義の確認** (`tensorflow/core/ops/math_ops.cc`)
   ```cpp
   // AddN Opのインターフェース定義（入力・出力・属性・形状推論）
   ```

#### 例2: 新しいOpの追加フロー

**概要:** TensorFlowに新しいOpを追加する際の実装フローです。

**処理フロー:**
```
Op定義(core/ops/) → カーネル実装(core/kernels/) → Python登録(python/ops/) → テスト
```

**詳細な追跡:**

1. **Op定義** (`tensorflow/core/ops/` に新しい `*_ops.cc`)
   - `REGISTER_OP` マクロでインターフェースを宣言
   - 入力、出力、属性、形状推論関数を定義

2. **カーネル実装** (`tensorflow/core/kernels/` に新しい `*_op.cc`)
   - `OpKernel` を継承するクラスを定義
   - `Compute` メソッドを実装
   - `REGISTER_KERNEL_BUILDER` でカーネルを登録

3. **Pythonバインディング** (`tensorflow/python/ops/`)
   - 自動生成の `gen_*_ops.py` が生成される
   - 必要に応じて手動のPythonラッパーを追加

4. **テスト** (`tensorflow/python/kernel_tests/` または `tensorflow/core/kernels/*_test.cc`)

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイント（Python API）を特定したか
- [ ] 自動生成コード（`gen_*`）の存在を確認したか
- [ ] C++のOp定義（`core/ops/`）を確認したか
- [ ] カーネル実装（`core/kernels/`）を確認したか
- [ ] デバイス別実装（CPU/GPU）を確認したか
- [ ] 形状推論の実装を確認したか
- [ ] 勾配計算の実装（`*_grad.py` / `*_grad.cc`）を確認したか
- [ ] テストコード（`*_test.py` / `*_test.cc`）を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. `README.md` - プロジェクト概要とインストール方法
2. 本コードリーディングガイドライン - アーキテクチャとディレクトリ構造
3. `tensorflow/core/framework/` のヘッダファイル - コアデータ構造の理解
4. `CONTRIBUTING.md` - コントリビューション方法

#### 特定のOpを理解したい場合
1. `tensorflow/python/ops/` - Python側のAPI定義を確認
2. `tensorflow/core/ops/` - C++側のOp定義（形状推論含む）を確認
3. `tensorflow/core/kernels/` - カーネル実装を確認
4. `tensorflow/python/kernel_tests/` - テストケースを確認

#### コンパイラ/最適化を理解したい場合
1. `tensorflow/compiler/mlir/README.md` - MLIRベースのコンパイラ概要
2. `tensorflow/core/grappler/` - グラフ最適化パスの実装
3. `tensorflow/compiler/tf2xla/` - XLAへの変換ロジック

#### 改修作業を行う場合
1. 対象機能のPythonテスト（`*_test.py`）を読む
2. Python API（`tensorflow/python/ops/`）の対象関数を確認
3. C++ Op定義（`tensorflow/core/ops/`）を確認
4. カーネル実装（`tensorflow/core/kernels/`）を修正
5. テストを追加・実行

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| `README.md` | プロジェクト概要、インストール手順 | 最初に読む |
| `CONTRIBUTING.md` | コントリビューションガイド | 開発開始時 |
| `RELEASE.md` | リリースノート | バージョン確認時 |
| `SECURITY.md` | セキュリティポリシー | セキュリティ関連作業時 |
| `CODE_OF_CONDUCT.md` | 行動規範 | コミュニティ参加時 |
| `CITATION.cff` | 引用情報 | 論文・レポート作成時 |
| `tensorflow/compiler/mlir/README.md` | MLIRコンパイラの概要 | コンパイラ関連作業時 |
| 公式ドキュメント (tensorflow.org) | API リファレンス、チュートリアル | 随時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `gen_*_ops.py` ファイルが見つからない
A: これらのファイルはビルド時に自動生成されます。Bazelでビルドを実行すると `bazel-genfiles/` 配下に生成されます。ソースコード上では `gen_*` をインポートしている箇所のコメントや対応する `core/ops/` 内のOp定義を参照してください。

#### Q: `REGISTER_OP` や `REGISTER_KERNEL_BUILDER` の定義はどこにあるか
A: `tensorflow/core/framework/op.h` と `tensorflow/core/framework/op_kernel.h` にそれぞれ定義されています。これらのマクロは静的初期化を利用してOpとカーネルをグローバルレジストリに登録します。

#### Q: Python側の `@tf_export` デコレータは何をしているか
A: `tensorflow/python/util/tf_export.py` で定義されており、関数やクラスをTensorFlowの公開APIとして登録します。`tf.reshape` のような短い名前で呼べるようにするための仕組みです。

#### Q: `*.proto` ファイルの変更はどこに影響するか
A: Protocol Buffers定義の変更は、C++の `*.pb.h` / `*.pb.cc` とPythonの `*_pb2.py` に影響します。ビルド時に自動生成されます。

#### Q: テストの実行方法は
A: Bazelを使用します。例: `bazel test //tensorflow/python/kernel_tests:array_ops_test`

### デバッグのヒント

1. **Python側のデバッグ**: `tf.debugging` モジュールの各種アサーション関数を使用
2. **C++側のデバッグ**: `VLOG(level)` マクロでログレベル別の出力が可能。`TF_CHECK_OK` でステータスのアサーション
3. **グラフの可視化**: TensorBoardを使用して計算グラフを可視化
4. **Eager vs Graph**: `tf.config.run_functions_eagerly(True)` でEager実行に切り替えてデバッグ可能
5. **ビルドエラー**: `configure.py` の設定を確認し、依存ライブラリ（CUDA, cuDNN等）のバージョンが互換性を持つか確認

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Op (Operation) | TensorFlowの計算単位。加算、行列積などの数学演算を表す |
| Kernel | Opの実際の計算実装。デバイス（CPU/GPU/TPU）ごとに異なる |
| Tensor | 多次元配列。TensorFlowの基本データ構造 |
| Graph | 計算を表す有向グラフ。ノードがOp、エッジがテンソル |
| Session | （TF1）グラフの実行環境 |
| Eager Execution | （TF2）即座に演算を実行するモード |
| XLA | Accelerated Linear Algebra。TFグラフのJIT/AOTコンパイラ |
| MLIR | Multi-Level Intermediate Representation。コンパイラ基盤 |
| Grappler | TensorFlowのグラフ最適化フレームワーク |
| SavedModel | モデルの永続化フォーマット |
| DTensor | 分散テンソル。マルチデバイス/マルチホストでのテンソル分散 |
| TF Lite | モバイル・組み込みデバイス向け軽量ランタイム |
| Bazel | Googleが開発したビルドシステム。TFの公式ビルドツール |
| Protocol Buffers | Googleのデータシリアライゼーション形式 |
| Eigen | 線形代数ライブラリ。CPU演算の基盤 |
| cuDNN | NVIDIA CUDA Deep Neural Networkライブラリ |
| oneDNN (MKL-DNN) | Intel CPUの最適化ライブラリ |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `tensorflow/__init__.py` | Pythonパッケージエントリーポイント | ランタイム初期化 |
| `tensorflow/core/framework/tensor.h` | Tensorクラス定義 | テンソルのC++実装 |
| `tensorflow/core/framework/op_kernel.h` | OpKernel基底クラス | カーネル実装の基礎 |
| `tensorflow/core/framework/op.h` | Op登録マクロ | REGISTER_OP定義 |
| `tensorflow/core/ops/array_ops.cc` | 配列操作Op定義 | Reshape, Pack等のOp |
| `tensorflow/core/kernels/` | カーネル実装群 | 全Opの計算実装 |
| `tensorflow/python/framework/ops.py` | Pythonグラフ構築 | Graph, Operation定義 |
| `tensorflow/python/ops/array_ops.py` | Python配列操作 | reshape, concat等のAPI |
| `tensorflow/python/eager/context.py` | Eager実行管理 | コンテキスト設定 |
| `tensorflow/tensorflow.bzl` | Bazelマクロ | ビルド設定ルール |
| `tensorflow/tf_version.bzl` | バージョン情報 | TF_VERSION = "2.21.0" |
| `configure.py` | ビルド設定 | CUDA, cuDNN等の設定 |
| `WORKSPACE` | Bazelワークスペース | 外部依存の定義 |

### C. 参考資料

- [TensorFlow公式ドキュメント](https://www.tensorflow.org/)
- [TensorFlow APIリファレンス (Python)](https://www.tensorflow.org/api_docs/python)
- [TensorFlow APIリファレンス (C++)](https://www.tensorflow.org/api_docs/cc)
- [TensorFlow GitHub リポジトリ](https://github.com/tensorflow/tensorflow)
- [Bazel公式ドキュメント](https://bazel.build/)
- [Protocol Buffers公式ドキュメント](https://protobuf.dev/)
- [XLA (Accelerated Linear Algebra)](https://www.tensorflow.org/xla)
- [MLIR プロジェクト](https://mlir.llvm.org/)
- [Eigen ライブラリ](https://eigen.tuxfamily.org/)
