# セキュリティ設計書

## 概要

本ドキュメントは、TensorFlowプロジェクトにおけるセキュリティ設計を記述する。TensorFlowは機械学習フレームワークであり、Webアプリケーションとは異なるセキュリティモデルを持つ。本書では、TensorFlowの脅威モデル、入力検証、通信セキュリティ、ファジングによる脆弱性検出、分散実行環境のセキュリティ、およびセキュリティアドバイザリ管理について整理する。

## 認証設計

### 認証方式

TensorFlowはライブラリ/フレームワークであり、エンドユーザー認証機構は内蔵していない。分散実行環境（`tf.train.Server`およびgRPCベースのデータサービス）における通信認証は、gRPC Credentials Factoryパターンによって提供される。

- **gRPC Credentials Factory**（`tensorflow/core/data/service/credentials_factory.h`、`credentials_factory.cc`）: プロトコルごとにサーバー認証情報（`grpc::ServerCredentials`）およびクライアント認証情報（`grpc::ChannelCredentials`）を生成するファクトリパターンを実装している。
- **デフォルト設定**: デフォルトでは`InsecureCredentialsFactory`が登録されており、`grpc::InsecureServerCredentials()`と`grpc::InsecureChannelCredentials()`を使用する。これは認証・暗号化なしの通信を意味する。
- **拡張ポイント**: `CredentialsFactory::Register()`メソッドにより、TLS/SSL対応のカスタムCredentialsFactoryを登録することが可能。プロダクション環境ではセキュアなCredentialsFactoryの登録が推奨される。

### セッション管理

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション管理 | 該当なし | TensorFlowはHTTPセッションベースのアプリケーションではないため、Webセッション管理機構は存在しない |
| 分散実行セッション | `SessionMgr`による管理 | `tensorflow/core/distributed_runtime/session_mgr.h`でワーカーセッションのライフサイクルを管理 |

## 認可設計

### 権限体系

| ロール | 権限 | 説明 |
| --- | --- | --- |
| モデル実行者 | 任意のグラフ実行 | `tf.train.Server`に接続可能なユーザーは任意の計算グラフを実行可能（SECURITY.mdに明記） |
| マルチテナント利用者 | テナント単位の分離が必要 | テナント分離メカニズムはユーザーの責任で設計・実装する必要がある |

### アクセス制御

TensorFlowのデフォルトのgRPCサーバー（`tf.train.Server`）は認可プロトコルを含まない。ネットワークアクセスが可能な全てのクライアントがグラフの送信・実行を行える。SECURITY.mdでは以下のように明記されている:

> "The default TensorFlow server does not include any authorization protocol and sends messages unencrypted. It accepts connections from anywhere, and executes the graphs it is sent without performing any checks."

セキュアな環境を実現するためには以下の対策が必要:
- ネットワーク分離（VPC、ファイアウォール等）
- カスタムCredentialsFactoryによるTLS/mTLSの導入
- レート制限・ACLの実装

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| gRPC通信 | デフォルトは非暗号化（`InsecureCredentialsFactory`）。`CredentialsFactory`の拡張により、TLS/SSLベースのセキュア通信をサポート可能 |
| データサービス通信 | `tensorflow/core/data/service/`以下でCredentialsFactory経由のgRPC通信を使用。プロトコル指定により暗号化レベルを切替可能 |
| HSTS | 該当なし（HTTP Webサーバーではないため） |
| tf.train.Server | 内部通信専用に設計。信頼できないネットワーク上での使用は非推奨 |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| gRPC通信（デフォルト） | 暗号化なし（InsecureCredentials） |
| gRPC通信（拡張時） | gRPC TLS/SSL（CredentialsFactory拡張による） |
| 保存データ（SavedModel、チェックポイント） | 暗号化機構は組み込まれていない。ファイルシステムレベルでの暗号化が必要 |
| クラウドストレージ | `tensorflow/core/platform/cloud/`のGCS連携ではHTTPS通信を使用 |

### 機密情報管理

- **モデルファイル**: SavedModelおよびチェックポイントには計算グラフ（実質的にプログラム）が含まれるため、信頼できないモデルの読み込みは任意コード実行と同等のリスクがある。
- **チェックポイント変数**: チェックポイントに含まれる変数値はファイルシステム操作やネットワーク操作に使用される場合があり、悪意のあるチェックポイントにより任意の読み取り/書き込み/実行が発生しうる。
- **環境変数・設定値**: `configure.py`で設定されるビルド構成（CUDAパス、NDKバージョン等）はビルド時にのみ使用され、実行時に機密情報として扱われるものではない。

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| メモリ破壊（バッファオーバーフロー、OOB Read/Write） | C++カーネルレベルでの入力検証、OSS-Fuzzによる継続的ファジング（C++・Python両方）、セキュリティアドバイザリによる修正パッチの提供 |
| 不正入力によるクラッシュ（Null Pointer Dereference、整数オーバーフロー等） | セキュリティファジングテスト（`tensorflow/security/fuzzing/`）により検出・修正。Python/C++両方のファジングヘルパーを整備 |
| 信頼できない入力データ | 安全と認定された入力形式：PNG, BMP, GIF, WAV, RAW, RAW_PADDED, CSV, PROTO。その他の形式はサンドボックス内での処理を推奨 |
| 信頼できないモデル/グラフ | サンドボックス内での実行を推奨。モデルは実質的にプログラムであり、信頼できないモデルの実行は任意コード実行に等しい |
| XSS | 該当なし（Webフロントエンドを持たないため） |
| SQLインジェクション | 該当なし（SQLデータベースを使用しないため） |
| CSRF | 該当なし（Webアプリケーションではないため） |

## ファジングによるセキュリティテスト

TensorFlowは包括的なファジング基盤を整備している。

### ファジング構成

| 種別 | 配置場所 | 説明 |
| --- | --- | --- |
| C++ファジング | `tensorflow/security/fuzzing/cc/` | TensorFlow C++ Opsに対するファジングテスト（`AreAttrValuesEqual_fuzz.cc`, `base64_fuzz.cc`, `checkpoint_reader_fuzz.cc`, `status_fuzz.cc`等） |
| Pythonファジング | `tensorflow/security/fuzzing/` | Python APIに対するファジングテスト（`abs_fuzz.py`, `add_fuzz.py`, `constant_fuzz.py`等） |
| ファジングヘルパー | `tensorflow/security/fuzzing/python_fuzzing.py` | atherisベースのファジングデータプロバイダラッパー。テンソル形状・値のランダム生成を安全な範囲に制限 |
| OSS-Fuzz連携 | 外部サービス | Google OSS-Fuzzとの統合により継続的なファジングを実施 |

### ファジングヘルパーの安全制限

`python_fuzzing.py`の`FuzzingHelper`クラスでは以下の安全制限を設けている:

- テンソルの最大次元サイズ: 8（OOMエラー防止）
- 整数範囲: -10000 から 10000
- 浮動小数点範囲: -10000.0 から 10000.0
- リスト最大長: 10000

## セキュリティアドバイザリ管理

### アドバイザリ管理体制

| 項目 | 内容 |
| --- | --- |
| 管理場所 | `tensorflow/security/advisory/` |
| 識別体系 | TFSA-YYYY-NNN形式（例: TFSA-2023-020） |
| 報告窓口 | Google Bug Hunters（https://g.co/vulnz） |
| 修正サイクル | 四半期リリースに合わせてバッチ修正 |
| 対象範囲 | SECURITY.mdで「安全」と定義されたシナリオで発生する脆弱性のみ |

### 主要な脆弱性カテゴリ（アドバイザリ分析）

セキュリティアドバイザリ（2018年 - 2023年）の分析により、以下の脆弱性カテゴリが確認されている:

| カテゴリ | 例 |
| --- | --- |
| ヒープバッファオーバーフロー | TFSA-2023-010（AvgPoolGrad）、TFSA-2022-168（QuantizeAndDequantizeV2） |
| 境界外読み取り（OOB Read） | TFSA-2023-020（GRUBlockCellGrad）、TFSA-2023-018（DynamicStitch） |
| Null Pointer Dereference | TFSA-2023-017（QuantizedMatMulWithBiasAndDequantize）、TFSA-2023-012（TensorArrayConcatV2） |
| 浮動小数点例外（FPE） | TFSA-2023-019（AvgPoolGrad with XLA）、TFSA-2023-008（AudioSpectrogram） |
| セグメンテーションフォルト | TFSA-2023-015（tfg-translate）、TFSA-2023-004（multiframe gif） |
| 整数オーバーフロー | TFSA-2023-014（EditDistance） |
| Double Free | TFSA-2023-003（Fractional(Max/Avg)Pool） |
| TFLiteバッファオーバーフロー | TFSA-2022-156（CONV_3D_TRANSPOSE） |

## マルチテナント環境のセキュリティ

SECURITY.mdに記載されたマルチテナント環境でのセキュリティ考慮事項:

| 項目 | リスク | 対策 |
| --- | --- | --- |
| テナント分離 | モデル/グラフ提供テナントによるコード実行 | ユーザー責任でネットワーク分離・サンドボックスを実装 |
| リソース割り当て | 1つのモデルによるDoSでサーバー全体停止 | レート制限、ACL、監視によるサーバー再起動 |
| モデル共有 | 悪意のあるモデルの検出困難 | モデル実行のサンドボックス化 |
| ハードウェア攻撃 | GPU/TPUサイドチャネル攻撃によるデータ漏洩 | ハードウェアベンダーへの脆弱性報告、物理的な分離 |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| セキュリティアドバイザリ | 脆弱性の詳細、影響バージョン、修正内容 | リポジトリに永続保存（`tensorflow/security/advisory/`） |
| OSS-Fuzzレポート | ファジングで検出されたバグ | Google OSS-Fuzz Trackerで管理 |
| CredentialsFactory登録ログ | 重複プロトコル登録時のERRORログ出力 | 標準ログ出力に依存 |

TensorFlowはランタイムレベルの監査ログ機構を内蔵していない。分散実行環境での監査ログが必要な場合は、gRPCインターセプターやプロキシレベルでの実装が必要となる。

## 備考

- TensorFlowは機械学習フレームワークであり、Webアプリケーション的なセキュリティ機構（CSRF対策、XSS対策、SQLインジェクション対策等）は該当しない。
- セキュリティモデルの核心は「信頼できないモデル/データの取り扱い」と「分散実行環境の保護」にある。
- Eager Modeではシェイプ推論関数がスキップされるため、一部のチェックが実行されない。モデルサービングにはEager Modeを使用しないことが推奨される。
- XLAコンパイルは推奨エントリポイント経由であれば安全。テスト・デバッグツールは信頼できないデータとの併用に注意が必要。
- 脆弱性レポートの対象は、SECURITY.mdで「安全」と定義されたユースケースで発生する問題に限定される。
