# バッチ設計書 1-wheel.sh

## 概要

本ドキュメントは、TensorFlowのCI/CDパイプラインにおけるPython wheelパッケージビルドバッチ `wheel.sh` の設計仕様を記述する。

### 本バッチの処理概要

本バッチは、TensorFlowのPython wheelパッケージをBazelビルドシステムを用いてビルドし、生成されたwheelファイルに対してauditwheel検証を実施した上で、Google Cloud Storage（GCS）のステージング領域へアップロードする処理を行う。

**業務上の目的・背景**：TensorFlowプロジェクトでは、各プラットフォーム・Python バージョン向けにwheelパッケージを配布する必要がある。本バッチはCI/CDパイプラインの中核をなし、ビルド成果物の生成からステージングまでを自動化することで、リリースプロセスの効率化と品質担保を実現する。Nightly ビルドおよびリリースビルドの両方に対応し、CUDA対応のGPU版wheelも生成可能である。

**バッチの実行タイミング**：CI/CDパイプラインのトリガーに基づき随時実行される。Nightlyビルドの場合は日次で実行され、リリースビルドの場合はリリースブランチへのコミット時に実行される。

**主要な処理内容**：
1. TFCI環境の初期化（setup.sh の読み込み）
2. GPU環境の場合、nvidia-smi による GPU 情報の記録
3. Nightlyビルドの場合、バージョン番号の更新およびwheel名の `tf_nightly` への変換
4. NumPy 1互換ビルドの場合、対応する要件ロックファイルの配置
5. Bazelによるwheelパッケージのビルド（`//tensorflow/tools/pip_package:wheel` ターゲット）
6. ビルド成果物（.whlファイル）の出力ディレクトリへのコピー
7. `rename_and_verify_wheels.sh` による auditwheel修復・サイズチェック・インストール検証
8. 追加wheel名が指定されている場合の再ビルドとコピー
9. GCSステージングが有効な場合、gsutil による成果物のアップロード
10. Bazelテストが有効な場合、wheelテストの実行

**前後の処理との関連**：本バッチの前段として `setup.sh` による環境初期化が行われる。後続処理として `upload.sh` によるGCS/PyPIへの最終アップロードが想定される。また、`rename_and_verify_wheels.sh` をサブプロセスとして呼び出す。

**影響範囲**：ビルド成果物（wheelファイル）の生成、`$TFCI_OUTPUT_DIR` 配下のファイル、GCSステージング領域（`$TFCI_ARTIFACT_STAGING_GCS_URI`）。

## バッチ種別

ビルド・パッケージング / 成果物ステージング

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（CI/CDトリガーに依存） |
| 実行時刻 | CI/CDパイプラインによる |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | CI/CDパイプライン（Kokoro等）からの呼び出し |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| TFCI環境変数の設定 | `$TFCI` 環境変数に適切な env ファイルパスが設定されていること |
| Bazelの利用可能性 | Bazelビルドツールがインストールされ利用可能であること |
| Python環境 | 指定バージョンの Python が利用可能であること |
| GCS認証情報 | GCSアップロードが有効な場合、適切な認証情報が設定されていること |

### 実行可否判定

`setup.sh` による環境初期化が正常に完了し、必要なTFCI環境変数がすべて設定されていること。スクリプトは `set -exo pipefail` で実行されるため、いずれかのコマンドが失敗した時点でバッチ全体が中断される。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| TFCI | string | Yes | なし | 環境設定ファイルのカンマ区切りパス |
| TFCI_NVIDIA_SMI_ENABLE | string | No | 0 | nvidia-smi実行の有効/無効 |
| TFCI_NIGHTLY_UPDATE_VERSION_ENABLE | string | No | 0 | Nightlyバージョン更新の有効/無効 |
| TFCI_WHL_NUMPY_VERSION | string | No | なし | NumPyバージョン（1の場合NumPy1互換） |
| TFCI_BAZEL_BAZELRC_ARGS | string | Yes | なし | Bazel設定ファイル引数 |
| TFCI_BAZEL_COMMON_ARGS | string | Yes | なし | Bazel共通引数 |
| TFCI_BUILD_PIP_PACKAGE_BASE_ARGS | string | Yes | なし | pip パッケージビルド基本引数 |
| TFCI_BUILD_PIP_PACKAGE_WHEEL_NAME_ARG | string | Yes | なし | wheel名指定引数 |
| TFCI_BUILD_PIP_PACKAGE_ADDITIONAL_WHEEL_NAMES | string | No | なし | 追加wheel名（スペース区切り） |
| TFCI_OUTPUT_DIR | string | Yes | なし | 出力ディレクトリパス |
| TFCI_FIND_BIN | string | Yes | なし | findコマンドのパス |
| TFCI_ARTIFACT_STAGING_GCS_ENABLE | string | No | 0 | GCSステージングの有効/無効 |
| TFCI_ARTIFACT_STAGING_GCS_URI | string | No | なし | GCSステージングURI |
| TFCI_WHL_BAZEL_TEST_ENABLE | string | No | 0 | Bazelテストの有効/無効 |
| TFCI_PYTHON_VERSION | string | No | なし | Pythonバージョン |
| TFCI_BAZEL_TARGET_SELECTING_CONFIG_PREFIX | string | No | なし | Bazelターゲット選択設定プレフィックス |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| TensorFlowソースコード | ソースファイル | Bazelビルド対象のソースコード一式 |
| TFCI環境設定ファイル | シェルスクリプト | `ci/official/envs/` 配下の環境変数定義ファイル |
| NumPy1要件ファイル | テキスト | `ci/official/requirements_updater/numpy1_requirements/*.txt` |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| $TFCI_OUTPUT_DIR/*.whl | wheel | ビルドされたPython wheelパッケージ |
| ./dist/*.whl | wheel | rename_and_verify用のwheelコピー |
| $TFCI_ARTIFACT_STAGING_GCS_URI | GCS | GCSステージング領域へのwheelアップロード |
| $TFCI_OUTPUT_DIR/script.log | テキスト | スクリプト実行ログ |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | `tensorflow-{version}-{python_tag}-{abi_tag}-{platform_tag}.whl` または `tf_nightly-{version}-{python_tag}-{abi_tag}-{platform_tag}.whl` |
| 出力先 | `$TFCI_OUTPUT_DIR` |
| 文字コード | バイナリ |
| 区切り文字 | 該当なし |

## 処理フロー

### 処理シーケンス

```
1. 環境初期化
   └─ setup.sh を source し、TFCI環境変数・Docker設定・クリーンアップトラップを設定
2. GPU情報記録（条件付き）
   └─ TFCI_NVIDIA_SMI_ENABLE=1 の場合、nvidia-smi を実行
3. Nightlyバージョン更新（条件付き）
   └─ TFCI_NIGHTLY_UPDATE_VERSION_ENABLE=1 の場合、update_version.py --nightly を実行し、wheel名をtf_nightlyに変換
4. NumPy1要件ファイル配置（条件付き）
   └─ TFCI_WHL_NUMPY_VERSION=1 の場合、numpy1用要件ファイルをルートにコピー
5. Bazelビルド実行
   └─ bazel build で //tensorflow/tools/pip_package:wheel ターゲットをビルド
6. wheelファイルのコピー
   └─ bazel-bin 配下の .whl ファイルを $TFCI_OUTPUT_DIR および ./dist にコピー
7. wheel検証
   └─ rename_and_verify_wheels.sh を実行（auditwheel修復・サイズチェック・インストールテスト）
8. 追加wheelビルド（条件付き）
   └─ TFCI_BUILD_PIP_PACKAGE_ADDITIONAL_WHEEL_NAMES が設定されている場合、各wheel名で再ビルド
9. GCSアップロード（条件付き）
   └─ TFCI_ARTIFACT_STAGING_GCS_ENABLE=1 の場合、gsutil cp でGCSにアップロード（-n で上書き防止）
10. Bazelテスト実行（条件付き）
    └─ TFCI_WHL_BAZEL_TEST_ENABLE=1 の場合、wheelテスト設定でbazel testを実行
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[setup.sh による環境初期化]
    B --> C{NVIDIA SMI有効?}
    C -->|Yes| D[nvidia-smi 実行]
    C -->|No| E{Nightly更新有効?}
    D --> E
    E -->|Yes| F[バージョン番号更新・wheel名変換]
    E -->|No| G{NumPy1?}
    F --> G
    G -->|Yes| H[NumPy1要件ファイル配置]
    G -->|No| I[Bazel wheelビルド]
    H --> I
    I --> J[wheelファイルを出力ディレクトリにコピー]
    J --> K[rename_and_verify_wheels.sh 実行]
    K --> L{追加wheel名あり?}
    L -->|Yes| M[追加wheelビルド・コピー]
    L -->|No| N{GCSステージング有効?}
    M --> N
    N -->|Yes| O[gsutil cp でGCSにアップロード]
    N -->|No| P{Bazelテスト有効?}
    O --> P
    P -->|Yes| Q[bazel test 実行]
    P -->|No| R[バッチ終了]
    Q --> R
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベース操作を行わない。ファイルシステムおよびGCSオブジェクトストレージに対する操作のみを行う。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 非ゼロ終了 | Bazelビルドエラー | ビルド対象のコンパイルエラー等 | ビルドログを確認し、ソースコードまたはBazel設定を修正 |
| 非ゼロ終了 | auditwheel検証エラー | wheelの共有ライブラリ依存性問題 | ビルド設定・依存関係を確認 |
| 2 | wheelサイズ超過 | 生成wheelがサイズ制限を超過 | TFCI_WHL_SIZE_LIMIT の確認またはビルド内容の見直し |
| 非ゼロ終了 | GCSアップロードエラー | 認証失敗・ネットワークエラー等 | 認証情報・ネットワーク接続を確認 |
| 非ゼロ終了 | インストール検証エラー | wheelのインストールまたはimportテスト失敗 | ビルド成果物の依存関係・互換性を確認 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | スクリプト自体にリトライ機能なし（CIシステム側で制御） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

### 障害時対応

`set -exo pipefail` により、いずれかのコマンドが失敗した時点でスクリプトが終了する。`setup.sh` で設定されたEXITトラップにより、Dockerクリーンアップおよびビルドサマリーの出力が行われる。失敗時はCI/CDシステムのログおよび `$TFCI_OUTPUT_DIR/script.log` を確認し、原因を特定する。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 該当なし（ファイルシステム・GCS操作） |
| コミットタイミング | 該当なし |
| ロールバック条件 | GCSアップロードは `-n` フラグにより既存ファイルを上書きしない。途中失敗時の部分成果物は手動クリーンアップが必要 |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1回の実行で1〜複数個のwheelファイルを生成 |
| 目標処理時間 | ビルド環境に依存（通常数十分〜数時間） |
| メモリ使用量上限 | Bazelビルドに依存（通常数GB以上） |

## 排他制御

本バッチはCI/CDパイプラインの1ジョブとして実行される。同一出力ディレクトリに対する並行実行は想定されていない。GCSアップロード時の `-n` フラグにより、既存ファイルの上書きが防止される。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 開始ログ | setup.sh 実行時 | TFCI環境変数の一覧 |
| 進捗ログ | 各コマンド実行時 | `set -x` により全コマンドがトレースログとして出力 |
| 終了ログ | バッチ終了時（EXITトラップ） | cleanup_summary.sh によるBazel ResultStoreリンク |
| エラーログ | エラー発生時 | `set -e` によりエラー発生コマンドの出力 |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| ジョブ実行結果 | 非ゼロ終了コード | CI/CDシステム（Kokoro等）の通知機能 |
| wheelサイズ | TFCI_WHL_SIZE_LIMIT | スクリプト内のサイズチェック（終了コード2） |

## 備考

- Windows環境（MSYS_NT）では、gsutilの実行にpowershellを使用するワークアラウンドが実装されている（b/389744576）
- NumPy 1互換ビルドは将来的に廃止予定（b/361369076）
- `--config=cuda_wheel` によりCUDA対応wheelがビルドされる
- `--verbose_failures` フラグによりビルド失敗時の詳細情報が出力される
