# バッチ設計書 11-setup.sh

## 概要

本ドキュメントは、TensorFlow CI/CDパイプラインにおける全CIスクリプト共通の初期化処理バッチ `setup.sh` の設計仕様を記述する。本バッチは他のすべてのCIスクリプトから `source` で読み込まれ、実行環境の基盤を構築する。

### 本バッチの処理概要

本バッチは、TensorFlowのCI/CDパイプラインにおいて全スクリプトが共通で必要とする初期化処理を一元的に担うユーティリティスクリプトである。環境変数の読込、Docker設定の委譲、クリーンアップトラップの登録など、CIジョブ実行の前提条件を整備する。

**業務上の目的・背景**：TensorFlowのCIパイプラインでは、多数のビルド・テスト・アップロードスクリプトが存在する。これらすべてに共通する初期化処理（環境変数設定、Docker環境準備、ログ出力設定、クリーンアップ処理登録）を個別スクリプトごとに実装すると保守性が著しく低下する。本バッチはこの課題を解決するため、共通初期化処理を一箇所に集約する。

**バッチの実行タイミング**：各CIスクリプト（wheel.sh、pycpp.sh、libtensorflow.sh等）の冒頭で `source` により読み込まれる。CI/CDパイプラインのジョブ実行時（プルリクエスト、ナイトリービルド、リリースビルド等）に随時実行される。

**主要な処理内容**：
1. シェルオプション設定（`set -exo pipefail -o history -o allexport`）によるエラーハンドリングとデバッグ出力の有効化
2. `TFCI_GIT_DIR` の算出とカレントディレクトリの移動
3. `TFCI` 環境変数で指定されたenvファイル群の読み込みと、既存環境変数の復元
4. インストーラーwheel用環境変数の条件付き設定
5. macOS固有セットアップの条件付き実行
6. 出力ディレクトリ（`TFCI_OUTPUT_DIR`）の作成とログのtee出力設定
7. `tfrun` ヘルパー関数の定義（ローカル実行用）
8. Windows（MSYS）環境固有のパス変換処理
9. Docker有効時の `setup_docker.sh` の読み込みによるDocker環境構築
10. `generate_index_html.sh` による概要HTMLの条件付き生成
11. Bazelバージョン確認のリトライ実行
12. EXITトラップによるクリーンアップ処理の登録（Docker停止、サマリー出力）

**前後の処理との関連**：本バッチはすべてのCIジョブスクリプト（No.1〜No.10の各バッチ）から最初に読み込まれる前提条件スクリプトである。内部的にNo.12（setup_docker.sh）、No.15（cleanup_docker.sh）、No.16（cleanup_summary.sh）、No.17（generate_index_html.sh）を呼び出す。

**影響範囲**：全CIスクリプトの実行環境に影響する。環境変数、Docker設定、ログ出力、クリーンアップ処理のすべてが本バッチの設定に依存する。

## バッチ種別

環境初期化 / CI共通セットアップ

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（CIジョブ実行時） |
| 実行時刻 | CIジョブのトリガーに依存 |
| 実行曜日 | 該当なし |
| 実行日 | 該当なし |
| トリガー | 各CIスクリプトからの `source` 呼び出し |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| TFCIリポジトリの存在 | 本スクリプトはTensorFlowリポジトリのルートから2階層下（`ci/official/utilities/`）に配置されている前提 |
| TFCI環境変数（任意） | `$TFCI` に環境設定ファイル名をカンマ区切りで指定するか、事前に `TFCI_*` 変数をエクスポート済みであること |
| Bazelのインストール | `bazel --version` が実行可能であること |

### 実行可否判定

本バッチは `source` で読み込まれるため、呼び出し元スクリプトが存在すれば常に実行される。Docker設定（`TFCI_DOCKER_ENABLE`）やインデックスHTML生成（`TFCI_INDEX_HTML_ENABLE`）は条件分岐により制御される。

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| TFCI | string | No | 空文字 | カンマ区切りのenvファイル名リスト（`ci/official/envs/` 配下） |
| TFCI_DOCKER_ENABLE | string | Yes | 環境依存 | Docker使用の有無（"1"で有効） |
| TFCI_INDEX_HTML_ENABLE | string | Yes | 環境依存 | index.html生成の有無（"1"で有効） |
| TFCI_INSTALLER_WHL_ENABLE | string | Yes | 環境依存 | インストーラーwheel設定の有無（"1"で有効） |
| TFCI_NIGHTLY_UPDATE_VERSION_ENABLE | string | Yes | 環境依存 | ナイトリーバージョン更新の有無（"1"で有効） |
| TFCI_OUTPUT_DIR | string | Yes | 環境依存 | 出力ディレクトリのパス |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| ci/official/envs/ci_default | envファイル | デフォルトCI設定変数 |
| ci/official/envs/{TFCI指定ファイル} | envファイル | TFCI変数で指定された環境設定ファイル群 |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| $TFCI_OUTPUT_DIR/script.log | テキストログ | スクリプト全体の実行ログ（tee出力） |
| $TFCI_OUTPUT_DIR/index.html | HTML | Kokoroビルドジョブ概要（条件付き） |
| シェル環境変数 | 変数 | TFCI_* 変数群のエクスポート |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | script.log |
| 出力先 | $TFCI_OUTPUT_DIR/ |
| 文字コード | UTF-8 |
| 区切り文字 | 該当なし（ログ形式） |

## 処理フロー

### 処理シーケンス

```
1. シェルオプション設定
   └─ set -exo pipefail -o history -o allexport
2. TFCI_GIT_DIR の算出・カレントディレクトリ変更
   └─ setup.sh の2階層上をリポジトリルートとして設定
3. TFCI環境変数ファイルの読み込み
   └─ ci_default を読み込み後、TFCI指定の各envファイルを順次source
   └─ 既存のTFCI_変数をFROM_ENVとして退避・復元
4. インストーラーwheel設定（条件付き）
   └─ TFCI_INSTALLER_WHL_ENABLE=1 の場合のみ実行
5. macOSセットアップ（条件付き）
   └─ OSTYPE が darwin* の場合のみ setup_macos.sh を source
6. 出力ディレクトリ作成・ログtee設定
   └─ TFCI_OUTPUT_DIR を作成し、script.log へのtee出力を開始
7. tfrun関数の定義
   └─ ローカル実行用のパススルー関数
8. Windows環境パス変換（条件付き）
   └─ MSYS_NT環境の場合のみパス変換を実行
9. Docker環境構築（条件付き）
   └─ TFCI_DOCKER_ENABLE=1 の場合、setup_docker.sh を source
10. index.html生成（条件付き）
    └─ TFCI_INDEX_HTML_ENABLE=1 の場合、generate_index_html.sh を実行
11. Bazelバージョン確認（リトライ付き）
    └─ 最大2回リトライ
12. EXITトラップ登録
    └─ cleanup関数（Docker停止 + サマリー出力）を登録
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[シェルオプション設定]
    B --> C[TFCI_GIT_DIR算出・cd]
    C --> D{TFCI変数設定あり?}
    D -->|あり| E[ci_defaultとenvファイル読込]
    D -->|なし| F[スキップ]
    E --> G[既存TFCI_変数復元]
    G --> H{INSTALLER_WHL有効?}
    F --> H
    H -->|Yes| I[installer wheel変数設定]
    H -->|No| J{macOS?}
    I --> J
    J -->|Yes| K[setup_macos.sh実行]
    J -->|No| L[出力DIR作成・ログtee設定]
    K --> L
    L --> M[tfrun関数定義]
    M --> N{Windows?}
    N -->|Yes| O[パス変換]
    N -->|No| P{Docker有効?}
    O --> P
    P -->|Yes| Q[setup_docker.sh実行]
    P -->|No| R{INDEX_HTML有効?}
    Q --> R
    R -->|Yes| S[index.html生成]
    R -->|No| T[Bazelバージョン確認]
    S --> T
    T --> U[EXITトラップ登録]
    U --> V[バッチ終了]
```

## データベース操作仕様

### 操作別データベース影響一覧

本バッチはデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 終了コード1 | 未定義変数エラー | `set -u` 有効時にTFCI_変数が未設定 | TFCI環境ファイルまたは環境変数を正しく設定する |
| 終了コード1 | パイプ失敗 | `set -o pipefail` によるパイプライン内エラー | 各コマンドのエラー原因を調査する |
| 終了コード非0 | Bazelバージョン取得失敗 | Bazelが未インストールまたはダウンロード失敗 | Bazelをインストールまたはネットワーク接続を確認する |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 2回（Bazelバージョン確認のみ） |
| リトライ間隔 | 即時リトライ（間隔なし） |
| リトライ対象エラー | `bazel --version` の実行失敗 |

### 障害時対応

本バッチが失敗した場合、呼び出し元のCIジョブ全体が失敗する。EXITトラップに登録されたcleanup関数が実行され、Docker環境のクリーンアップ情報表示とResultStoreリンクの抽出・表示が行われる。根本原因はスクリプトログ（`$TFCI_OUTPUT_DIR/script.log`）で確認する。

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 該当なし（シェルスクリプト実行） |
| コミットタイミング | 該当なし |
| ロールバック条件 | 該当なし（EXITトラップによるクリーンアップのみ） |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 1回（CIジョブ開始時に1回実行） |
| 目標処理時間 | 数秒〜数十秒（Docker pull時は数分） |
| メモリ使用量上限 | 特に制限なし（シェルスクリプト） |

## 排他制御

本バッチ自体は排他制御を行わない。ただし、Docker環境が有効な場合、`tf` という名前のコンテナが既に存在するかどうかをチェックし、存在する場合は再作成しない（setup_docker.sh側の制御）。

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 開始ログ | TFCI変数読込時 | `==TFCI==:` プレフィックス付きの設定情報 |
| 進捗ログ | 各コマンド実行時 | `set -x` によるコマンドトレース |
| 終了ログ | EXITトラップ時 | クリーンアップ処理の結果 |
| エラーログ | エラー発生時 | `set -e` によるエラー時の即時終了とトレース |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| CIジョブ全体の成否 | 終了コード非0 | Kokoro CI / GitHub PR ステータス |
| Bazelバージョン取得 | 2回リトライ後も失敗 | CIジョブログ |

## 備考

- 本スクリプトのコメントには「Make as FEW changes to this file as possible（このファイルへの変更は最小限に）」という注意書きがあり、変更時は慎重な検討が必要である。
- `tfrun` 関数はパイプを含むコマンドには使用できない設計である。パイプが必要な場合は `utilities/` 配下に専用スクリプトを作成する方針。
- ソースファイルパス: `ci/official/utilities/setup.sh`
