# 機能設計書 115-分散ランタイム

## 概要

本ドキュメントは、TensorFlowにおけるgRPCベースの分散実行ランタイムの設計を記述する。

### 本機能の処理概要

複数のマシン（ワーカー）間でTensorFlowの計算グラフを分散実行するためのランタイムインフラストラクチャを提供する。Master-Worker アーキテクチャに基づき、gRPCプロトコルを用いたプロセス間通信、セッション管理、グラフの分散配置・実行を実現する。

**業務上の目的・背景**：大規模な機械学習モデルの訓練では、単一マシンの計算リソースでは不十分なケースが多い。分散ランタイムは、複数のGPU/TPUサーバーにまたがるデータ並列・モデル並列の分散学習を実現するための基盤であり、スケーラブルなMLワークロードの実行に不可欠である。

**機能の利用シーン**：マルチワーカー分散学習、パラメータサーバ構成での学習、gRPCベースのリモート推論、クラスタ上でのジョブスケジューリングなどで利用される。

**主要な処理内容**：
1. Master - クライアントからのセッション作成・グラフ登録・実行要求を受け付け、ワーカーへ分散配置
2. Worker - Masterからの指示に基づきグラフの部分実行を行い、テンソルの送受信を担当
3. SessionMgr - ワーカー側でのセッション状態管理
4. GraphMgr - 分散グラフの登録・実行管理
5. Coordination Service - マルチワーカー間の調整・エラー伝播

**関連システム・外部連携**：gRPC通信フレームワーク、Protocol Buffer（通信プロトコル）、共通ランタイム（No.116）、コレクティブ操作（No.71）と連携する。

**権限による制御**：特になし。ただしgRPCサーバーへの接続はネットワークレベルで制御される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 該当なし | - | 本機能はバックエンドランタイム機能であり、直接的な画面関連はない |

## 機能種別

ランタイムインフラストラクチャ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| ServerDef | protobuf | Yes | サーバー定義（クラスタ構成、ジョブ名、タスクID等） | - |
| GraphDef | protobuf | Yes | 実行する計算グラフ定義 | - |
| ConfigProto | protobuf | No | 実行設定（デバイス配置、メモリ設定等） | - |
| session_handle | string | Yes | セッション識別子 | 一意であること |
| step_id | int64 | Yes | 実行ステップ識別子 | グローバルユニーク |

### 入力データソース

gRPCリクエスト（CreateSession、RunStep等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| session_handle | string | 作成されたセッションのハンドル |
| tensor_response | TensorProto | 実行結果のテンソル |
| status | Status | 操作の成功/失敗ステータス |
| step_stats | StepStats | 実行統計情報 |

### 出力先

gRPCレスポンスとして呼び出し元に返却

## 処理フロー

### 処理シーケンス

```
1. セッション作成（Master::CreateSession）
   └─ クライアントからCreateSessionRequestを受信
   └─ ワーカーへCreateWorkerSessionを配信
   └─ MasterSessionを生成しセッションハンドルを返却

2. グラフ登録（Worker::RegisterGraph）
   └─ MasterからRegisterGraphRequestを受信
   └─ GraphMgr::Registerでグラフをパーティション分割
   └─ 各デバイスにExecutorを生成

3. ステップ実行（Master::RunStep → Worker::RunGraph）
   └─ クライアントからRunStepRequestを受信
   └─ Masterがワーカーへ分散実行を指示
   └─ GraphMgr::ExecuteAsyncで並列実行
   └─ Rendezvouz経由でテンソル送受信

4. セッション終了（Master::CloseSession）
   └─ ワーカーへDeleteWorkerSessionを配信
   └─ MasterSessionをクリーンアップ
```

### フローチャート

```mermaid
flowchart TD
    A[クライアント] -->|CreateSession| B[Master]
    B -->|CreateWorkerSession| C[Worker 1]
    B -->|CreateWorkerSession| D[Worker 2]
    A -->|RunStep| B
    B -->|RunGraph| C
    B -->|RunGraph| D
    C -->|RecvTensor| D
    D -->|RecvTensor| C
    C -->|結果| B
    D -->|結果| B
    B -->|結果| A
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-115-01 | セッションGC | 一定時間非アクティブなセッションは自動削除 | session_gc_seconds設定時 |
| BR-115-02 | リクエスト重複検出 | RecentRequestIdsにより重複リクエストを検出 | 全リクエスト |
| BR-115-03 | マスター再起動検出 | ワーカーはmaster_incarnationでマスター再起動を検出し、古いセッションを自動削除 | CreateSession時 |
| BR-115-04 | Coordination Service | CoordinationServiceAgentによるワーカー間調整 | マルチワーカー構成時 |

### 計算ロジック

該当なし（ランタイムインフラストラクチャ）

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| Unavailable | 通信エラー | gRPC接続失敗 | リトライまたはクラスタ構成確認 |
| NotFound | セッション未発見 | 無効なsession_handleの使用 | 新しいセッションを作成 |
| Aborted | セッション失効 | マスター再起動によるセッション無効化 | セッションを再作成 |
| Internal | グラフ実行エラー | ワーカー上でのOp実行エラー | エラーメッセージに基づきデバッグ |

### リトライ仕様

gRPC通信レベルでのリトライはCancellableCallで管理。タイムアウトはCallOptionsで設定可能。

## トランザクション仕様

セッション単位でのリソース管理。セッション削除時に関連リソースを一括解放。

## パフォーマンス要件

gRPC通信のレイテンシがボトルネック。テンソル転送にはRecvTensorの最適化（GrpcWorkerでの特殊データ構造使用）が適用される。

## セキュリティ考慮事項

gRPC通信はTLSによる暗号化が可能。ネットワークレベルでのアクセス制御を推奨。

## 備考

- GrpcWorkerはWorkerクラスのサブクラスであり、RecvTensorAsyncを最適化実装している。
- 分散ランタイムは主にTensorFlow 1.x系の分散学習で使用され、TF2.xではDistribution Strategy APIが推奨される。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: アーキテクチャを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | master.h | `tensorflow/core/distributed_runtime/master.h` | Masterクラスのインタフェース（CreateSession, RunStep等） |
| 1-2 | worker.h | `tensorflow/core/distributed_runtime/worker.h` | Workerクラスのインタフェース（RegisterGraph, RunGraph等） |

**読解のコツ**: Master-Workerパターンを理解する。MasterはクライアントAPI（CreateSession, RunStep）を処理し、内部的にWorkerに処理を委譲する。WorkerはWorkerInterfaceを実装し、gRPCサブクラス（GrpcWorker）で通信レイヤーを提供する。

#### Step 2: セッション管理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | session_mgr.h | `tensorflow/core/distributed_runtime/session_mgr.h` | SessionMgrクラスによるワーカー側セッション管理 |

**主要処理フロー**:
1. **42-46行目**: SessionMgrクラス宣言 - WorkerEnv、WorkerCacheFactory、CoordinationServiceRpcHandlerを依存として受け取る
2. **55-64行目**: CreateSession - セッションの生成。isolate_session_stateによるセッション分離制御
3. **73-80行目**: master_task/master_incarnation付きCreateSession - マスター再起動検出機能
4. **97行目**: DeleteSession - セッションの削除
5. **105行目**: GetCoordinationServiceAgent - Coordination Serviceエージェントへのアクセス

#### Step 3: グラフ管理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | graph_mgr.h | `tensorflow/core/distributed_runtime/graph_mgr.h` | GraphMgrクラスによる分散グラフ管理 |

**主要処理フロー**:
- **81-84行目**: GraphMgrクラス宣言 - WorkerEnvとDeviceMgrに依存
- **88-94行目**: Register - グラフの登録。GraphDefをパーティション分割し各デバイスにExecutorを生成
- **102-108行目**: ExecuteAsync - 登録済みグラフの非同期実行。step_id、NamedTensorsによる入出力管理
- **116行目**: Deregister - グラフの登録解除

### プログラム呼び出し階層図

```
Client (Python)
    │
    ├─ Master
    │      ├─ CreateSession()
    │      │      └─ Worker::CreateWorkerSessionAsync()
    │      ├─ RunStep()
    │      │      └─ Worker::RunGraphAsync()
    │      │             └─ GraphMgr::ExecuteAsync()
    │      │                    └─ StartParallelExecutors()
    │      │                           └─ Executor::RunAsync()
    │      └─ CloseSession()
    │             └─ Worker::DeleteWorkerSessionAsync()
    │
    └─ SessionMgr (Worker側)
           ├─ CreateSession()
           ├─ WorkerSessionForSession()
           └─ DeleteSession()
```

### データフロー図

```
[入力]                      [処理]                         [出力]

CreateSessionReq ──▶ Master::CreateSession ──────▶ session_handle
RunStepReq       ──▶ Master::RunStep ────────────▶ TensorResponse
                        │
                        ├─▶ Worker1::RunGraph ────▶ partial results
                        │       └─ Rendezvous (tensor exchange)
                        └─▶ Worker2::RunGraph ────▶ partial results
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| master.h / master.cc | `tensorflow/core/distributed_runtime/master.h` | ソース | Masterクラス（セッション管理、RunStep実行） |
| master_session.h | `tensorflow/core/distributed_runtime/master_session.h` | ソース | MasterSessionクラス（個別セッション管理） |
| worker.h / worker.cc | `tensorflow/core/distributed_runtime/worker.h` | ソース | Workerクラス（グラフ実行、テンソル送受信） |
| worker_interface.h | `tensorflow/core/distributed_runtime/worker_interface.h` | ソース | WorkerInterfaceの抽象定義 |
| session_mgr.h | `tensorflow/core/distributed_runtime/session_mgr.h` | ソース | SessionMgrクラス（ワーカー側セッション管理） |
| graph_mgr.h / graph_mgr.cc | `tensorflow/core/distributed_runtime/graph_mgr.h` | ソース | GraphMgrクラス（分散グラフ管理） |
| master_env.h | `tensorflow/core/distributed_runtime/master_env.h` | ソース | MasterEnv構造体（Master環境設定） |
| worker_env.h | `tensorflow/core/distributed_runtime/worker_env.h` | ソース | WorkerEnv構造体（Worker環境設定） |
| base_rendezvous_mgr.h | `tensorflow/core/distributed_runtime/base_rendezvous_mgr.h` | ソース | Rendezvous管理（テンソル送受信） |
| cancellable_call.h | `tensorflow/core/distributed_runtime/cancellable_call.h` | ソース | キャンセル可能なgRPC呼び出し |
| recent_request_ids.h | `tensorflow/core/distributed_runtime/recent_request_ids.h` | ソース | リクエスト重複検出 |
| cluster_function_library_runtime.h | `tensorflow/core/distributed_runtime/cluster_function_library_runtime.h` | ソース | クロスプロセス関数呼び出し |
