# 機能設計書 116-共通ランタイム

## 概要

本ドキュメントは、TensorFlowにおけるセッション管理・エグゼキュータ・メモリアロケータなどの共通ランタイムの設計を記述する。

### 本機能の処理概要

TensorFlowの計算グラフを実行するための共通ランタイム基盤を提供する。DirectSession（ローカル実行）、Executor（グラフ実行エンジン）、デバイス管理、メモリアロケータ、コレクティブ実行など、TensorFlowの実行基盤の中核を構成する。

**業務上の目的・背景**：TensorFlowのすべての計算は共通ランタイム上で実行される。ローカル実行、分散実行、Eager実行のいずれにおいても、デバイス管理、メモリ割り当て、Op実行の基盤として機能する。高効率な計算グラフの実行はMLワークロードのパフォーマンスに直結する。

**機能の利用シーン**：あらゆるTensorFlow操作の実行基盤として暗黙的に利用される。tf.Session（TF1.x）によるグラフ実行、tf.function（TF2.x）によるトレースされたグラフの実行、デバイス配置の管理などに関わる。

**主要な処理内容**：
1. DirectSession - ローカルマシンでの計算グラフ実行セッション
2. Executor - 計算グラフのノードを順次実行するエンジン
3. SessionFactory - セッションインスタンスの生成ファクトリ
4. デバイス管理（DeviceMgr） - CPU/GPU/TPUデバイスの管理
5. メモリ管理（BFCAllocator等） - GPU/CPUメモリの割り当て・解放
6. コレクティブ実行 - AllReduce等のコレクティブ操作の実行管理

**関連システム・外部連携**：分散ランタイム（No.115）、TFRTランタイム（No.117）、Eager実行（No.87）、XLAコンパイラ（No.75）と連携する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 該当なし | - | 本機能はバックエンドランタイム機能 |

## 機能種別

ランタイムインフラストラクチャ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| GraphDef | protobuf | Yes | 実行する計算グラフ定義 | - |
| SessionOptions | protobuf | No | セッション設定（デバイス、メモリ等） | - |
| RunOptions | protobuf | No | 実行オプション（トレース、タイムアウト等） | - |
| feed_dict | map<string, Tensor> | No | 入力テンソルのフィード | - |
| fetch_names | list<string> | No | 取得するテンソル名リスト | - |

### 入力データソース

TensorFlow Python API（tf.Session.run等）からの呼び出し

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output_tensors | list<Tensor> | 実行結果のテンソルリスト |
| run_metadata | RunMetadata | 実行メタデータ（コスト、統計等） |
| status | Status | 操作の成功/失敗ステータス |

### 出力先

呼び出し元への戻り値として返却

## 処理フロー

### 処理シーケンス

```
1. セッション作成
   └─ SessionFactory::NewSessionで適切なセッション実装を選択
   └─ DirectSessionを生成（ローカル実行の場合）

2. グラフ拡張
   └─ GraphDefをSessionにExtend
   └─ グラフ最適化（Grappler）の適用
   └─ デバイス配置の決定

3. 実行（DirectSession::Run）
   └─ フィードテンソルの設定
   └─ Executorの生成・実行
   └─ ノードの依存関係に基づく順序実行
   └─ フェッチテンソルの収集

4. メモリ管理
   └─ BFCAllocatorによるGPUメモリの効率的割り当て
   └─ メモリ再利用とフラグメンテーション低減
```

### フローチャート

```mermaid
flowchart TD
    A[クライアント] -->|Session.run| B[DirectSession]
    B --> C[グラフ最適化]
    C --> D[デバイス配置]
    D --> E[Executor生成]
    E --> F[ノード実行]
    F --> G{全ノード完了?}
    G -->|No| F
    G -->|Yes| H[結果返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-116-01 | デバイスコロケーション | 同一グラフ内のOp間でのデバイス配置制約 | グラフ構築時 |
| BR-116-02 | メモリ再利用 | BFCAllocatorによるメモリプールからの再利用優先 | メモリ割り当て時 |
| BR-116-03 | コスト分析 | Executorは実行コストを収集し、将来の最適化に活用 | コストモデル有効時 |

### 計算ロジック

該当なし（ランタイムインフラストラクチャ）

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ResourceExhausted | メモリ不足 | GPU/CPUメモリが不足 | バッチサイズ削減、メモリ設定変更 |
| InvalidArgument | グラフエラー | 不正なグラフ定義 | グラフ定義を修正 |
| Internal | 実行エラー | Op実行中のエラー | エラーメッセージに基づきデバッグ |
| Cancelled | キャンセル | CancellationManagerによるキャンセル | 正常動作 |

### リトライ仕様

セッションレベルでのリトライは呼び出し元の責任。

## トランザクション仕様

該当なし

## パフォーマンス要件

Executorはデータフロー駆動で非同期にノードを実行し、デバイス間のパイプライン並列を最大化する。BFCAllocatorはメモリフラグメンテーションを最小化する。

## セキュリティ考慮事項

特になし。ローカル実行基盤のため、セキュリティ制御はOS/プロセスレベルに依存。

## 備考

- DirectSessionはTF1.x系のSession APIで使用される。TF2.xではEager実行がデフォルトだが、tf.functionによるグラフ実行時にも共通ランタイムが利用される。
- BFCAllocator（Best-Fit with Coalescing）はGPUメモリ管理の中核。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: セッション管理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | direct_session.h | `tensorflow/core/common_runtime/direct_session.h` | DirectSessionクラスのインタフェース |
| 1-2 | session_factory.h | `tensorflow/core/common_runtime/session_factory.h` | SessionFactoryによるセッション生成パターン |

**読解のコツ**: DirectSessionはSessionインタフェースを実装し、ローカルマシンでの計算グラフ実行を担当する。SessionFactoryパターンにより、ローカル/分散の実行モードを透過的に切り替えられる。

#### Step 2: グラフ実行エンジンを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | executor.h | `tensorflow/core/common_runtime/executor.h` | Executorインタフェース - グラフ実行の中核 |
| 2-2 | executor.cc | `tensorflow/core/common_runtime/executor.cc` | Executor実装 - ノードのスケジューリングと実行 |

#### Step 3: メモリ管理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | bfc_allocator.h | `tensorflow/core/common_runtime/bfc_allocator.h` | BFC（Best-Fit with Coalescing）アロケータ |

### プログラム呼び出し階層図

```
Python (tf.Session.run / tf.function)
    │
    ├─ DirectSession
    │      ├─ Extend() - グラフ拡張
    │      ├─ Run() - グラフ実行
    │      │      ├─ CreateExecutors()
    │      │      │      └─ NewLocalExecutor()
    │      │      └─ RunInternal()
    │      │             └─ Executor::RunAsync()
    │      └─ Close() - セッション終了
    │
    ├─ DeviceMgr - デバイス管理
    │
    └─ BFCAllocator - メモリ管理
           ├─ AllocateRaw()
           └─ DeallocateRaw()
```

### データフロー図

```
[入力]                    [処理]                         [出力]

GraphDef      ──────▶ DirectSession::Extend ──────▶ 内部グラフ構造
feed_dict     ──────▶ DirectSession::Run ─────────▶ output_tensors
SessionOptions ─────▶                                  + run_metadata
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| direct_session.h/.cc | `tensorflow/core/common_runtime/direct_session.h` | ソース | ローカル実行セッション |
| executor.h/.cc | `tensorflow/core/common_runtime/executor.h` | ソース | グラフ実行エンジン |
| executor_factory.h/.cc | `tensorflow/core/common_runtime/executor_factory.h` | ソース | Executorファクトリ |
| session_factory.h/.cc | `tensorflow/core/common_runtime/session_factory.h` | ソース | セッションファクトリ |
| session.cc | `tensorflow/core/common_runtime/session.cc` | ソース | Sessionベースクラス |
| bfc_allocator.h | `tensorflow/core/common_runtime/bfc_allocator.h` | ソース | BFCメモリアロケータ |
| collective_executor_mgr.h/.cc | `tensorflow/core/common_runtime/collective_executor_mgr.h` | ソース | コレクティブ実行管理 |
| colocation_graph.h | `tensorflow/core/common_runtime/colocation_graph.h` | ソース | デバイスコロケーション制約グラフ |
| buf_rendezvous.h/.cc | `tensorflow/core/common_runtime/buf_rendezvous.h` | ソース | バッファRendezvous |
