# 機能設計書 121-候補サンプリング操作

## 概要

本ドキュメントは、TensorFlowにおける候補サンプリング操作（Candidate Sampling Operations）の機能設計を記述する。NCE（Noise Contrastive Estimation）やサンプルドソフトマックスなどの近似学習手法に必要な負例候補のサンプリング機能を提供する。

### 本機能の処理概要

**業務上の目的・背景**：大規模なクラス数（数万〜数百万クラス）を持つ分類問題では、全クラスに対するソフトマックス計算が計算量的に非現実的となる。候補サンプリング操作は、学習時に全クラスの一部のみをサンプリングすることで、計算コストを大幅に削減しながらモデルを効率的に学習するための基盤機能である。

**機能の利用シーン**：自然言語処理におけるWord2Vecの学習（Negative Sampling）、大規模語彙のLanguage Modelにおけるサンプルドソフトマックス、推薦システムにおけるNCE損失の計算など、出力クラス数が非常に大きいモデルの学習時に利用される。

**主要な処理内容**：
1. 指定された分布（一様分布、対数一様分布、ユニグラム分布など）に基づく負例候補のサンプリング
2. 正解クラスおよびサンプリングされた候補の期待出現回数の計算
3. 偶発的なヒット（正解と一致するサンプル）の検出と除外

**関連システム・外部連携**：TensorFlowのNN操作（nn_ops）と連携し、NCE損失やサンプルドソフトマックス損失の計算に使用される。

**権限による制御**：特になし。全ユーザが利用可能な計算操作である。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はバックエンド計算操作であり、直接関連する画面はない |

## 機能種別

計算処理（候補サンプリングおよび期待値計算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| true_classes | int64 (2D) | Yes | 正解クラスIDの行列 [batch_size, num_true] | 2次元テンソルであること |
| num_true | int | Yes | 各サンプルの正解クラス数 | >= 1 |
| num_sampled | int | Yes | サンプリングする候補数 | >= 1 |
| unique | bool | Yes | 重複なしサンプリングを行うか | - |
| range_max | int | Yes (一部Op) | サンプリング範囲の最大値（クラス数） | >= 1 |
| seed / seed2 | int | No | 乱数シード | デフォルト 0 |
| vocab_file | string | No | FixedUnigramの語彙ファイルパス | デフォルト空文字 |
| distortion | float | No | FixedUnigramの歪みパラメータ | デフォルト 1.0 |
| unigrams | list(float) | No | FixedUnigramのユニグラム確率リスト | デフォルト空リスト |

### 入力データソース

プログラム内部で生成されるテンソルデータ。学習ループ内で正解ラベルが入力として渡される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| sampled_candidates | int64 [num_sampled] | サンプリングされた候補クラスID |
| true_expected_count | float [batch_size, num_true] | 正解クラスの期待サンプリング回数 |
| sampled_expected_count | float [num_sampled] | サンプリング候補の期待サンプリング回数 |

### 出力先

後続のNCE損失計算またはサンプルドソフトマックス損失計算への入力テンソル。

## 処理フロー

### 処理シーケンス

```
1. サンプリング分布の初期化
   └─ 指定された分布タイプ（一様、対数一様、ユニグラム等）に応じたサンプラーの構築
2. 候補のサンプリング
   └─ num_sampled個の候補をサンプリング（uniqueフラグに応じて重複制御）
3. 期待値の計算
   └─ 正解クラスおよびサンプリング候補の期待出現回数を計算
4. 結果の出力
   └─ sampled_candidates, true_expected_count, sampled_expected_countを返却
```

### フローチャート

```mermaid
flowchart TD
    A[入力: true_classes] --> B{サンプラータイプ判定}
    B -->|Uniform| C[一様分布サンプリング]
    B -->|LogUniform| D[対数一様分布サンプリング]
    B -->|Unigram| E[ユニグラム分布サンプリング]
    B -->|All| F[全候補返却]
    C --> G[期待出現回数計算]
    D --> G
    E --> G
    F --> G
    G --> H[出力: sampled_candidates, expected_counts]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-121-1 | サンプリング範囲制約 | サンプリングされる候補IDは0からrange_max-1の範囲内 | 全サンプラー |
| BR-121-2 | 一意性保証 | unique=trueの場合、サンプリング結果に重複なし | unique属性がtrue |
| BR-121-3 | 対数一様分布特性 | 低いIDほど高い確率でサンプリングされる（Zipf分布近似） | LogUniformCandidateSampler |
| BR-121-4 | FixedUnigram語彙指定 | vocab_fileまたはunigramsのいずれかで分布を指定 | FixedUnigramCandidateSampler |

### 計算ロジック

- **一様分布**: P(class) = 1 / range_max
- **対数一様分布**: P(class) = (log(class + 2) - log(class + 1)) / log(range_max + 1)
- **ユニグラム分布**: P(class) = count(class)^distortion / sum(count(all)^distortion)
- **期待出現回数**: unique=trueの場合は複雑な組み合わせ計算、falseの場合はnum_sampled * P(class)

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。全ての処理はインメモリのテンソル演算である。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgument | 入力検証エラー | true_classesが2次元でない | 正しい形状のテンソルを入力する |
| InvalidArgument | 入力検証エラー | num_trueとtrue_classesの列数が一致しない | パラメータを一致させる |
| InvalidArgument | パラメータエラー | num_sampled < 1 または range_max < 1 | 正の整数を指定する |

### リトライ仕様

リトライは不要。決定論的な計算（seedが固定の場合）。

## トランザクション仕様

トランザクション管理なし。ステートフルOp（乱数状態を保持）であるが、トランザクション制御は行わない。

## パフォーマンス要件

サンプリング操作は全クラスに対するソフトマックス計算のO(range_max)をO(num_sampled)に削減することが目的。num_sampledはrange_maxに比べて十分小さい値を設定する。

## セキュリティ考慮事項

特になし。計算グラフ内部で使用される演算Opであり、外部入出力は発生しない。

## 備考

- 6種類のサンプラーOp: UniformCandidateSampler, LogUniformCandidateSampler, LearnedUnigramCandidateSampler, ThreadUnsafeUnigramCandidateSampler, FixedUnigramCandidateSampler, AllCandidateSampler
- ComputeAccidentalHitsは正解とサンプル候補の偶発的一致を検出するユーティリティOp

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Op定義の入出力テンソル形状と属性の定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | candidate_sampling_ops.cc | `tensorflow/core/ops/candidate_sampling_ops.cc` | Op登録マクロによる入出力定義と属性定義 |

**読解のコツ**: TensorFlowのREGISTER_OPマクロは、Opの入出力テンソル型・属性・形状推論関数を宣言する。`.Input()`、`.Output()`、`.Attr()`チェーンで仕様を読み取れる。

#### Step 2: エントリーポイントを理解する

Shape推論関数がOp定義のエントリーポイントとなる。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | candidate_sampling_ops.cc | `tensorflow/core/ops/candidate_sampling_ops.cc` | CandidateSamplerShapeFn関数 |

**主要処理フロー**:
1. **27-42行目**: CandidateSamplerShapeFn - 全サンプラーOpに共通の形状推論関数。num_sampledとnum_trueから出力形状を決定
2. **46-58行目**: UniformCandidateSampler - 一様分布サンプラーの登録
3. **60-72行目**: LogUniformCandidateSampler - 対数一様分布サンプラーの登録
4. **74-86行目**: LearnedUnigramCandidateSampler - 学習済みユニグラムサンプラーの登録
5. **102-120行目**: FixedUnigramCandidateSampler - 固定ユニグラムサンプラー（追加属性あり）
6. **122-133行目**: AllCandidateSampler - 全候補返却サンプラー
7. **135-164行目**: ComputeAccidentalHits - 偶発的ヒット検出Op

#### Step 3: 共通形状推論ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | candidate_sampling_ops.cc | `tensorflow/core/ops/candidate_sampling_ops.cc` | CandidateSamplerShapeFn |

**主要処理フロー**:
- **29行目**: num_sampled属性を取得
- **31行目**: num_true属性を取得
- **34行目**: true_classesが2次元であることを検証
- **35行目**: batch_sizeをtrue_classesの第0次元から取得
- **37-41行目**: 3つの出力形状を設定: [num_sampled], [batch_size, num_true], [num_sampled]

### プログラム呼び出し階層図

```
candidate_sampling_ops.cc (Op定義)
    |
    +-- CandidateSamplerShapeFn (形状推論: 全サンプラー共通)
    |
    +-- REGISTER_OP("UniformCandidateSampler")
    +-- REGISTER_OP("LogUniformCandidateSampler")
    +-- REGISTER_OP("LearnedUnigramCandidateSampler")
    +-- REGISTER_OP("ThreadUnsafeUnigramCandidateSampler")
    +-- REGISTER_OP("FixedUnigramCandidateSampler")
    +-- REGISTER_OP("AllCandidateSampler")
    +-- REGISTER_OP("ComputeAccidentalHits")
```

### データフロー図

```
[入力]                      [処理]                           [出力]

true_classes ──────────▶ CandidateSampler ──────────▶ sampled_candidates
[batch_size, num_true]     (分布に基づくサンプリング)        [num_sampled]
                                |
                                +──────────────────▶ true_expected_count
                                |                       [batch_size, num_true]
                                |
                                +──────────────────▶ sampled_expected_count
                                                        [num_sampled]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| candidate_sampling_ops.cc | `tensorflow/core/ops/candidate_sampling_ops.cc` | ソース | Op定義（入出力・属性・形状推論） |
| shape_inference.h | `tensorflow/core/framework/shape_inference.h` | ヘッダ | 形状推論フレームワーク |
| op.h | `tensorflow/core/framework/op.h` | ヘッダ | REGISTER_OPマクロ定義 |
