# 機能設計書 124-バッチ操作

## 概要

本ドキュメントは、TensorFlowにおけるバッチ操作（Batch Operations）の機能設計を記述する。リクエストの動的バッチング処理を提供し、複数の推論リクエストを自動的にまとめてバッチ処理することで、GPUやTPUの利用効率を最大化する。

### 本機能の処理概要

**業務上の目的・背景**：推論サービングにおいて、個別のリクエストを1件ずつ処理するとGPU/TPUの並列処理能力を十分に活用できない。動的バッチングにより、複数のリクエストを自動的にまとめてバッチ化し、スループットを向上させる。TensorFlow Servingでの推論パフォーマンス最適化の核心機能である。

**機能の利用シーン**：TensorFlow Servingによるオンライン推論サービスにおいて、複数のクライアントからの推論リクエストを動的にバッチ化して処理する場面。また、`@tf.nondifferentiable_batch_function`デコレータを使用して、Pythonレベルでバッチ化された推論関数を定義する場面で使用される。

**主要な処理内容**：
1. BatchFunction Op: 複数リクエストのテンソルを第1軸に沿って結合し、指定された関数をバッチ実行
2. Batch Op: 入力テンソルを動的にバッチ化し、バッチインデックスとIDを付与
3. Unbatch Op: バッチ化されたテンソルを個々のリクエストに分離
4. UnbatchGrad Op: アンバッチ操作の勾配計算
5. 優先度付きバッチング（高優先度・低優先度の混合処理ポリシー）

**関連システム・外部連携**：TensorFlow Serving、gRPCベースの推論サーバ、TPUバッチスケジューラと連携。

**権限による制御**：特になし。全ユーザが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はバックエンドバッチング操作であり、直接関連する画面はない |

## 機能種別

データ連携 / 計算処理（リクエストの動的バッチング）

## 入力仕様

### 入力パラメータ

#### BatchFunction Op

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| in_tensors | Tin | Yes | バッチ化する入力テンソル群 | - |
| captured_tensors | Tcaptured | Yes | キャプチャされたテンソル群 | - |
| f | func | Yes | バッチ実行する関数 | - |
| num_batch_threads | int | Yes | バッチ処理スレッド数 | - |
| max_batch_size | int | Yes | 最大バッチサイズ | - |
| batch_timeout_micros | int | Yes | バッチタイムアウト（マイクロ秒） | - |
| max_enqueued_batches | int | No | 最大キュー深度 | デフォルト 10 |
| allowed_batch_sizes | list(int) | No | 許可されたバッチサイズリスト | デフォルト空リスト |
| enable_large_batch_splitting | bool | No | 大バッチ分割の有効化 | デフォルト false |
| low_priority_max_batch_size | int | No | 低優先度の最大バッチサイズ | デフォルト 0 |
| mixed_priority_policy | string | No | 混合優先度ポリシー | デフォルト low_priority_padding_with_max_batch_size |
| batch_padding_policy | string | No | パディングポリシー | デフォルト PAD_UP |

#### Batch Op

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| in_tensors | T | Yes | バッチ化する入力テンソル群 | - |
| num_batch_threads | int | Yes | バッチ処理スレッド数 | - |
| max_batch_size | int | Yes | 最大バッチサイズ | - |
| batch_timeout_micros | int | Yes | バッチタイムアウト | - |
| grad_timeout_micros | int | Yes | 勾配計算タイムアウト | - |

### 入力データソース

推論リクエストから供給される入力テンソル。

## 出力仕様

### 出力データ

#### BatchFunction Op

| 項目名 | 型 | 説明 |
|--------|-----|------|
| out_tensors | Tout | バッチ実行結果の出力テンソル群 |

#### Batch Op

| 項目名 | 型 | 説明 |
|--------|-----|------|
| batched_tensors | T | バッチ化されたテンソル群 |
| batch_index | int64 [unknown, 3] | バッチ内の各リクエストのインデックス情報 |
| id | int64 (scalar) | バッチの一意識別子 |

### 出力先

BatchFunction Opの出力は各リクエストに分配。Batch Opの出力は処理関数への入力として使用。

## 処理フロー

### 処理シーケンス

```
1. リクエストのキューイング
   └─ 到着した推論リクエストをバッチキューに追加
2. バッチ形成条件の判定
   └─ max_batch_size到達またはbatch_timeout_micros経過で発火
3. テンソルのバッチ化
   └─ 複数リクエストのテンソルを第1軸に沿って結合
4. バッチ処理の実行
   └─ 指定された関数fをバッチ化されたテンソルで実行
5. 結果のアンバッチ
   └─ バッチ結果を個々のリクエストに分離して返却
```

### フローチャート

```mermaid
flowchart TD
    A[リクエスト到着] --> B[バッチキューに追加]
    B --> C{バッチ形成条件?}
    C -->|max_batch_size到達| D[バッチ化]
    C -->|timeout経過| D
    C -->|待機中| B
    D --> E{large_batch_splitting?}
    E -->|Yes| F[バッチ分割]
    E -->|No| G[バッチ処理実行]
    F --> G
    G --> H[結果アンバッチ]
    H --> I[各リクエストへ応答]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-124-1 | バッチサイズ制限 | max_batch_sizeを超える入力は無効化 | 常時 |
| BR-124-2 | パディング | allowed_batch_sizesが指定された場合、次の許可サイズまでパディング | allowed_batch_sizesが非空 |
| BR-124-3 | 大バッチ分割 | enable_large_batch_splitting=trueで大きな入力を分割実行 | 属性が有効時 |
| BR-124-4 | 優先度制御 | mixed_priority_policyにより高/低優先度の混合処理を制御 | 低優先度パラメータ指定時 |
| BR-124-5 | パディングポリシー | PAD_UP/BATCH_DOWN/MINIMIZE_TPU_COST_PER_REQUESTの3種 | batch_padding_policy指定時 |

### 計算ロジック

- **バッチ化**: 第1軸に沿ったtf.concat相当の操作
- **アンバッチ**: batch_indexに基づく第1軸に沿ったスライシング
- **MINIMIZE_TPU_COST_PER_REQUEST**: batch_cost(size) / real_request_countを最小化する貪欲法

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgument | 入力エラー | 入力がTensor以外 | Tensorのみを渡す |
| DeadlineExceeded | タイムアウト | timeout_micros超過でバッチ未完成 | タイムアウト値を調整 |
| ResourceExhausted | キュー満杯 | max_enqueued_batches超過 | キュー深度を増やすかリクエストレートを調整 |

### リトライ仕様

タイムアウト時は不完全なバッチ（現在キューにあるリクエストのみ）で処理を実行。

## トランザクション仕様

トランザクション管理なし。SetIsDistributedCommunication()により分散通信Opとしてマークされる。

## パフォーマンス要件

バッチングによりGPU/TPUの並列処理能力を最大限活用し、スループットを向上させる。batch_timeout_microsとmax_batch_sizeのバランスがレイテンシとスループットのトレードオフを決定する。

## セキュリティ考慮事項

特になし。ただし、異なるクライアントのリクエストが同一バッチ内で処理されるため、サイドチャネル攻撃への考慮が必要な場合がある。

## 備考

- Python APIの`@tf.nondifferentiable_batch_function`デコレータが高レベルインターフェースを提供
- SparseTensorはサポートされていない
- 戻り値はTensorまたはTensorのリスト/タプルのみ

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

バッチングの入出力テンソルとOp属性を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | batch_ops.cc | `tensorflow/core/ops/batch_ops.cc` | 4つのOp定義（BatchFunction, Batch, Unbatch, UnbatchGrad）の入出力と属性 |

**読解のコツ**: `.SetIsDistributedCommunication()`は分散環境での特別な扱いを示す。`mixed_priority_policy`の4種類のポリシーはコメント（55-79行目）で詳細に説明されている。

#### Step 2: PythonラッパーAPIを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | batch_ops.py | `tensorflow/python/ops/batch_ops.py` | batch_functionデコレータの実装 |

**主要処理フロー**:
1. **28-35行目**: batch_functionデコレータの定義とパラメータ
2. **86-98行目**: decorator内部 - 入力関数をtf.functionでトレースし、concrete_functionを取得
3. **100-117行目**: gen_batch_ops.batch_functionの呼び出し - C++ Opへの委譲
4. **118-119行目**: nest.pack_sequence_asによる出力の再構造化

#### Step 3: C++ Op定義の詳細を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | batch_ops.cc | `tensorflow/core/ops/batch_ops.cc` | BatchFunction Op |
| 3-2 | batch_ops.cc | `tensorflow/core/ops/batch_ops.cc` | Batch/Unbatch/UnbatchGrad Op |

**主要処理フロー**:
- **22-108行目**: BatchFunction Op - 最も機能豊富なバッチングOp。優先度制御、パディングポリシー、大バッチ分割をサポート
- **110-141行目**: Batch Op - 基本的なバッチ化Op。batch_indexとidを出力
- **143-158行目**: Unbatch Op - バッチテンソルを個別テンソルに分離
- **160-172行目**: UnbatchGrad Op - アンバッチの勾配計算

### プログラム呼び出し階層図

```
batch_ops.py (Python API)
    |
    +-- batch_function() デコレータ
    |     +-- def_function.function() ... tf.functionトレース
    |     +-- gen_batch_ops.batch_function() ... C++ Opへ委譲
    |     +-- nest.pack_sequence_as() ... 出力再構造化
    |
batch_ops.cc (C++ Op定義)
    |
    +-- REGISTER_OP("BatchFunction")   ... 関数ベースバッチング
    +-- REGISTER_OP("Batch")           ... テンソルバッチ化
    +-- REGISTER_OP("Unbatch")         ... アンバッチ
    +-- REGISTER_OP("UnbatchGrad")     ... 勾配計算
```

### データフロー図

```
[入力]                    [処理]                       [出力]

リクエスト1 ──┐
リクエスト2 ──┼──▶ Batch/BatchFunction ──────▶ バッチ化テンソル
リクエスト3 ──┘      (動的バッチング)              batch_index, id
                           |
                           v
バッチ化テンソル ──▶ 推論関数f(batch) ──────▶ バッチ結果
                           |
                           v
バッチ結果 ──────▶ Unbatch ──────────────▶ 個別結果1, 2, 3
(batch_index)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| batch_ops.cc | `tensorflow/core/ops/batch_ops.cc` | ソース | C++ Op定義（4つのOp） |
| batch_ops.py | `tensorflow/python/ops/batch_ops.py` | ソース | Python APIラッパー（デコレータ） |
| gen_batch_ops.py | `tensorflow/python/ops/gen_batch_ops.py` | 自動生成 | 自動生成されたPythonバインディング |
| common_shape_fns.h | `tensorflow/core/framework/common_shape_fns.h` | ヘッダ | 共通形状推論 |
| op.h | `tensorflow/core/framework/op.h` | ヘッダ | REGISTER_OPマクロ |
