# 機能設計書 129-混合精度学習

## 概要

本ドキュメントは、TensorFlowにおける混合精度学習（Mixed Precision Training）の機能設計を記述する。FP16/BF16とFP32を組み合わせた混合精度学習のポリシー管理、自動キャスト変数、損失スケーリングを提供する。

### 本機能の処理概要

**業務上の目的・背景**：ディープラーニングモデルの学習において、全ての演算をFP32で行うと計算量とメモリ使用量が大きくなる。混合精度学習は、計算をFP16/BF16で行いつつ変数をFP32で保持することで、メモリ使用量を削減しGPUのTensorCoreを活用した高速化を実現する。特にNVIDIA Volta世代以降のGPU（compute capability 7.0以上）で大きな性能向上が得られる。

**機能の利用シーン**：大規模モデル（画像分類、自然言語処理、音声認識等）の学習において、GPUメモリの制約を緩和しつつ学習速度を向上させたい場合に使用される。`tf.keras.mixed_precision.set_global_policy('mixed_float16')`で全レイヤーに一括適用可能。

**主要な処理内容**：
1. Policy: dtype policy（計算dtype・変数dtype）の定義と管理
2. グローバルポリシーの設定・取得（set_global_policy / global_policy）
3. AutoCastVariable: FP32変数を計算時に自動的にFP16/BF16にキャスト
4. LossScaleOptimizer: 損失スケーリングによるFP16アンダーフロー防止
5. デバイス互換性チェック（compute capability 7.0以上の確認）

**関連システム・外部連携**：Kerasレイヤーシステム、オプティマイザ、GPUデバイス管理。

**権限による制御**：特になし。全ユーザが利用可能。ただしGPUのcompute capabilityにより性能効果が異なる。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はバックエンド学習最適化機能であり、直接関連する画面はない |

## 機能種別

計算処理 / 設定管理（混合精度ポリシーの管理と適用）

## 入力仕様

### 入力パラメータ

#### Policy

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | string | Yes | ポリシー名 | mixed_float16/mixed_bfloat16/dtype名 |

#### set_global_policy

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| policy | Policy or string | Yes | 設定するポリシー | 浮動小数点型のみ |

#### LossScaleOptimizer

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| optimizer | Optimizer | Yes | ラップ対象のオプティマイザ | - |
| loss_scale | LossScale/int/float/string | No | 損失スケール | デフォルト 'dynamic' |

### 入力データソース

プログラム設定による指定。レイヤーコンストラクタのdtype引数。

## 出力仕様

### 出力データ

#### Policy

| 項目名 | 型 | 説明 |
|--------|-----|------|
| compute_dtype | string | 計算に使用するdtype（例: float16） |
| variable_dtype | string | 変数のdtype（例: float32） |
| name | string | ポリシー名 |

### 出力先

Kerasレイヤーのdtype_policyプロパティ、グローバルポリシー設定。

## 処理フロー

### 処理シーケンス

```
1. ポリシーの設定
   └─ set_global_policy('mixed_float16')でグローバルポリシーを設定
2. デバイス互換性チェック
   └─ GPU compute capability >= 7.0の確認（警告ログ出力）
3. レイヤー構築時の適用
   └─ add_weightでFP32変数を作成し、AutoCastVariableでラップ
4. 計算時の自動キャスト
   └─ Layer.call実行時にAutoCastVariableがFP16/BF16にキャスト
5. 損失スケーリング
   └─ LossScaleOptimizerが勾配のスケーリング・アンスケーリングを管理
```

### フローチャート

```mermaid
flowchart TD
    A[set_global_policy mixed_float16] --> B[デバイス互換性チェック]
    B --> C[レイヤー構築]
    C --> D[変数作成: FP32 + AutoCastVariable]
    D --> E[forward pass: FP16計算]
    E --> F[損失計算]
    F --> G[LossScaleOptimizer]
    G --> H{勾配が有限?}
    H -->|Yes| I[損失スケール解除 + 重み更新]
    H -->|No| J[スケール縮小 + ステップスキップ]
    I --> K[次のイテレーション]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-129-1 | mixed_float16 | 計算dtype=float16、変数dtype=float32 | name='mixed_float16' |
| BR-129-2 | mixed_bfloat16 | 計算dtype=bfloat16、変数dtype=float32 | name='mixed_bfloat16' |
| BR-129-3 | 自動LossScale | Model.compileがmixed_float16の場合自動でLossScaleOptimizerをラップ | mixed_float16ポリシー |
| BR-129-4 | GPU要件 | compute capability 7.0以上で高速動作 | mixed_float16 |
| BR-129-5 | 浮動小数点制約 | set_global_policyには浮動小数点型のみ設定可能 | グローバルポリシー設定時 |
| BR-129-6 | V2動作要件 | TF2またはV2 dtype behavior有効時のみ利用可能 | set_global_policy |

### 計算ロジック

- **AutoCast**: FP32変数をenable_auto_cast_variablesコンテキスト内でFP16/BF16に自動キャスト
- **動的損失スケーリング**: スケール値を動的に調整（NaN/Inf検出時に縮小、安定時に拡大）
- **固定損失スケーリング**: ユーザ指定のスケール値で固定

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | 型エラー | name引数にDType型を渡した | DType.nameプロパティを使用 |
| ValueError | 値エラー | 未知のポリシー名 | mixed_float16/mixed_bfloat16/dtype名を使用 |
| ValueError | 値エラー | 非浮動小数点型のグローバルポリシー設定 | 浮動小数点型を指定 |
| ValueError | 値エラー | _float32_vars接尾辞付きポリシー名（削除済み） | 新しいポリシー名を使用 |
| ValueError | 競合エラー | mixed_precision_graph_rewriteと併用 | どちらか一方のみ使用 |

### リトライ仕様

リトライ不要。設定操作。

## トランザクション仕様

トランザクション管理なし。グローバルステートの変更は即座に反映。

## パフォーマンス要件

- mixed_float16: GPU compute capability 7.0以上で2-3倍の学習速度向上（TensorCore活用）
- mixed_bfloat16: TPUでの標準的な精度設定
- メモリ使用量: FP16/BF16により約50%削減

## セキュリティ考慮事項

特になし。

## 備考

- PolicyV1はdeprecated（loss_scaleフィールドを持つ旧版）
- policy_scope()コンテキストマネージャでスコープ付きポリシー設定が可能
- `_infer`ポリシーはTF1互換性のための内部用
- get_layer_policy.pyでレイヤーからポリシーを取得可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Policyクラスのcompute_dtypeとvariable_dtypeの関係が混合精度の核心。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | policy.py | `tensorflow/python/keras/mixed_precision/policy.py` | Policyクラスの定義と_parse_name |

**読解のコツ**: Policyクラスはnameからcompute_dtypeとvariable_dtypeをパースする。`mixed_float16` -> (float16, float32)、`mixed_bfloat16` -> (bfloat16, float32)、`float32` -> (float32, float32)のマッピングが`_parse_name`メソッドで行われる。

#### Step 2: グローバルポリシー管理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | policy.py | `tensorflow/python/keras/mixed_precision/policy.py` | global_policy, set_global_policy |

**主要処理フロー**:
1. **185-192行目**: Policy.__init__ - name文字列からcompute_dtype/variable_dtypeをパース
2. **196-242行目**: _parse_name - ポリシー名のパース。mixed_float16/mixed_bfloat16の特別処理
3. **409-438行目**: global_policy - グローバルポリシーの取得。未設定時はfloatx()から生成
4. **457-511行目**: set_global_policy - グローバルポリシーの設定。浮動小数点型チェック、graph_rewrite競合チェック

#### Step 3: AutoCastVariableを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | autocast_variable.py | `tensorflow/python/keras/mixed_precision/autocast_variable.py` | AutoCastVariableクラス |

**主要処理フロー**:
- **33行目**: _autocast_dtype - スレッドローカルな自動キャスト先dtype
- **49-70行目**: AutoCastVariable - FP32変数をラップし、enable_auto_cast_variablesコンテキスト内で自動キャスト
- **72-80行目**: __init__ - 浮動小数点リソース変数のみ受け入れ

#### Step 4: デバイス互換性チェックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | device_compatibility_check.py | `tensorflow/python/keras/mixed_precision/device_compatibility_check.py` | GPUのcompute capability検証 |

**主要処理フロー**:
- **57-127行目**: _log_device_compatibility_check - GPU compute capability 7.0以上の確認とログ出力
- **132-147行目**: log_device_compatibility_check - 初回のみ互換性チェックを実行（グローバルフラグ管理）

#### Step 5: 損失スケーリングを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | loss_scale_optimizer.py | `tensorflow/python/keras/mixed_precision/loss_scale_optimizer.py` | LossScaleOptimizer |
| 5-2 | loss_scale.py | `tensorflow/python/keras/mixed_precision/loss_scale.py` | Keras用損失スケールユーティリティ |

**主要処理フロー**:
- **65-75行目**: _is_all_finite - 全勾配が有限値かチェック
- **95-99行目**: _assign_if_finite - 有限値の場合のみ変数に代入
- **25-58行目** (loss_scale.py): serialize/deserialize/get - 損失スケールオブジェクトの管理

### プログラム呼び出し階層図

```
policy.py (ポリシー管理)
    |
    +-- Policy クラス
    |     +-- _parse_name() ... mixed_float16 → (float16, float32)
    |     +-- compute_dtype / variable_dtype プロパティ
    |
    +-- set_global_policy() / global_policy()
    |     +-- _check_if_mixed_precision_graph_rewrite_is_enabled()
    |
autocast_variable.py (自動キャスト)
    |
    +-- AutoCastVariable クラス
    |     +-- __init__() ... FP32変数をラップ
    |     +-- (Layer.call内で自動的にFP16にキャスト)
    |
device_compatibility_check.py (GPU互換性)
    |
    +-- log_device_compatibility_check()
    |     +-- _log_device_compatibility_check()
    |           +-- compute_capability >= (7, 0) を検証
    |
loss_scale_optimizer.py (損失スケーリング)
    |
    +-- LossScaleOptimizer クラス
    |     +-- _is_all_finite() ... 勾配の有限性チェック
    |     +-- _assign_if_finite() ... 条件付き変数更新
    |
loss_scale.py (損失スケールユーティリティ)
    +-- get() / serialize() / deserialize()
```

### データフロー図

```
[設定]                    [学習ループ]                  [結果]

set_global_policy ──▶ Layer構築 ──────────────▶ AutoCastVariable
('mixed_float16')      (FP32変数作成)              (FP32 → FP16自動キャスト)
                             |
                             v
                       forward pass (FP16) ──▶ loss (FP16)
                             |
                             v
                       LossScaleOptimizer
                       (スケーリング・勾配チェック)
                             |
                             v
                       backward pass ──────▶ 重み更新 (FP32)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| policy.py | `tensorflow/python/keras/mixed_precision/policy.py` | ソース | Policyクラスとグローバルポリシー管理 |
| autocast_variable.py | `tensorflow/python/keras/mixed_precision/autocast_variable.py` | ソース | AutoCastVariable（自動dtype変換） |
| loss_scale_optimizer.py | `tensorflow/python/keras/mixed_precision/loss_scale_optimizer.py` | ソース | LossScaleOptimizer（損失スケーリング） |
| loss_scale.py | `tensorflow/python/keras/mixed_precision/loss_scale.py` | ソース | Keras用損失スケールユーティリティ |
| device_compatibility_check.py | `tensorflow/python/keras/mixed_precision/device_compatibility_check.py` | ソース | GPUデバイス互換性チェック |
| get_layer_policy.py | `tensorflow/python/keras/mixed_precision/get_layer_policy.py` | ソース | レイヤーからのポリシー取得 |
| test_util.py | `tensorflow/python/keras/mixed_precision/test_util.py` | テスト | テストユーティリティ |
| __init__.py | `tensorflow/python/keras/mixed_precision/__init__.py` | ソース | パッケージ初期化 |
