# 機能設計書 132-Kerasコールバック

## 概要

本ドキュメントは、TensorFlowのKerasにおけるコールバック機構（Callbacks）の機能設計について記載する。

### 本機能の処理概要

Kerasコールバックは、モデルの訓練・評価・推論の各段階でユーザ定義の処理を差し込むためのフック機構を提供する。

**業務上の目的・背景**：ディープラーニングモデルの訓練は長時間かつ複雑なプロセスであり、訓練中のモデル状態の監視・制御が重要となる。コールバック機構により、早期終了、チェックポイント保存、学習率調整、ログ記録などの横断的な関心事を訓練ループにプラグイン形式で組み込むことができる。

**機能の利用シーン**：モデルのfit/evaluate/predict呼び出し時に、エポック開始・終了、バッチ開始・終了などの各イベントでカスタム処理を実行する。過学習防止のための早期終了、最良モデルの自動保存、TensorBoardへのログ出力、学習率の動的調整など多様な用途で使用される。

**主要な処理内容**：
1. CallbackListによる複数コールバックの一括管理・呼び出し
2. Callback基底クラスのフックメソッド（on_epoch_begin/end, on_batch_begin/end等）
3. EarlyStoppingによる訓練の早期終了判定
4. ModelCheckpointによるモデルの定期保存
5. TensorBoardによるログの可視化出力
6. LearningRateSchedulerによる学習率の動的調整
7. ReduceLROnPlateauによるプラトー検出時の学習率減衰
8. CSVLoggerによるCSV形式のログ出力
9. BackupAndRestoreによる訓練状態のバックアップ・復元

**関連システム・外部連携**：TensorBoard（外部可視化ツール）、RemoteMonitor（HTTPリクエストによるリモート通知）、チェックポイント管理（No.64）、プロファイラ（No.98）と連携する。

**権限による制御**：特になし。全てのユーザが利用可能。

## 関連画面

本機能は画面機能マッピングにおいて直接的な関連画面は定義されていない。プログラミングAPIとして利用される。

## 機能種別

イベント処理 / 監視・制御

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| callbacks | list[Callback] | No | コールバックインスタンスのリスト | 各要素がCallbackインスタンスであること |
| monitor | str | No | 監視する評価指標名（EarlyStopping等） | モデルのメトリクス名と一致すること |
| patience | int | No | 改善なしで許容するエポック数 | 非負整数 |
| filepath | str | No | チェックポイント保存先パス（ModelCheckpoint） | 有効なファイルパス |
| log_dir | str | No | TensorBoardログ出力先ディレクトリ | 有効なディレクトリパス |

### 入力データソース

model.fit()の訓練ループから自動的に呼び出される。logs辞書（損失値・評価指標値）がイベントハンドラに渡される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| History.history | dict | エポックごとの損失・評価指標の履歴 |
| ModelCheckpoint | ファイル | 保存されたモデルファイル（HDF5/SavedModel） |
| TensorBoard logs | ファイル | TensorBoard用のイベントファイル |
| CSVLogger output | ファイル | CSV形式の訓練ログ |

### 出力先

メモリ（History）、ファイルシステム（ModelCheckpoint, TensorBoard, CSVLogger）、HTTP（RemoteMonitor）。

## 処理フロー

### 処理シーケンス

```
1. CallbackList初期化
   └─ コールバックリストの構築、デフォルトコールバック（History, ProgbarLogger）追加

2. on_train_begin
   └─ 全コールバックの訓練開始処理

3. エポックループ
   ├─ on_epoch_begin
   ├─ バッチループ
   │   ├─ on_train_batch_begin
   │   ├─ [訓練ステップ実行]
   │   └─ on_train_batch_end
   ├─ on_epoch_end
   │   ├─ EarlyStopping: 改善判定
   │   ├─ ModelCheckpoint: 保存判定
   │   ├─ ReduceLROnPlateau: 学習率調整判定
   │   └─ TensorBoard: メトリクス書き出し
   └─ model.stop_training判定

4. on_train_end
   └─ 全コールバックの訓練終了処理
```

### フローチャート

```mermaid
flowchart TD
    A[model.fit 開始] --> B[CallbackList.on_train_begin]
    B --> C{エポックループ}
    C --> D[on_epoch_begin]
    D --> E{バッチループ}
    E --> F[on_train_batch_begin]
    F --> G[train_step実行]
    G --> H[on_train_batch_end]
    H --> E
    E -->|全バッチ完了| I[on_epoch_end]
    I --> J{stop_training?}
    J -->|No| C
    J -->|Yes| K[on_train_end]
    C -->|全エポック完了| K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-132-01 | デフォルトコールバック | History とProgbarLoggerは常に追加される | model.fit呼び出し時 |
| BR-132-02 | EarlyStopping判定 | patience回連続で改善がない場合に訓練停止 | EarlyStopping使用時 |
| BR-132-03 | ModelCheckpoint保存条件 | save_best_only=Trueの場合、モニタリング指標が改善した時のみ保存 | ModelCheckpoint使用時 |
| BR-132-04 | バッチフック最適化 | バッチフックが不要なコールバックのみの場合、バッチフック呼び出しをスキップ | パフォーマンス最適化 |

### 計算ロジック

EarlyStoppingの改善判定：`mode='min'`の場合は`current < best - min_delta`、`mode='max'`の場合は`current > best + min_delta`で改善を判定する。

## データベース操作仕様

本機能にデータベース操作はない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | 設定エラー | monitorに存在しないメトリクス名を指定 | メトリクス名をモデルのcompile設定と一致させる |
| TerminateOnNaN | NaN検出 | 訓練中にlossがNaNになった場合 | 学習率の調整、データの確認 |
| RuntimeError | IO エラー | ModelCheckpointの保存先にアクセスできない場合 | ファイルパス・権限の確認 |

### リトライ仕様

BackupAndRestoreコールバックにより、訓練が中断された場合に最後のバックアップ状態から復元・再開が可能。

## トランザクション仕様

該当なし。

## パフォーマンス要件

CallbackListではバッチフック呼び出しのオーバーヘッドを最小化するため、バッチフックの有無を事前チェックし、不要な場合はフック呼び出しをスキップする最適化が実装されている（callbacks.py 244-249行目）。

## セキュリティ考慮事項

- RemoteMonitorは外部HTTPサーバへデータを送信するため、送信先の信頼性を確認すること
- ModelCheckpointの保存先パスに適切なアクセス制御を設定すること

## 備考

- ParameterServerStrategy使用時はバッチレベルのコールバックが制限される
- カスタムコールバックはCallback基底クラスを継承して作成する
- コールバックのタイミングチェック機能により、バッチ処理に対してコールバックが遅い場合に警告が出力される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | callbacks.py | `tensorflow/python/keras/callbacks.py` | Callback基底クラス（591行目）のフックメソッド定義 |

**読解のコツ**: Callbackクラスのon_*メソッドは全てデフォルトで何もしない（pass）。サブクラスで必要なメソッドのみをオーバーライドする Template Method パターン。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | callbacks.py | `tensorflow/python/keras/callbacks.py` | CallbackListクラス（201行目）がコールバックの一括管理を担当 |

**主要処理フロー**:
1. **201-260行目**: CallbackList.__init__ - コールバックリストの初期化、デフォルトコールバック追加、パフォーマンス最適化フラグ設定
2. **244-249行目**: バッチフック呼び出しの最適化判定
3. **263-280行目**: _add_default_callbacks - History/ProgbarLoggerの自動追加

#### Step 3: 各コールバック実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | callbacks.py | `tensorflow/python/keras/callbacks.py` | 各コールバッククラスの実装 |

**主要処理フロー**:
- **902行目**: BaseLogger - バッチメトリクスの平均計算
- **950行目**: TerminateOnNaN - NaN検出による訓練停止
- **968行目**: ProgbarLogger - プログレスバー表示
- **1134行目**: History - 訓練履歴の記録
- **1173行目**: ModelCheckpoint - モデルの定期保存
- **1588行目**: BackupAndRestore - 訓練状態のバックアップ
- **1701行目**: EarlyStopping - 早期終了
- **1844行目**: RemoteMonitor - リモート通知
- **1906行目**: LearningRateScheduler - 学習率スケジューリング
- **2016行目**: TensorBoard - TensorBoard連携
- **2585行目**: ReduceLROnPlateau - プラトー検出時の学習率減衰
- **2708行目**: CSVLogger - CSV出力
- **2790行目**: LambdaCallback - ラムダ式ベースコールバック

### プログラム呼び出し階層図

```
model.fit() (training.py)
    |
    +-- CallbackList (callbacks.py:201)
    |       |
    |       +-- on_train_begin()
    |       +-- on_epoch_begin()
    |       +-- on_train_batch_begin()
    |       +-- on_train_batch_end()
    |       +-- on_epoch_end()
    |       +-- on_train_end()
    |
    +-- Callback subclasses
            |
            +-- BaseLogger (902) - バッチメトリクス平均
            +-- ProgbarLogger (968) - プログレスバー
            +-- History (1134) - 履歴記録
            +-- ModelCheckpoint (1173) - モデル保存
            +-- EarlyStopping (1701) - 早期終了
            +-- TensorBoard (2016) - 可視化ログ
            +-- ReduceLROnPlateau (2585) - 学習率減衰
            +-- CSVLogger (2708) - CSVログ
            +-- LambdaCallback (2790) - カスタム処理
```

### データフロー図

```
[入力]                    [処理]                          [出力]

model.fit()の       ---> CallbackList.on_*()        ---> model.stop_training (bool)
訓練ループ                |                                |
                         +-> EarlyStopping          ---> stop_training = True
logs (dict)         ---> |                                |
                         +-> ModelCheckpoint        ---> saved model files
                         |                                |
                         +-> TensorBoard            ---> event files
                         |                                |
                         +-> History                ---> history dict
                         |                                |
                         +-> CSVLogger              ---> CSV file
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| callbacks.py | `tensorflow/python/keras/callbacks.py` | ソース | 全コールバッククラスの定義 |
| training.py | `tensorflow/python/keras/engine/training.py` | ソース | コールバックを呼び出すModel.fitの実装 |
| distributed_file_utils.py | `tensorflow/python/keras/distribute/distributed_file_utils.py` | ソース | 分散環境でのファイル操作ユーティリティ |
| worker_training_state.py | `tensorflow/python/keras/distribute/worker_training_state.py` | ソース | ワーカー訓練状態管理 |
| profiler_v2.py | `tensorflow/python/profiler/profiler_v2.py` | ソース | TensorBoardコールバックで使用されるプロファイラ |
