# 機能設計書 133-Keras初期化子

## 概要

本ドキュメントは、TensorFlowのKerasにおける重み初期化子（Initializers）の機能設計について記載する。

### 本機能の処理概要

Keras初期化子は、ニューラルネットワークの重みパラメータの初期値を生成するための各種初期化手法を提供する。

**業務上の目的・背景**：ニューラルネットワークの訓練収束性能は重みの初期値に大きく依存する。不適切な初期化は勾配消失・勾配爆発を引き起こし、訓練が失敗する原因となる。Glorot、He、Lecunなどの理論に基づく初期化手法を提供することで、モデルの安定した訓練を支援する。

**機能の利用シーン**：Kerasレイヤーの`kernel_initializer`、`bias_initializer`パラメータにて、レイヤー生成時に重みの初期化方法を指定する際に使用される。デフォルトでは`glorot_uniform`が使用される。

**主要な処理内容**：
1. Initializer基底クラスによる共通インターフェース（`__call__`、`get_config`、`from_config`）の定義
2. 定数初期化（Zeros、Ones、Constant）
3. ランダム初期化（RandomUniform、RandomNormal、TruncatedNormal）
4. 分散スケーリング初期化（GlorotUniform、GlorotNormal、HeUniform、HeNormal、LecunUniform、LecunNormal）
5. 直交初期化（Orthogonal）、恒等初期化（Identity）
6. VarianceScalingによる汎用的な分散調整初期化
7. シリアライゼーション対応（get_config/from_config）

**関連システム・外部連携**：コア層（No.24）、畳み込み層（No.15）、リカレント層（No.21）など、全てのKerasレイヤーの重み初期化に使用される。

**権限による制御**：特になし。全てのユーザが利用可能。

## 関連画面

本機能は画面機能マッピングにおいて直接的な関連画面は定義されていない。

## 機能種別

計算処理 / テンソル生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| shape | tuple | Yes | 生成するテンソルの形状 | 正の整数のタプル |
| dtype | DType | No | 生成するテンソルのデータ型（デフォルト: floatx()） | 数値型またはブール型 |
| seed | int | No | 乱数シード（再現性確保用） | 整数値 |
| minval | float | No | 一様分布の下限（RandomUniform） | maxval未満 |
| maxval | float | No | 一様分布の上限（RandomUniform） | minval超 |
| mean | float | No | 正規分布の平均（RandomNormal） | 実数値 |
| stddev | float | No | 正規分布の標準偏差（RandomNormal） | 正の実数 |
| scale | float | No | VarianceScalingのスケーリング係数 | 正の実数 |
| mode | str | No | VarianceScalingのモード（fan_in/fan_out/fan_avg） | 指定値のいずれか |
| distribution | str | No | VarianceScalingの分布（truncated_normal/uniform/untruncated_normal） | 指定値のいずれか |

### 入力データソース

Kerasレイヤーのbuild()メソッドから呼び出される。レイヤーの入力形状に基づいてshapeが決定される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| tensor | Tensor | 指定された形状・分布で初期化されたテンソル |

### 出力先

レイヤーのカーネル・バイアス変数の初期値として使用される。

## 処理フロー

### 処理シーケンス

```
1. 初期化子インスタンスの生成
   └─ パラメータ（seed, minval, maxval等）の設定

2. __call__(shape, dtype)の呼び出し
   └─ バリデーション（dtype検証、shape検証）
   └─ パーティション形状の処理（分散学習用）

3. テンソルの生成
   └─ 各初期化手法に応じた乱数生成・定数生成
   └─ fan_in/fan_outの計算（VarianceScaling系）

4. テンソルの返却
   └─ 生成されたテンソルをレイヤーのadd_weight()に返す
```

### フローチャート

```mermaid
flowchart TD
    A[Layer.build] --> B[add_weight with initializer]
    B --> C[initializer.__call__ shape, dtype]
    C --> D{初期化手法}
    D -->|Zeros/Ones| E[constant_op.constant]
    D -->|RandomUniform| F[random_ops.random_uniform]
    D -->|RandomNormal| G[random_ops.random_normal]
    D -->|GlorotUniform| H[VarianceScaling fan_avg, uniform]
    D -->|HeNormal| I[VarianceScaling fan_in, truncated_normal]
    D -->|Orthogonal| J[QR分解による直交行列生成]
    E --> K[初期化済みテンソル]
    F --> K
    G --> K
    H --> K
    I --> K
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-133-01 | デフォルトdtype | dtypeが未指定の場合、`backend.floatx()`（デフォルトfloat32）を使用 | dtype=None時 |
| BR-133-02 | string型排除 | Zeros/Onesはstring型をサポートしない | dtype=string時にValueError |
| BR-133-03 | パーティション形状 | 分散学習時はパーティション形状・オフセットがkwargsで渡される | 分散学習環境 |
| BR-133-04 | VarianceScaling計算 | fan_in/fan_outはカーネル形状から算出。fan_in=receptive_field*input_channels | VarianceScaling系使用時 |

### 計算ロジック

VarianceScalingの分散計算:
- `mode='fan_in'`: `scale / fan_in`
- `mode='fan_out'`: `scale / fan_out`
- `mode='fan_avg'`: `scale / ((fan_in + fan_out) / 2)`

Glorot初期化: `VarianceScaling(scale=1.0, mode='fan_avg', distribution='uniform')`
He初期化: `VarianceScaling(scale=2.0, mode='fan_in', distribution='truncated_normal')`

## データベース操作仕様

本機能にデータベース操作はない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | 型エラー | Zeros/Onesでstring型を指定 | 数値型を使用する |
| ValueError | 型エラー | RandomUniformで浮動小数点でも整数でもない型を指定 | 対応する型を使用する |
| ValueError | 不正引数 | VarianceScalingで不正なmode/distributionを指定 | 有効な値を指定する |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

初期化はモデルビルド時に一度だけ実行される。大規模モデルでは多数のレイヤーの初期化が行われるが、各初期化操作自体は軽量である。

## セキュリティ考慮事項

- seedを固定することで再現可能な初期化を実現できるが、セキュリティ用途の乱数生成には不適
- `_RandomGenerator`クラスがステートレスおよびステートフルな乱数生成を管理

## 備考

- TF2とTF1で初期化子の実装が分離されている（initializers_v2.py / initializers_v1.py）
- カスタム初期化子はInitializer基底クラスを継承して`__call__`と`get_config`を実装する
- `partition_shape`と`partition_offset`はモデル並列化（パーティション化）で使用される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | initializers_v2.py | `tensorflow/python/keras/initializers/initializers_v2.py` | Initializer基底クラス（34行目）の__call__/get_config/from_configインターフェース |

**読解のコツ**: 各初期化子クラスは`__call__(shape, dtype, **kwargs)`を実装する。kwargsには分散学習用の`partition_shape`と`partition_offset`が含まれる場合がある。

#### Step 2: 定数初期化子を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | initializers_v2.py | `tensorflow/python/keras/initializers/initializers_v2.py` | Zeros（115行目）、Ones（151行目）、Constant（187行目） |

**主要処理フロー**:
- **131-148行目**: Zeros.__call__ - dtype検証後にarray_ops.zerosでテンソル生成
- **167-184行目**: Ones.__call__ - dtype検証後にarray_ops.onesでテンソル生成
- **213-226行目**: Constant.__call__ - constant_op.constantで定数テンソル生成

#### Step 3: ランダム初期化子を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | initializers_v2.py | `tensorflow/python/keras/initializers/initializers_v2.py` | RandomUniform（232行目）、RandomNormal（292行目）、TruncatedNormal |

**主要処理フロー**:
- **257-282行目**: RandomUniform.__call__ - _RandomGeneratorを使用した一様分布乱数生成
- **292行目以降**: RandomNormal.__call__ - 正規分布乱数生成

#### Step 4: VarianceScaling系初期化子を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | initializers_v2.py | `tensorflow/python/keras/initializers/initializers_v2.py` | VarianceScalingクラスとその派生（GlorotUniform, HeNormal等） |

**主要処理フロー**:
- VarianceScalingがscale/mode/distributionに基づいて分散を計算
- GlorotUniform/GlorotNormal/HeUniform/HeNormal/LecunUniform/LecunNormalはVarianceScalingのプリセット

### プログラム呼び出し階層図

```
Layer.build() / add_weight()
    |
    +-- Initializer.__call__(shape, dtype)
            |
            +-- Zeros -> array_ops.zeros()
            +-- Ones -> array_ops.ones()
            +-- Constant -> constant_op.constant()
            +-- RandomUniform -> _RandomGenerator.random_uniform()
            +-- RandomNormal -> _RandomGenerator.random_normal()
            +-- TruncatedNormal -> _RandomGenerator.truncated_normal()
            +-- VarianceScaling -> _compute_fans() -> random distribution
            |       +-- GlorotUniform (fan_avg, uniform)
            |       +-- GlorotNormal (fan_avg, truncated_normal)
            |       +-- HeUniform (fan_in, uniform)
            |       +-- HeNormal (fan_in, truncated_normal)
            |       +-- LecunUniform (fan_in, uniform)
            |       +-- LecunNormal (fan_in, truncated_normal)
            +-- Orthogonal -> gen_linalg_ops.qr()
            +-- Identity -> linalg_ops.eye()
```

### データフロー図

```
[入力]                    [処理]                          [出力]

shape (tuple)       ---> Initializer.__call__()     ---> initialized Tensor
dtype (DType)            |
seed (int)               +-> _validate_kwargs()
                         +-> _get_dtype()
                         +-> random_ops / array_ops
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| initializers_v2.py | `tensorflow/python/keras/initializers/initializers_v2.py` | ソース | TF2向け初期化子の実装 |
| initializers_v1.py | `tensorflow/python/keras/initializers/initializers_v1.py` | ソース | TF1向け初期化子（互換性） |
| __init__.py | `tensorflow/python/keras/initializers/__init__.py` | ソース | モジュール公開定義 |
| random_ops.py | `tensorflow/python/ops/random_ops.py` | ソース | 乱数生成オペレーション |
| stateless_random_ops.py | `tensorflow/python/ops/stateless_random_ops.py` | ソース | ステートレス乱数生成 |
| array_ops.py | `tensorflow/python/ops/array_ops.py` | ソース | zeros/ones等のテンソル生成 |
| linalg_ops.py | `tensorflow/python/ops/linalg_ops.py` | ソース | QR分解等（Orthogonal初期化用） |
