# 機能設計書 135-スコープドアロケータ

## 概要

本ドキュメントは、TensorFlowのスコープドアロケータ（Scoped Allocator）操作の機能設計について記載する。

### 本機能の処理概要

スコープドアロケータは、複数のテンソルのメモリを単一のバッキングテンソルから割り当てることで、メモリの効率的な再利用を実現する内部最適化機構である。

**業務上の目的・背景**：大規模なニューラルネットワークの訓練・推論ではメモリ使用量が重要なボトルネックとなる。スコープドアロケータは、関連する複数のテンソルを単一の連続メモリ領域から割り当てることで、メモリフラグメンテーションを削減し、AllReduceなどの集約操作でのメモリコピーを最小化する。

**機能の利用シーン**：TensorFlowランタイムの内部最適化として、Grappler最適化パスの一部として自動的に適用される。直接的なユーザ呼び出しは想定されていない（実験的な内部操作）。

**主要な処理内容**：
1. `_ScopedAllocator`オペレーション：バッキングテンソル（1D連続メモリ）の割り当て
2. `_ScopedAllocatorConcat`オペレーション：バッキングテンソルからの複数テンソルの読み取り専用参照結合
3. `_ScopedAllocatorSplit`オペレーション：結合テンソルからの個別テンソルへの分割参照

**関連システム・外部連携**：Grappler最適化（No.84）、コレクティブ操作（No.71）、NCCL集約操作（No.126）と連携する。

**権限による制御**：内部操作のため、ユーザ権限制御は不要。

## 関連画面

本機能は画面機能マッピングにおいて関連画面は定義されていない。

## 機能種別

メモリ管理 / 内部最適化

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| shapes | list[shape] | Yes | 割り当てるテンソルのシェイプリスト（_ScopedAllocator） | 有効なシェイプのリスト |
| shape | shape | Yes | 出力テンソルのシェイプ（バッキングテンソル） | 有効なシェイプ |
| T | type | Yes | テンソルのデータ型 | TensorFlowデータ型 |
| sa_name | string | Yes | ScopedAllocatorのノード名 | 文字列 |
| id | int | Yes | scope_id（非負整数） | 非負整数 |
| expected_call_count | int | Yes | 期待される割り当て呼び出し回数 | 正整数 |
| N | int | Yes | テンソル数（Concat/Split時）| 2以上の整数 |
| reshape | bool | No | 出力形状を変更するか（デフォルト: false） | ブール値 |

### 入力データソース

TensorFlowランタイムによる内部呼び出し。Grappler最適化パスにより計算グラフに挿入される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output (Allocator) | Tensor[T] | バッキングテンソルへの参照 |
| output (Concat) | Tensor[T] | 結合されたテンソル（バッキングテンソルの読み取り専用参照） |
| output (Split) | list[Tensor[T]] | 分割された個別テンソルのリスト |

### 出力先

TensorFlowの計算グラフ内の後続オペレーションへの入力。

## 処理フロー

### 処理シーケンス

```
1. _ScopedAllocator
   └─ shapes/shape/T/sa_name/id/expected_call_countに基づいてバッキングテンソルを割り当て
   └─ ScopedAllocatorMgrにバッキングテンソルを登録

2. 下流Opsの実行
   └─ 各Opがバッキングテンソルの部分領域に出力をエイリアス

3. _ScopedAllocatorConcat
   └─ バッキングテンソルとエイリアステンソルを入力として受け取り
   └─ バッキングテンソルの読み取り専用参照を出力

4. _ScopedAllocatorSplit
   └─ 結合テンソルを受け取り、個別のエイリアステンソルを出力
```

### フローチャート

```mermaid
flowchart TD
    A[_ScopedAllocator] --> B[バッキングテンソル割り当て]
    B --> C[下流Ops: バッキングテンソルの部分領域にエイリアス出力]
    C --> D[_ScopedAllocatorConcat]
    D --> E[結合テンソル 読み取り専用参照]
    E --> F[AllReduce等の集約操作]
    F --> G[_ScopedAllocatorSplit]
    G --> H[個別テンソルへの分割]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-135-01 | 内部操作限定 | 全てのScopedAllocator Opsは実験的な内部操作であり、不正使用は未定義動作 | 常時 |
| BR-135-02 | ステートフル | 全てのScopedAllocator OpsはIsStatefulとマークされている | 常時 |
| BR-135-03 | Concat入力制約 | Concatのinputsはバッキングテンソルのエイリアスでなければならない | _ScopedAllocatorConcat使用時 |
| BR-135-04 | Split入力制約 | Splitのsplitリストはバッキングテンソルからのエイリアス | _ScopedAllocatorSplit使用時 |

### 計算ロジック

バッキングテンソルは全ての個別テンソルを1D配列として連結した形状を持つ。各テンソルはバッキングテンソルの連続する部分領域へのエイリアスとなる。

## データベース操作仕様

本機能にデータベース操作はない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| Internal Error | ランタイムエラー | expected_call_countと実際の呼び出し回数が不一致 | グラフ最適化の検証 |
| Internal Error | ランタイムエラー | エイリアステンソルがバッキングテンソルの領域外を参照 | グラフ最適化の検証 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

スコープドアロケータの主目的がパフォーマンス最適化であるため、メモリ割り当てのオーバーヘッドは最小限に抑えられている。AllReduceなどの集約操作でメモリコピーを回避することで、通信帯域幅の効率的な利用を実現する。

## セキュリティ考慮事項

- 内部操作であり、直接的なユーザ入力は受け付けないため、セキュリティリスクは限定的
- 不正使用（unsafe ways）の可能性がドキュメントに明記されている

## 備考

- 全てのオペレーション名がアンダースコア（`_`）で始まり、内部操作であることを示す
- Grappler最適化のScopedAllocatorOptimizationパスにより自動挿入される
- 主にAllReduce操作のメモリ効率化に使用される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | scoped_allocator_ops.cc | `tensorflow/core/ops/scoped_allocator_ops.cc` | 3つのOp定義（_ScopedAllocator, _ScopedAllocatorConcat, _ScopedAllocatorSplit） |

**読解のコツ**: `REGISTER_OP`マクロによるOp登録パターン。各OpのInput/Output/Attr定義とDoc文字列を確認する。

#### Step 2: Op定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | scoped_allocator_ops.cc | `tensorflow/core/ops/scoped_allocator_ops.cc` | Op登録の詳細 |

**主要処理フロー**:
- **21-49行目**: `_ScopedAllocator` - バッキングテンソルの割り当てOp。shapes, shape, T, sa_name, id, expected_call_count属性を持つ
- **51-83行目**: `_ScopedAllocatorConcat` - バッキングテンソルからの結合参照Op。backing入力 + N個のinputs
- **85-117行目**: `_ScopedAllocatorSplit` - 結合テンソルからの分割Op。concat入力 + N個のsplit入力

### プログラム呼び出し階層図

```
Grappler Optimization Pass
    |
    +-- ScopedAllocatorOptimizer
            |
            +-- _ScopedAllocator (scoped_allocator_ops.cc:21)
            |       +-- ScopedAllocatorMgr (メモリ管理)
            |
            +-- _ScopedAllocatorConcat (scoped_allocator_ops.cc:51)
            |       +-- バッキングテンソルの読み取り参照
            |
            +-- _ScopedAllocatorSplit (scoped_allocator_ops.cc:85)
                    +-- エイリアステンソルの分割出力
```

### データフロー図

```
[入力]                    [処理]                          [出力]

shapes, shape       ---> _ScopedAllocator          ---> backing tensor (1D)
                         |
backing + aliases   ---> _ScopedAllocatorConcat    ---> concatenated reference
                         |
concat + split      ---> _ScopedAllocatorSplit     ---> individual tensors
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| scoped_allocator_ops.cc | `tensorflow/core/ops/scoped_allocator_ops.cc` | ソース | Op定義（3つのOp登録） |
