# 機能設計書 138-Go言語バインディング

## 概要

本ドキュメントは、TensorFlowのGo言語バインディングの機能設計について記載する。

### 本機能の処理概要

Go言語バインディングは、Go言語からTensorFlowの機能を利用するためのCGo（C言語バインディング）ベースのAPIを提供する。

**業務上の目的・背景**：Go言語はクラウドインフラやマイクロサービスで広く使用されており、Go言語アプリケーションからTensorFlowモデルの推論を実行するニーズがある。C言語API（tensorflow/c/c_api.h）をCGoで橋渡しすることで、Goネイティブなインターフェースを提供する。

**機能の利用シーン**：Go言語で実装されたサーバサイドアプリケーションからSavedModelの読み込み・推論実行、テンソルの生成・操作、計算グラフの構築・実行など。

**主要な処理内容**：
1. Tensor型：多次元配列（テンソル）の生成・読み取り・型変換
2. Graph型：計算グラフの構築・操作の追加・インポート
3. Session型：計算グラフの実行（Run）・デバイス一覧取得
4. SavedModel型：SavedModel形式モデルの読み込み・推論
5. Operation型：グラフ内のオペレーション（ノード）の参照
6. Context型：Eager実行コンテキスト
7. Shape型：テンソル形状の表現
8. Signature型：SavedModelのシグネチャ情報

**関連システム・外部連携**：C言語API（No.141）を経由してTensorFlowランタイムと連携する。SavedModel保存（No.62）で生成されたモデルを読み込む。

**権限による制御**：特になし。

## 関連画面

本機能は画面機能マッピングにおいて直接的な関連画面は定義されていない。

## 機能種別

外部連携 / 言語バインディング

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| exportDir | string | Yes（LoadSavedModel） | SavedModelのエクスポートディレクトリ | 有効なディレクトリパス |
| tags | []string | Yes（LoadSavedModel） | モデルのタグ | 文字列スライス |
| options | *SessionOptions | No | セッションオプション | nilまたは有効なオプション |
| graph | *Graph | Yes（NewSession） | 計算グラフ | 有効なGraphインスタンス |
| value | interface{} | Yes（NewTensor） | テンソルの値 | プリミティブ型または多次元配列 |

### 入力データソース

Go言語プログラムからのAPI呼び出し。SavedModelファイル、Protocol Bufferシリアライズされたグラフ定義。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| *Tensor | Tensor | 生成されたテンソルまたは実行結果テンソル |
| *Session | Session | 計算セッション |
| *Graph | Graph | 計算グラフ |
| *SavedModel | SavedModel | 読み込まれたSavedModel（Session + Graph + Signatures） |
| []Device | Device | デバイス一覧 |
| error | error | エラー情報 |

### 出力先

Go言語プログラムのメモリ空間。

## 処理フロー

### 処理シーケンス

```
1. SavedModelの読み込み
   └─ LoadSavedModel(exportDir, tags, options)
       └─ C.TF_LoadSessionFromSavedModel呼び出し
       └─ MetaGraphDefのデシリアライズ
       └─ Signatureの生成

2. グラフの構築・操作
   └─ NewGraph() でグラフ生成
   └─ graph.Import(buf) でProtocol Bufferからインポート
   └─ graph.Operation(name) でオペレーション検索

3. セッションの作成・実行
   └─ NewSession(graph, options) でセッション生成
   └─ session.Run(feeds, fetches, targets) で実行
   └─ session.Close() でリソース解放

4. テンソルの操作
   └─ NewTensor(value) でテンソル生成
   └─ tensor.DataType() / tensor.Shape() で情報取得
   └─ tensor.Value() でGo値に変換
```

### フローチャート

```mermaid
flowchart TD
    A[Go Application] --> B{操作}
    B -->|モデル推論| C[LoadSavedModel]
    B -->|グラフ実行| D[NewGraph + NewSession]
    C --> E[savedModel.Session.Run]
    D --> F[session.Run feeds, fetches]
    E --> G[結果テンソル取得]
    F --> G
    G --> H[tensor.Value でGo値変換]
    H --> I[session.Close / リソース解放]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-138-01 | CGo依存 | 全てのAPI呼び出しはCGoを経由してC言語APIに委譲される | 常時 |
| BR-138-02 | リソース解放 | Session/GraphはClose()で明示的にリソース解放が必要 | オブジェクト使用後 |
| BR-138-03 | スレッドセーフ | Sessionは並行呼び出し（concurrent Run）をサポート | 複数goroutine使用時 |
| BR-138-04 | ファイナライザ | SessionにはGoのruntime.SetFinalizerが設定されている | GC時 |
| BR-138-05 | モデルエクスポート不可 | Go言語からのモデルエクスポート機能は未提供 | 常時 |

### 計算ロジック

特段の計算ロジックはない。TensorFlowランタイムへの委譲。

## データベース操作仕様

本機能にデータベース操作はない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| error | ステータスエラー | C言語APIがエラーステータスを返した場合 | エラーメッセージを確認 |
| error | 型変換エラー | 非対応のGo型をテンソルに変換しようとした場合 | 対応するプリミティブ型を使用 |
| error | IOエラー | SavedModelの読み込みに失敗した場合 | パス・権限を確認 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

CGoの呼び出しにはオーバーヘッドが存在する。高頻度のテンソル生成・解放はパフォーマンスに影響する可能性がある。バッチ処理による呼び出し回数の最小化を推奨。

## セキュリティ考慮事項

- C言語APIとの境界でのメモリ安全性はGoのruntime.SetFinalizerとC.free()により管理
- 信頼できないソースからのSavedModel/Protocol Bufferの読み込みは任意コード実行のリスクがある

## 備考

- Go言語バインディングはtensorflow/go/パッケージとして提供
- Androidプラットフォーム向けのビルド設定（android.go）が含まれる
- TensorFlowのバージョン情報はversion.goで管理される
- 現在のGoバインディングはモデルのエクスポート（保存）機能を持たず、他言語で作成されたモデルの読み込み・推論に特化している

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | tensor.go | `tensorflow/go/tensor.go` | DataType定義（42-78行目）、Tensor構造体 |
| 1-2 | shape.go | `tensorflow/go/shape.go` | Shape型の定義 |

**読解のコツ**: Go言語のCGoインターフェース（`import "C"`）に注目。C言語の`TF_*`関数を直接呼び出している。`#include "tensorflow/c/c_api.h"`がC言語APIとの接続点。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | saved_model.go | `tensorflow/go/saved_model.go` | LoadSavedModel関数（52行目）- 最も一般的な使用パターン |

**主要処理フロー**:
1. **52行目**: LoadSavedModel関数の定義
2. **53-67行目**: Cの文字列変換、タグ配列の準備
3. **68-72行目**: C.TF_LoadSessionFromSavedModel呼び出し
4. **78-84行目**: MetaGraphDefのデシリアライズとSignature生成
5. **89-91行目**: Session/Graphのラップとファイナライザ設定

#### Step 3: セッション実行を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | session.go | `tensorflow/go/session.go` | Session構造体（40-48行目）とNewSession（52-67行目） |

**主要処理フロー**:
- **40-48行目**: Session構造体（C.TF_Sessionへのポインタ + sync.WaitGroup/Mutex）
- **52-67行目**: NewSession - C.TF_NewSession呼び出しとファイナライザ設定
- Run()メソッドでfeeds/fetches/targetsを指定して実行

#### Step 4: グラフ操作を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | graph.go | `tensorflow/go/graph.go` | Graph構造体（52-54行目）とImport/Operation |

**主要処理フロー**:
- **52-54行目**: Graph構造体（C.TF_Graphへのポインタ）
- **56-72行目**: GraphImportOptions構造体
- Import()でProtocol Bufferからグラフをインポート

### プログラム呼び出し階層図

```
Go Application
    |
    +-- LoadSavedModel (saved_model.go:52)
    |       +-- C.TF_LoadSessionFromSavedModel
    |       +-- proto.Unmarshal (MetaGraphDef)
    |       +-- generateSignatures
    |       +-- return *SavedModel {Session, Graph, Signatures}
    |
    +-- NewSession (session.go:52)
    |       +-- C.TF_NewSession
    |       +-- runtime.SetFinalizer
    |
    +-- NewGraph (graph.go)
    |       +-- C.TF_NewGraph
    |
    +-- NewTensor (tensor.go)
    |       +-- C.TF_NewTensor / C.TF_AllocateTensor
    |
    +-- Session.Run
    |       +-- C.TF_SessionRun
    |
    +-- Session.Close
            +-- C.TF_CloseSession / C.TF_DeleteSession
```

### データフロー図

```
[入力]                    [処理]                          [出力]

Go values           ---> NewTensor()                ---> *Tensor (C memory)
(int, float,             |
 [][]float32, etc.)      +-> C.TF_NewTensor

SavedModel dir      ---> LoadSavedModel()          ---> *SavedModel
                         |                                {Session, Graph, Signatures}
                         +-> C.TF_LoadSessionFromSavedModel

feeds (*Tensor)     ---> Session.Run()             ---> []*Tensor (results)
fetches (Output)         |
                         +-> C.TF_SessionRun
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| tensor.go | `tensorflow/go/tensor.go` | ソース | Tensor型・DataType定義 |
| session.go | `tensorflow/go/session.go` | ソース | Session型（計算実行） |
| graph.go | `tensorflow/go/graph.go` | ソース | Graph型（計算グラフ） |
| saved_model.go | `tensorflow/go/saved_model.go` | ソース | SavedModel読み込み |
| operation.go | `tensorflow/go/operation.go` | ソース | Operation型（グラフノード） |
| context.go | `tensorflow/go/context.go` | ソース | Eager実行コンテキスト |
| shape.go | `tensorflow/go/shape.go` | ソース | Shape型（テンソル形状） |
| signature.go | `tensorflow/go/signature.go` | ソース | Signature型（モデルシグネチャ） |
| tensor_handle.go | `tensorflow/go/tensor_handle.go` | ソース | TensorHandle（Eager用） |
| status.go | `tensorflow/go/status.go` | ソース | ステータス管理 |
| attrs.go | `tensorflow/go/attrs.go` | ソース | オペレーション属性 |
| version.go | `tensorflow/go/version.go` | ソース | バージョン情報 |
| doc.go | `tensorflow/go/doc.go` | ソース | パッケージドキュメント |
| lib.go | `tensorflow/go/lib.go` | ソース | ライブラリ初期化 |
| android.go | `tensorflow/go/android.go` | ソース | Androidプラットフォーム設定 |
