# 機能設計書 141-C言語API

## 概要

本ドキュメントは、TensorFlowのC言語API（`tensorflow/c/`）の機能設計を記載する。C言語APIは、TensorFlowの計算グラフ構築、セッション実行、テンソル操作、カーネル登録などの機能を、C言語の安定したABI（Application Binary Interface）として外部言語バインディングに提供する低レベルインタフェースである。

### 本機能の処理概要

**業務上の目的・背景**：TensorFlowはPythonをメインの言語として提供しているが、Go、Java、Rust、Swift、Haskell等の他言語からTensorFlowを利用するニーズがある。C言語APIは、全ての主要プログラミング言語がFFI（Foreign Function Interface）を通じて呼び出し可能な共通インタフェースとして設計されている。これにより、各言語バインディングの開発が容易になり、TensorFlowのエコシステム拡大に寄与する。

**機能の利用シーン**：
- 他言語（Go、Java、Rust等）からTensorFlowの機能を呼び出す際のブリッジ層として利用
- C/C++アプリケーションに直接TensorFlowの推論機能を組み込む場合
- プラグインシステム（カスタムオペレーション、カスタムカーネル、ファイルシステムプラグイン）の登録
- SavedModelの読み込みと推論実行

**主要な処理内容**：
1. テンソルの生成・操作・削除（TF_Tensor系API）
2. 計算グラフの構築（TF_Graph、TF_Operation、TF_OperationDescription系API）
3. セッションの作成と実行（TF_Session、TF_SessionRun系API）
4. SavedModelの読み込み（TF_LoadSessionFromSavedModel）
5. カスタムオペレーション/カーネルの登録（TF_OpDefinitionBuilder、TF_KernelBuilder系API）
6. デバイス情報の取得（TF_DeviceList系API）
7. 分散学習サーバの管理（TF_Server系API）
8. 勾配計算（TF_AddGradients系API）
9. チェックポイントの読み取り（TF_CheckpointReader系API）
10. 形状推論機能（TF_ShapeInferenceContext系API）

**関連システム・外部連携**：TensorFlowのコアC++ランタイム、XLAコンパイラ、分散ランタイム（gRPC）、各種デバイスドライバ（CUDA、NNAPI等）

**権限による制御**：C言語APIレベルでの権限制御はなく、呼び出し元プロセスの実行権限に依存する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はライブラリAPIであり、画面との直接の関連はない |

## 機能種別

ライブラリAPI / FFIインタフェース / 計算処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| graph | TF_Graph* | Yes | 計算グラフオブジェクト | NULLチェック |
| session_options | TF_SessionOptions* | Yes | セッション設定 | NULLチェック |
| op_type | const char* | Yes | 操作の型名 | 登録済みOp名であること |
| oper_name | const char* | Yes | 操作のノード名 | グラフ内で一意であること |
| inputs | TF_Output* / TF_Tensor** | Yes | 入力テンソル/出力参照 | 型・形状の整合性 |
| run_options | TF_Buffer* | No | 実行オプション（RunOptions proto） | 有効なprotobufシリアライゼーション |
| export_dir | const char* | Yes (LoadSavedModel時) | SavedModelのパス | ディレクトリが存在すること |

### 入力データソース

- メモリ上のテンソルデータ
- ディスク上のSavedModelファイル
- ディスク上のチェックポイントファイル
- Protocol Bufferでシリアライズされた設定データ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output_values | TF_Tensor** | セッション実行結果のテンソル配列 |
| status | TF_Status* | 操作結果のステータス（TF_OK / エラーコード） |
| output_graph_def | TF_Buffer* | シリアライズされたGraphDef |
| output_op_def | TF_Buffer* | シリアライズされたOpDef |
| device_list | TF_DeviceList* | 利用可能デバイスのリスト |

### 出力先

- 呼び出し元プログラムのメモリ空間（テンソルデータ、ステータス情報）
- TF_Status経由のエラー情報

## 処理フロー

### 処理シーケンス

```
1. 初期化フェーズ
   └─ TF_NewGraph / TF_NewSessionOptions でグラフ・セッション設定を生成
2. グラフ構築フェーズ
   ├─ TF_NewOperation で演算ノードのDescriptionを作成
   ├─ TF_AddInput / TF_SetAttr* で入力・属性を設定
   └─ TF_FinishOperation でノードをグラフに追加
3. セッション作成フェーズ
   └─ TF_NewSession / TF_LoadSessionFromSavedModel でセッションを生成
4. 実行フェーズ
   └─ TF_SessionRun で入力テンソルを与えてグラフを実行し出力を取得
5. クリーンアップフェーズ
   └─ TF_CloseSession / TF_DeleteSession / TF_DeleteGraph で資源を解放
```

### フローチャート

```mermaid
flowchart TD
    A[グラフ生成 TF_NewGraph] --> B[オペレーション構築 TF_NewOperation]
    B --> C[入力/属性設定 TF_AddInput / TF_SetAttr*]
    C --> D[オペレーション完了 TF_FinishOperation]
    D --> E{追加ノードあり?}
    E -->|Yes| B
    E -->|No| F[セッション生成 TF_NewSession]
    F --> G[グラフ実行 TF_SessionRun]
    G --> H{エラー?}
    H -->|No| I[出力テンソル取得]
    H -->|Yes| J[TF_Status確認・エラーハンドリング]
    I --> K[クリーンアップ TF_DeleteSession / TF_DeleteGraph]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-141-01 | オブジェクト所有権 | 生成したオブジェクト（TF_Tensor, TF_Session等）は呼び出し側が明示的にDelete関数で解放する責任を持つ | 全API呼び出し |
| BR-141-02 | ステータスクリア | TF_Status*引数を持つ全API呼び出しは、成功時にステータスをクリアし、失敗時にエラー情報を設定する | TF_Status*引数がある関数 |
| BR-141-03 | スレッドセーフティ | TF_Graphは指定された方法で使用する限りスレッドセーフ。TF_SessionRunは同一セッションに対して並行呼び出し可能 | マルチスレッド環境 |
| BR-141-04 | NULL安全 | 全てのDelete関数はnullptrに対して安全に呼び出し可能 | リソース解放時 |

### 計算ロジック

C言語API自体は計算ロジックを持たず、TensorFlowコアランタイムのC++実装に処理を委譲する。

## データベース操作仕様

該当なし（本機能はインメモリ計算ライブラリであり、データベースとの直接連携はない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TF_INVALID_ARGUMENT | 引数不正 | 不正な引数（NULL、範囲外インデックス等） | 入力パラメータの検証 |
| TF_NOT_FOUND | 未検出 | 指定されたOp名/ノード名が見つからない | 名前の確認 |
| TF_ALREADY_EXISTS | 重複 | 同名のオペレーションが既にグラフに存在 | 一意な名前を使用 |
| TF_OUT_OF_RANGE | 範囲外 | テンソルインデックスが範囲外 | インデックス範囲の確認 |
| TF_INTERNAL | 内部エラー | ランタイム内部の予期しないエラー | ログ確認・再試行 |

### リトライ仕様

C言語APIレベルでのリトライ機構はない。リトライが必要な場合は呼び出し元で実装する。

## トランザクション仕様

該当なし。TF_SessionRunはアトミックに実行されるが、データベーストランザクションの概念は適用されない。

## パフォーマンス要件

- C言語ラッパー層のオーバーヘッドは最小限（関数ポインタ経由の単純なディスパッチ）
- テンソルデータのゼロコピー受け渡しをサポート（deallocator関数の指定による）
- セッション実行のパフォーマンスはC++ APIと同等

## セキュリティ考慮事項

- 信頼できないモデル（GraphDef / SavedModel）のロードは任意コード実行と同等のリスクがある
- TF_Serverは認証・暗号化を備えていないため、信頼できないネットワーク環境での使用は推奨されない
- プラグインライブラリの動的ロード（TF_LoadLibrary）はファイルシステム上の任意の共有ライブラリを読み込むため注意が必要

## 備考

- C言語APIは安定性を重視して設計されており、Python APIと比べて機能が限定的である
- Eager Execution関連のC APIは `tensorflow/c/eager/c_api.h` に別途定義されている
- 実験的APIは `tensorflow/c/c_api_experimental.h` に分離されている

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、C APIで使用されるオペークな構造体と公開データ型を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | c_api.h | `tensorflow/c/c_api.h` | TF_Graph, TF_Operation, TF_Session, TF_Output, TF_Input等の主要な型定義。30-75行目のAPI設計方針コメントが重要 |
| 1-2 | c_api_internal.h | `tensorflow/c/c_api_internal.h` | オペーク構造体の内部実装。C++のtensorflow::Graphやtensorflow::Session等への参照を保持 |
| 1-3 | tf_tensor.h | `tensorflow/c/tf_tensor.h` | TF_Tensor型の定義。テンソルの生成・データアクセス・削除API |
| 1-4 | tf_status.h | `tensorflow/c/tf_status.h` | TF_Status型の定義。エラーコードとメッセージの管理 |
| 1-5 | tf_datatype.h | `tensorflow/c/tf_datatype.h` | TF_DataType列挙型。データ型の定義 |

**読解のコツ**: C APIの全てのオブジェクトはオペークポインタとして渡される。内部実装を理解するには `c_api_internal.h` を参照する必要がある。`TF_CAPI_EXPORT extern` マクロはDLL/共有ライブラリからのエクスポートを示す。

#### Step 2: コアAPIエントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | c_api.h | `tensorflow/c/c_api.h` | 主要API関数のシグネチャ。グラフ構築（132-398行目）、セッション管理（1204-1326行目）、関数定義（1025-1183行目）、サーバ管理（1540-1571行目）の4つの主要カテゴリに分かれる |
| 2-2 | c_api.cc | `tensorflow/c/c_api.cc` | 各API関数のC++実装。C APIからC++コアランタイムへの変換・委譲処理 |

**主要処理フロー**:
1. **83行目**: TF_Version() - バージョン文字列の返却
2. **102-122行目**: TF_NewSessionOptions / TF_SetTarget / TF_SetConfig - セッション設定
3. **135-139行目**: TF_NewGraph / TF_DeleteGraph - グラフの生成・削除
4. **232-233行目**: TF_NewOperation - オペレーション構築開始
5. **398-399行目**: TF_FinishOperation - オペレーションをグラフに登録
6. **1212-1214行目**: TF_NewSession - セッション生成
7. **1273-1286行目**: TF_SessionRun - グラフ実行
8. **1230-1233行目**: TF_LoadSessionFromSavedModel - SavedModel読み込み

#### Step 3: カスタムオペレーション登録APIを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ops.h | `tensorflow/c/ops.h` | Op定義ビルダーAPI。TF_OpDefinitionBuilder系関数と形状推論コンテキスト。24-67行目の使用例コメントが参考になる |
| 3-2 | ops.cc | `tensorflow/c/ops.cc` | Op定義ビルダーの実装 |

**主要処理フロー**:
- **96-97行目**: TF_NewOpDefinitionBuilder - Opビルダー生成
- **105-106行目**: TF_RegisterOpDefinition - Op登録
- **143-144行目**: TF_OpDefinitionBuilderAddAttr - 属性追加
- **226-229行目**: TF_OpDefinitionBuilderSetShapeInferenceFunction - 形状推論関数設定

#### Step 4: カスタムカーネル登録APIを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | kernels.h | `tensorflow/c/kernels.h` | カーネルビルダーAPI。TF_KernelBuilder系関数、OpKernelContext経由の入出力操作 |
| 4-2 | kernels.cc | `tensorflow/c/kernels.cc` | カーネルビルダーの実装 |

**主要処理フロー**:
- **98-102行目**: TF_NewKernelBuilder - カーネルビルダー生成（create/compute/delete関数ポインタを指定）
- **138-140行目**: TF_RegisterKernelBuilder - カーネル登録
- **182-183行目**: TF_GetInput - カーネル計算時の入力テンソル取得
- **210-212行目**: TF_SetOutput - カーネル計算時の出力テンソル設定

#### Step 5: 実験的APIを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | c_api_experimental.h | `tensorflow/c/c_api_experimental.h` | XLAコンパイル設定、チェックポイント読み取り、形状推論、プラグインデバイスロード等の実験的API |

### プログラム呼び出し階層図

```
呼び出し元（Go/Java/Rust等の言語バインディング）
    |
    +-- TF_NewGraph()
    |       +-- new tensorflow::Graph()  [c_api.cc]
    |
    +-- TF_NewOperation()
    |       +-- new tensorflow::NodeBuilder()  [c_api.cc]
    |
    +-- TF_AddInput() / TF_SetAttr*()
    |       +-- NodeBuilder::Input() / NodeBuilder::Attr()
    |
    +-- TF_FinishOperation()
    |       +-- NodeBuilder::Finalize()  [c_api.cc]
    |       +-- ShapeRefiner::AddNode()
    |
    +-- TF_NewSession()
    |       +-- tensorflow::NewSession()  [c_api.cc]
    |           +-- DirectSession / GrpcSession
    |
    +-- TF_SessionRun()
    |       +-- Session::Run()  [c_api.cc]
    |           +-- Executor::RunAsync()
    |               +-- OpKernel::Compute()
    |
    +-- TF_LoadSessionFromSavedModel()
    |       +-- tensorflow::LoadSavedModel()  [c_api.cc]
    |           +-- cc/saved_model/loader.h
    |
    +-- TF_DeleteSession() / TF_DeleteGraph()
            +-- delete Session / delete Graph
```

### データフロー図

```
[入力]                          [処理]                              [出力]

テンソルデータ(メモリ) -------> TF_NewTensor() -------> TF_Tensor*
                                    |
GraphDef(Proto) -----------> TF_GraphImportGraphDef() -> TF_Graph*
                                    |
SavedModel(ディスク) -------> TF_LoadSessionFromSavedModel() -> TF_Session*
                                    |
TF_Tensor*(入力テンソル) ---> TF_SessionRun() -------> TF_Tensor**(出力テンソル)
                                    |
                                    +-------> TF_Status*(実行結果ステータス)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| c_api.h | `tensorflow/c/c_api.h` | ソース(ヘッダ) | メインC APIヘッダ。グラフ構築・セッション実行・関数定義等の公開API |
| c_api.cc | `tensorflow/c/c_api.cc` | ソース(実装) | C APIの主要実装。C++コアランタイムへの委譲 |
| c_api_experimental.h | `tensorflow/c/c_api_experimental.h` | ソース(ヘッダ) | 実験的C API。XLA設定、チェックポイント読み取り等 |
| c_api_experimental.cc | `tensorflow/c/c_api_experimental.cc` | ソース(実装) | 実験的APIの実装 |
| c_api_internal.h | `tensorflow/c/c_api_internal.h` | ソース(ヘッダ) | オペーク構造体の内部定義 |
| ops.h | `tensorflow/c/ops.h` | ソース(ヘッダ) | カスタムOp登録・形状推論API |
| ops.cc | `tensorflow/c/ops.cc` | ソース(実装) | Op登録APIの実装 |
| kernels.h | `tensorflow/c/kernels.h` | ソース(ヘッダ) | カスタムカーネル登録API |
| kernels.cc | `tensorflow/c/kernels.cc` | ソース(実装) | カーネル登録APIの実装 |
| kernels_experimental.h | `tensorflow/c/kernels_experimental.h` | ソース(ヘッダ) | 実験的カーネルAPI |
| tf_tensor.h | `tensorflow/c/tf_tensor.h` | ソース(ヘッダ) | TF_Tensor型API |
| tf_status.h | `tensorflow/c/tf_status.h` | ソース(ヘッダ) | TF_Status型API |
| tf_datatype.h | `tensorflow/c/tf_datatype.h` | ソース(ヘッダ) | データ型定義 |
| tf_buffer.h | `tensorflow/c/tf_buffer.h` | ソース(ヘッダ) | バッファ型API |
| c_api_macros.h | `tensorflow/c/c_api_macros.h` | ソース(ヘッダ) | エクスポートマクロ定義 |
| env.h | `tensorflow/c/env.h` | ソース(ヘッダ) | 環境API（ファイル操作等） |
| safe_ptr.h | `tensorflow/c/safe_ptr.h` | ソース(ヘッダ) | CオブジェクトのRAIIラッパー |
| checkpoint_reader.h | `tensorflow/c/checkpoint_reader.h` | ソース(ヘッダ) | チェックポイント読み取り |
| python_api.h | `tensorflow/c/python_api.h` | ソース(ヘッダ) | Python C API対応関数 |
| eager/ | `tensorflow/c/eager/` | ディレクトリ | Eager Execution用C API |
| experimental/ | `tensorflow/c/experimental/` | ディレクトリ | 実験的機能（ストリームエグゼキュータ等） |
| BUILD | `tensorflow/c/BUILD` | ビルド設定 | Bazelビルドルール |
| README.md | `tensorflow/c/README.md` | ドキュメント | C APIの概要・ダウンロードリンク |
