# 機能設計書 35-学習率スケジューリング

## 概要

本ドキュメントは、TensorFlow/Kerasにおける学習率スケジューリング機能の設計を記載する。学習率スケジューリングは `tf.keras.optimizers.schedules` モジュールで提供され、訓練ステップに応じて学習率を動的に変化させる機能群である。

### 本機能の処理概要

**業務上の目的・背景**：固定の学習率では、訓練初期は大きなステップで損失を下げたい一方、後半は小さなステップで精密な最適化を行いたいという要求に対応できない。学習率スケジューリングにより、訓練中に学習率を動的に調整し、収束速度と最終的な精度の両方を改善する。

**機能の利用シーン**：オプティマイザの `learning_rate` パラメータに `LearningRateSchedule` インスタンスを渡すことで利用される。訓練ステップに応じて学習率が自動的に変化する。

**主要な処理内容**：
1. `LearningRateSchedule` 抽象基底クラスで共通インターフェースを定義
2. `ExponentialDecay`: 指数的に学習率を減衰
3. `PiecewiseConstantDecay`: 区間ごとに異なる定数学習率
4. `PolynomialDecay`: 多項式に基づく減衰
5. `InverseTimeDecay`: 逆時間減衰
6. `CosineDecay`: コサインアニーリング
7. `CosineDecayRestarts`: ウォームリスタート付きコサインアニーリング
8. 各スケジュールは `__call__(step)` で現在のステップに応じた学習率を返すCallable

**関連システム・外部連携**：OptimizerV2の `_decayed_lr` メソッドがスケジュールオブジェクトを呼び出し、現在のステップに対する学習率を取得する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 該当なし | - | 画面機能マッピングに関連画面なし |

## 機能種別

計算処理（学習率制御）

## 入力仕様

### 入力パラメータ（LearningRateSchedule共通）

全てのスケジュールは `__call__(step)` メソッドで `step`（整数またはTensor）を受け取る。

#### ExponentialDecay

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| initial_learning_rate | float | Yes | 初期学習率 | - |
| decay_steps | int | Yes | 減衰が適用されるステップ数 | 正の整数 |
| decay_rate | float | Yes | 減衰率 | - |
| staircase | bool | No | Trueの場合、階段状に減衰。デフォルトFalse | - |

#### PiecewiseConstantDecay

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| boundaries | list[int] | Yes | 境界ステップのリスト | 昇順 |
| values | list[float] | Yes | 各区間の学習率値 | len(values) == len(boundaries) + 1 |

#### PolynomialDecay

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| initial_learning_rate | float | Yes | 初期学習率 | - |
| decay_steps | int | Yes | 減衰ステップ数 | 正の整数 |
| end_learning_rate | float | No | 最終学習率。デフォルト0.0001 | - |
| power | float | No | 多項式のべき乗。デフォルト1.0 | - |
| cycle | bool | No | 減衰後にサイクルするか。デフォルトFalse | - |

#### InverseTimeDecay

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| initial_learning_rate | float | Yes | 初期学習率 | - |
| decay_steps | int | Yes | 減衰ステップ数 | 正の整数 |
| decay_rate | float | Yes | 減衰率 | - |
| staircase | bool | No | 階段状減衰。デフォルトFalse | - |

#### CosineDecay

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| initial_learning_rate | float | Yes | 初期学習率 | - |
| decay_steps | int | Yes | 減衰ステップ数 | 正の整数 |
| alpha | float | No | 最小学習率の比率。デフォルト0.0 | - |

#### CosineDecayRestarts

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| initial_learning_rate | float | Yes | 初期学習率 | - |
| first_decay_steps | int | Yes | 最初のサイクルのステップ数 | 正の整数 |
| t_mul | float | No | サイクル期間の乗数。デフォルト2.0 | - |
| m_mul | float | No | 初期学習率の乗数。デフォルト1.0 | - |
| alpha | float | No | 最小学習率の比率。デフォルト0.0 | - |

### 入力データソース

- 訓練ステップ数（オプティマイザのiterationsカウンタ）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 学習率 | Tensor(float) | 現在のステップに対応する学習率 |

### 出力先

オプティマイザの変数更新時の学習率として使用される。

## 処理フロー

### 処理シーケンス

```
1. スケジュールインスタンスの生成（パラメータ設定）
2. オプティマイザの_decayed_lrメソッドが呼ばれる
3. スケジュールの__call__(step)が呼ばれる
4. stepに応じた学習率がTensorとして返される
5. オプティマイザが返された学習率を使用して変数を更新
```

### フローチャート

```mermaid
flowchart TD
    A[OptimizerV2._decayed_lr呼び出し] --> B{learning_rateの型}
    B -->|LearningRateSchedule| C[schedule.__call__(step)]
    B -->|float| D[固定値を返す]
    C --> E{スケジュール種別}
    E -->|ExponentialDecay| F[initial_lr * decay_rate ^ (step/decay_steps)]
    E -->|PiecewiseConstant| G[boundariesに基づく定数値]
    E -->|PolynomialDecay| H[多項式減衰計算]
    E -->|InverseTimeDecay| I[initial_lr / (1 + decay_rate * step/decay_steps)]
    E -->|CosineDecay| J[cosineアニーリング計算]
    E -->|CosineDecayRestarts| K[ウォームリスタート計算]
    F --> L[学習率Tensorを返す]
    G --> L
    H --> L
    I --> L
    J --> L
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-35-1 | ExponentialDecay | lr = initial_lr * decay_rate ^ (global_step / decay_steps)。staircaseの場合、floorを適用 | ExponentialDecay使用時 |
| BR-35-2 | PiecewiseConstantDecay | boundaries[i-1] <= step < boundaries[i] の場合、values[i]を返す | PiecewiseConstantDecay使用時 |
| BR-35-3 | PolynomialDecay | lr = (initial_lr - end_lr) * (1 - step/decay_steps)^power + end_lr | PolynomialDecay使用時 |
| BR-35-4 | InverseTimeDecay | lr = initial_lr / (1 + decay_rate * step / decay_steps) | InverseTimeDecay使用時 |
| BR-35-5 | CosineDecay | lr = initial_lr * (alpha + (1-alpha) * 0.5 * (1 + cos(pi * step / decay_steps))) | CosineDecay使用時 |
| BR-35-6 | CosineDecayRestarts | ウォームリスタートのコサインアニーリング。t_mulで期間を増加、m_mulで振幅を減少 | CosineDecayRestarts使用時 |
| BR-35-7 | シリアライゼーション | get_config()で全パラメータを辞書として返し、from_config()で復元可能 | 全スケジュール |

### 計算ロジック

各スケジュールの計算式は上記ビジネスルールに記載の通り。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

データベース操作は発生しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | パラメータエラー | PiecewiseConstantDecayでboundariesとvaluesの長さ不整合 | len(values) == len(boundaries) + 1 にする |
| TypeError | 型エラー | stepに不正な型が渡された | 整数値またはTensorを渡す |

### リトライ仕様

リトライは不要。

## トランザクション仕様

特になし。

## パフォーマンス要件

- 各スケジュールの__call__は軽量な数学演算のみで構成
- TensorFlow Opsとして計算されるため、グラフモードでも効率的

## セキュリティ考慮事項

特になし。

## 備考

- LearningRateScheduleはオプティマイザのSerializable互換

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | learning_rate_schedule.py | `tensorflow/python/keras/optimizer_v2/learning_rate_schedule.py` | LearningRateSchedule基底クラスの構造 |

**読解のコツ**: `LearningRateSchedule` は `__call__` と `get_config` を定義する抽象クラス。TensorFlow 2.xでは `tf.keras.optimizers.schedules` 名前空間で公開される。

#### Step 2: 各スケジュールの実装

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | learning_rate_schedule.py | `tensorflow/python/keras/optimizer_v2/learning_rate_schedule.py` | ExponentialDecay.__call__ |
| 2-2 | learning_rate_schedule.py | `tensorflow/python/keras/optimizer_v2/learning_rate_schedule.py` | PiecewiseConstantDecay.__call__ |
| 2-3 | learning_rate_schedule.py | `tensorflow/python/keras/optimizer_v2/learning_rate_schedule.py` | PolynomialDecay.__call__ |
| 2-4 | learning_rate_schedule.py | `tensorflow/python/keras/optimizer_v2/learning_rate_schedule.py` | CosineDecay.__call__ |

**主要処理フロー**:
1. **44-60行目**: `LearningRateSchedule` 抽象基底クラス（`__call__`, `get_config`）
2. **63-132行目**: `ExponentialDecay` - decay_rate^(step/decay_steps)に基づく減衰
3. **135-226行目**: `PiecewiseConstantDecay` - boundariesに基づく区間定数
4. **229-330行目**: `PolynomialDecay` - (1-step/decay_steps)^powerに基づく減衰
5. **333-398行目**: `InverseTimeDecay` - 1/(1+decay_rate*step)に基づく減衰
6. **401-470行目**: `CosineDecay` - cos(pi*step/decay_steps)に基づくアニーリング
7. **473-580行目**: `CosineDecayRestarts` - ウォームリスタート付きコサインアニーリング

#### Step 3: オプティマイザとの連携

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | optimizer_v2.py | `tensorflow/python/keras/optimizer_v2/optimizer_v2.py` | _decayed_lrメソッド |

**読解のコツ**: `_decayed_lr` メソッドは `learning_rate` がスケジュールオブジェクトの場合に `__call__(self.iterations)` を呼び出して現在の学習率を取得する。

### プログラム呼び出し階層図

```
OptimizerV2._decayed_lr()
    |
    +-- isinstance(lr, LearningRateSchedule)
    |       |
    |       +-- ExponentialDecay.__call__(step)
    |       +-- PiecewiseConstantDecay.__call__(step)
    |       +-- PolynomialDecay.__call__(step)
    |       +-- InverseTimeDecay.__call__(step)
    |       +-- CosineDecay.__call__(step)
    |       +-- CosineDecayRestarts.__call__(step)
    |
    +-- math_ops.cast(lr, dtype)  [float値の場合]
```

### データフロー図

```
[入力]               [処理]                    [出力]

ステップ数 -------> LearningRateSchedule.__call__(step) -------> 学習率Tensor
                         |
初期学習率 -------->     |
decay_steps ------->     |
decay_rate -------->     |  (ExponentialDecay)
                         |
boundaries -------->     |
values ------------>     |  (PiecewiseConstantDecay)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| learning_rate_schedule.py | `tensorflow/python/keras/optimizer_v2/learning_rate_schedule.py` | ソース | 全学習率スケジュールの実装（約580行） |
| optimizer_v2.py | `tensorflow/python/keras/optimizer_v2/optimizer_v2.py` | ソース | _decayed_lrメソッドからの呼び出し元 |
| generic_utils.py | `tensorflow/python/keras/utils/generic_utils.py` | ソース | シリアライゼーションユーティリティ |
