# 機能設計書 40-損失関数（Loss Functions）

## 概要

本ドキュメントは、TensorFlow/Kerasにおける損失関数（Loss Functions）の機能設計を記載する。損失関数はモデルの予測値と正解値の差異を定量化する関数群であり、`tensorflow/python/keras/losses.py` に実装される。

### 本機能の処理概要

**業務上の目的・背景**：機械学習モデルの訓練において、モデルの予測精度を評価し最適化の方向を定めるために損失関数が必要となる。Kerasは回帰用、分類用、ランキング用など多様な損失関数を提供し、`model.compile(loss=...)` や直接呼び出しで利用される。

**機能の利用シーン**：
- `model.compile(loss='categorical_crossentropy')` による文字列指定
- `model.compile(loss=tf.keras.losses.CategoricalCrossentropy())` によるインスタンス指定
- カスタム訓練ループ内での直接呼び出し

**主要な処理内容**：
1. `Loss` 抽象基底クラスで共通インターフェース（`__call__`, `call`, `get_config`）を定義
2. `LossFunctionWrapper` で関数ベースの損失を `Loss` クラスにラップ
3. 各種具象クラス（MeanSquaredError, CategoricalCrossentropy等）を提供
4. reduction パラメータによる集約方法の制御（AUTO, SUM, NONE, SUM_OVER_BATCH_SIZE）
5. sample_weight による重み付き損失計算のサポート

**関連システム・外部連携**：Keras Model.compile/fit、losses_utils（重み付き損失計算）、分散学習（tf.distribute.Strategy）。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | 該当なし | - | 画面機能マッピングに関連画面なし |

## 機能種別

計算処理（損失計算）

## 入力仕様

### Loss.__call__() メソッド

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| y_true | Tensor | Yes | 正解ラベル。shape=[batch_size, d0, ..., dN] | - |
| y_pred | Tensor | Yes | 予測値。shape=[batch_size, d0, ..., dN] | - |
| sample_weight | Tensor | No | サンプル重み。scalar, [batch_size], [batch_size, d0, ..., dN-1] | broadcastable |

### Lossコンストラクタ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| reduction | ReductionV2 | No | 集約方法。デフォルトAUTO | Reduction列挙値 |
| name | str | No | 損失関数名 | - |

### 追加パラメータ（クラス固有）

| クラス名 | 追加パラメータ | 説明 |
|---------|-------------|------|
| BinaryCrossentropy | from_logits(bool), label_smoothing(float), axis(int) | ロジット/確率、ラベルスムージング |
| CategoricalCrossentropy | from_logits(bool), label_smoothing(float), axis(int) | ロジット/確率、ラベルスムージング |
| SparseCategoricalCrossentropy | from_logits(bool) | ロジット/確率 |

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| loss | Tensor(float) | 計算された損失値。reductionがNONEの場合はバッチ次元を持つ |

### 出力先

呼び出し元に損失テンソルとして返却。

## 処理フロー

### 処理シーケンス

```
1. Loss.__call__(y_true, y_pred, sample_weight):
   └─ name_scopeの設定
   └─ Eagerモード: self.call() を直接呼び出し
   └─ グラフモード: autograph.tf_convert(self.call) で変換後呼び出し
   └─ losses_utils.compute_weighted_loss() で重み付き損失と集約を計算
2. LossFunctionWrapper.call(y_true, y_pred):
   └─ squeeze_or_expand_dimensions で次元調整
   └─ ag_fn(y_true, y_pred, **kwargs) でラップされた関数呼び出し
3. Reduction処理:
   └─ AUTO: SUM_OVER_BATCH_SIZEに変換（分散学習時はエラー）
   └─ SUM: tf.reduce_sum
   └─ NONE: そのまま返却
   └─ SUM_OVER_BATCH_SIZE: tf.reduce_sum / batch_size
```

### フローチャート

```mermaid
flowchart TD
    A[Loss.__call__ 呼び出し] --> B[name_scope設定]
    B --> C{Eagerモード?}
    C -->|Yes| D[self.call 直接呼び出し]
    C -->|No| E[autograph.tf_convert 後に呼び出し]
    D --> F[losses_utils.compute_weighted_loss]
    E --> F
    F --> G{sample_weight?}
    G -->|あり| H[重み付き損失計算]
    G -->|なし| I[損失のみ]
    H --> J{reduction}
    I --> J
    J -->|AUTO/SUM_OVER_BATCH_SIZE| K[平均化]
    J -->|SUM| L[合計]
    J -->|NONE| M[そのまま]
    K --> N[損失テンソル返却]
    L --> N
    M --> N
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-40-1 | 分散学習制約 | tf.distribute.Strategy使用時、AUTOとSUM_OVER_BATCH_SIZEはビルトイン訓練ループ外でエラー | _get_reduction()呼び出し時 |
| BR-40-2 | AUTO解決 | reductionがAUTOの場合、SUM_OVER_BATCH_SIZEに変換 | _get_reduction()呼び出し時 |
| BR-40-3 | 次元調整 | y_trueとy_predの次元がずれている場合、squeeze_or_expand_dimensionsで調整 | LossFunctionWrapper.call()時 |
| BR-40-4 | Autograph変換 | グラフモードではself.callをAutographで変換 | グラフモード実行時 |
| BR-40-5 | name_scope | selfの_name_scopeでOps名前空間を設定。lambda名の場合は'lambda'に変換 | __call__()時 |

### 提供される損失クラス一覧

| クラス名 | 計算式 | 用途 |
|---------|--------|------|
| MeanSquaredError | mean(square(y_true - y_pred)) | 回帰 |
| MeanAbsoluteError | mean(abs(y_true - y_pred)) | 回帰 |
| MeanAbsolutePercentageError | 100 * mean(abs((y_true - y_pred) / y_true)) | 回帰 |
| MeanSquaredLogarithmicError | mean(square(log(y_true+1) - log(y_pred+1))) | 回帰 |
| BinaryCrossentropy | binary_crossentropy(y_true, y_pred) | 2クラス分類 |
| CategoricalCrossentropy | categorical_crossentropy(y_true, y_pred) | 多クラス分類（one-hot） |
| SparseCategoricalCrossentropy | sparse_categorical_crossentropy(y_true, y_pred) | 多クラス分類（整数ラベル） |
| Hinge | mean(max(1 - y_true * y_pred, 0)) | SVM系分類 |
| SquaredHinge | mean(square(max(1 - y_true * y_pred, 0))) | SVM系分類 |
| CategoricalHinge | max(neg - pos + 1, 0) | 多クラスSVM系 |
| Poisson | y_pred - y_true * log(y_pred) | カウントデータ |
| LogCosh | log(cosh(y_pred - y_true)) | 回帰 |
| Huber | 条件付きで二乗/線形 | ロバスト回帰 |
| KLDivergence | y_true * log(y_true / y_pred) | 分布間距離 |
| CosineSimilarity | -sum(l2_normalize(y_true) * l2_normalize(y_pred)) | 類似度 |

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

データベース操作は発生しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | 分散学習制約 | Strategy外でAUTO/SUM_OVER_BATCH_SIZE使用 | SUM/NONEを使用 |
| ValueError | sample_weight形状不正 | sample_weightがbroadcastableでない | 正しい形状を指定 |
| ValueError | reduction不正 | 未知のreduction値 | Reduction列挙値を使用 |

### リトライ仕様

リトライは不要。

## トランザクション仕様

特になし。

## パフォーマンス要件

- 全損失関数はTensorFlow Opsで実装され、GPU上で効率的に実行
- Autograph変換によりグラフモードでも最適化される
- sample_weightの適用はelement-wise乗算で効率的

## セキュリティ考慮事項

特になし。

## 備考

- 関数ベースの損失（`tf.keras.losses.mean_squared_error` 等）も別途提供される
- シリアライゼーション: `serialize`/`deserialize`/`get` 関数で文字列名での解決が可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 基底クラスの構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | losses.py | `tensorflow/python/keras/losses.py` | Loss基底クラス |

**主要処理フロー**:
1. **48-84行目**: `Loss` クラスのdocstring - 基本概念と使用方法
2. **86-103行目**: `__init__` - reduction, name, _allow_sum_over_batch_sizeの設定
3. **109-117行目**: `_set_name_scope` - name_scopeの正規化（lambda対応）
4. **119-156行目**: `__call__` - メインの呼び出しロジック
5. **147-150行目**: Eagerモード判定とcall呼び出し
6. **154-156行目**: `compute_weighted_loss` で重み付き損失計算
7. **174-188行目**: `call` 抽象メソッド（サブクラスで実装必須）
8. **190-212行目**: `_get_reduction` - 分散学習制約チェックとAUTO解決

#### Step 2: LossFunctionWrapperを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | losses.py | `tensorflow/python/keras/losses.py` | LossFunctionWrapperクラス |

**主要処理フロー**:
- **215-265行目**: `LossFunctionWrapper` - 関数ベース損失のラッパー
- **241-242行目**: `__init__` でfnとkwargsを保存
- **244-258行目**: `call` でsqueeze_or_expand_dimensionsとag_fn呼び出し
- **254-255行目**: `y_pred, y_true = losses_utils.squeeze_or_expand_dimensions(y_pred, y_true)`

#### Step 3: 具象損失クラスの実装パターン

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | losses.py | `tensorflow/python/keras/losses.py` | MeanSquaredError, BinaryCrossentropy等 |

**主要処理フロー**:
- **268-323行目**: `MeanSquaredError` - 最もシンプルな例。`super().__init__(mean_squared_error, name=name, reduction=reduction)`
- **502-601行目**: `BinaryCrossentropy` - from_logits, label_smoothing, axisパラメータ付き
- **604-684行目**: `CategoricalCrossentropy` - one-hotラベル用
- **686-758行目**: `SparseCategoricalCrossentropy` - 整数ラベル用
- **761-816行目**: `Hinge` - SVM系損失
- **937-989行目**: `Poisson` - カウントデータ用

**読解のコツ**: 各損失クラスは `LossFunctionWrapper` を継承し、`super().__init__` で対応する関数（例: `mean_squared_error`）を渡す。関数自体はファイル末尾に定義されている。

### プログラム呼び出し階層図

```
tf.keras.losses.CategoricalCrossentropy (losses.py)
    |
    +-- LossFunctionWrapper.__init__(categorical_crossentropy, ...)
    |
    +-- Loss.__call__(y_true, y_pred, sample_weight)
    |       |
    |       +-- backend.name_scope(self._name_scope)
    |       +-- [Eager] self.call(y_true, y_pred)
    |       |   [Graph] autograph.tf_convert(self.call)(y_true, y_pred)
    |       |       |
    |       |       +-- LossFunctionWrapper.call()
    |       |           +-- losses_utils.squeeze_or_expand_dimensions()
    |       |           +-- categorical_crossentropy(y_true, y_pred, **kwargs)
    |       |               +-- nn.softmax_cross_entropy_with_logits [from_logits]
    |       |
    |       +-- losses_utils.compute_weighted_loss(losses, sample_weight, reduction)
    |               +-- [Reduction logic]
    |
    +-- Loss.get_config() -> dict
    +-- Loss.from_config(config) -> instance
```

### データフロー図

```
[入力]               [処理]                    [出力]

y_true ------------> Loss.__call__
y_pred ------------> |
sample_weight -----> |
                     |
                     +-> self.call(y_true, y_pred)
                     |       |
                     |       v
                     |   element-wise損失値
                     |       |
                     +-> compute_weighted_loss
                     |       |
                     |   sample_weight適用
                     |       |
                     +-> reduction適用
                             |
                             v
                        損失スカラー or バッチ損失
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| losses.py | `tensorflow/python/keras/losses.py` | ソース | 損失クラスの主実装（約1500行） |
| losses_utils.py | `tensorflow/python/keras/utils/losses_utils.py` | ソース | compute_weighted_loss, squeeze_or_expand_dimensions |
| losses_impl.py | `tensorflow/python/ops/losses/losses_impl.py` | ソース | 低レベル損失関数実装 |
| nn.py | `tensorflow/python/ops/nn.py` | ソース | softmax_cross_entropy_with_logits等 |
| backend.py | `tensorflow/python/keras/backend.py` | ソース | name_scope, binary_crossentropy等 |
| generic_utils.py | `tensorflow/python/keras/utils/generic_utils.py` | ソース | serialize/deserialize_keras_object |
