# 機能設計書 42-正則化（Regularizers）

## 概要

本ドキュメントは、TensorFlow/Kerasにおける正則化（Regularizers）機能の設計を記述する。L1・L2・L1L2正則化によりモデルの重みにペナルティを課し、過学習を防止する機能を提供する。

### 本機能の処理概要

本機能は、ニューラルネットワークの学習時に、重みパラメータに対してL1（絶対値）やL2（二乗）のペナルティ項を損失関数に加算することで、モデルの汎化性能を向上させる正則化手法を提供する。

**業務上の目的・背景**：機械学習モデルは訓練データに過剰に適合（過学習）するリスクがある。正則化は、重みの大きさを抑制することで、モデルの複雑さを制限し、未知のデータに対する汎化性能を向上させる。特に限られた訓練データでの学習や、大規模モデルの過学習防止に不可欠な技術である。

**機能の利用シーン**：（1）Dense層やConv層の`kernel_regularizer`引数で重みにL1/L2ペナルティを適用する場面、（2）`bias_regularizer`でバイアス項に正則化を適用する場面、（3）`activity_regularizer`で層の出力に正則化を適用する場面、（4）カスタム正則化関数を定義して特殊なペナルティを適用する場面。

**主要な処理内容**：
1. Regularizerベースクラスによる正則化インターフェース定義
2. L1正則化：`l1 * reduce_sum(abs(x))`のペナルティ計算
3. L2正則化：`l2 * reduce_sum(square(x))`のペナルティ計算
4. L1L2正則化：L1とL2の両方のペナルティを組み合わせた計算
5. シリアライズ/デシリアライズによる正則化設定の保存・復元
6. 文字列識別子（"l1"、"l2"、"l1_l2"）からの正則化インスタンス生成

**関連システム・外部連携**：Kerasの各層（Dense、Conv2D等）のkernel_regularizer/bias_regularizer/activity_regularizer引数、モデル保存時のHDF5/SavedModelフォーマット。

**権限による制御**：特になし。全てのユーザが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに本機能の関連画面なし |

## 機能種別

計算処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| l1 | float | No | L1正則化係数 | 有限の数値（NaN, Inf不可）。デフォルト0.01 |
| l2 | float | No | L2正則化係数 | 有限の数値（NaN, Inf不可）。デフォルト0.01 |
| x | Tensor | Yes | 正則化対象の重みテンソル | __call__時の入力 |

### 入力データソース

Kerasレイヤーの重みテンソル（kernel, bias等）。レイヤーのbuild時に正則化関数が登録され、フォワードパス時に自動呼び出しされる。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| regularization_loss | Tensor (scalar) | 正則化ペナルティ値 |

### 出力先

Kerasレイヤーのlossesプロパティに追加され、総損失関数に加算される。

## 処理フロー

### 処理シーケンス

```
1. 正則化インスタンス生成（__init__）
   └─ l1/l2係数の検証（_check_penalty_number）
   └─ 係数をfloatx()型にキャスト
2. レイヤー構築時に正則化関数を登録
   └─ kernel_regularizer / bias_regularizer / activity_regularizer
3. フォワードパス時に__call__が呼ばれる
   └─ L1: l1 * reduce_sum(abs(x))
   └─ L2: l2 * reduce_sum(square(x))
   └─ L1L2: L1ペナルティ + L2ペナルティ
4. 損失にペナルティ値が加算される
```

### フローチャート

```mermaid
flowchart TD
    A[Regularizer生成] --> B[係数検証 _check_penalty_number]
    B --> C[係数をfloatx型にキャスト]
    C --> D[レイヤーに登録]
    D --> E{フォワードパス}
    E --> F{正則化タイプ}
    F -->|L1| G[l1 * sum_abs_x]
    F -->|L2| H[l2 * sum_square_x]
    F -->|L1L2| I[L1ペナルティ + L2ペナルティ]
    G --> J[layer.lossesに追加]
    H --> J
    I --> J
    J --> K[総損失に加算]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-42-01 | 係数検証 | 正則化係数はfloatまたはintで、NaN/Infは不可 | __init__時 |
| BR-42-02 | デフォルト係数 | L1/L2の文字列指定時のデフォルト係数は0.01 | 文字列"l1"/"l2"指定時 |
| BR-42-03 | L1L2デフォルト | L1L2()のデフォルトはl1=0, l2=0。文字列"l1_l2"の場合はl1=0.01, l2=0.01 | コンストラクタ/文字列指定 |
| BR-42-04 | activity_regularizer | activity_regularizerの出力はバッチサイズで除算される | レイヤー適用時 |

### 計算ロジック

- **L1**: `loss = l1 * reduce_sum(abs(x))`
- **L2**: `loss = l2 * reduce_sum(square(x))`
- **L1L2**: `loss = l1 * reduce_sum(abs(x)) + l2 * reduce_sum(square(x))`

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | 入力エラー | 係数がfloat/intでない、またはNaN/Inf | 有限の数値を指定 |
| TypeError | 引数エラー | 認識されない引数が渡された | 正しい引数名を確認 |
| NotImplementedError | 未実装 | ベースクラスのget_configが呼ばれた | サブクラスでget_configを実装 |

### リトライ仕様

リトライ不要。

## トランザクション仕様

トランザクション管理なし。

## パフォーマンス要件

- 正則化の計算は重みテンソルのサイズに線形に比例する
- reduce_sumとabs/square操作のみであり、計算コストは極めて低い

## セキュリティ考慮事項

特になし。

## 備考

- カスタム正則化は関数ベース（Callableを返す関数）またはクラスベース（Regularizerを継承）で作成可能
- HDF5形式での保存・復元にはget_configの実装とregister_keras_serializableでの登録が必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | regularizers.py | `tensorflow/python/keras/regularizers.py` | Regularizerベースクラスの__call__インターフェース |

**読解のコツ**: Regularizerはステートレスな関数オブジェクトであり、Metricと異なり内部状態を持たない。__call__(x)で重みテンソルを受け取り、スカラーのペナルティ値を返す。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | regularizers.py | `tensorflow/python/keras/regularizers.py` | get()関数による正則化インスタンスの取得 |

**主要処理フロー**:
1. **358-370行目**: get()関数。識別子（dict/string/callable）から正則化インスタンスを解決
2. **346-355行目**: deserialize()関数。設定辞書からの復元。"l1_l2"文字列の特別処理
3. **342-343行目**: serialize()関数。正則化インスタンスの直列化

#### Step 3: 正則化クラスの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | regularizers.py | `tensorflow/python/keras/regularizers.py` | L1L2クラスの実装 |
| 3-2 | regularizers.py | `tensorflow/python/keras/regularizers.py` | L1/L2クラスの実装 |

**主要処理フロー**:
- **26-38行目**: _check_penalty_number - 係数の有効性検証（float/int、非NaN/Inf）
- **208-249行目**: L1L2クラス - l1とl2の両ペナルティ計算
- **240-246行目**: L1L2.__call__ - 0定数から開始し、l1/l2が非ゼロの場合にペナルティを加算
- **252-282行目**: L1クラス - l1 * reduce_sum(abs(x))
- **285-315行目**: L2クラス - l2 * reduce_sum(square(x))

### プログラム呼び出し階層図

```
tf.keras.layers.Dense(kernel_regularizer='l2')
    │
    ├─ regularizers.get('l2')
    │      └─ deserialize('l2')
    │             └─ L2(l2=0.01)
    │                    └─ _check_penalty_number(0.01)
    │
    └─ Layer.build() → 正則化関数を登録
           │
           └─ Layer.__call__() → フォワードパス
                  │
                  └─ L2.__call__(kernel_weight)
                         └─ l2 * reduce_sum(square(x))
                                └─ layer.losses に追加
```

### データフロー図

```
[入力]                [処理]                       [出力]

重みテンソル ────▶ Regularizer.__call__(x) ──▶ ペナルティ値 (scalar)
  (kernel)              │                            │
                        ├─ L1: abs → sum → * l1      ▼
                        ├─ L2: square → sum → * l2   layer.losses
                        └─ L1L2: L1 + L2                │
                                                        ▼
                                                    総損失に加算
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| regularizers.py | `tensorflow/python/keras/regularizers.py` | ソース | 正則化クラス群の全実装 |
| base_layer.py | `tensorflow/python/keras/engine/base_layer.py` | ソース | レイヤーへの正則化登録・適用処理 |
| generic_utils.py | `tensorflow/python/keras/utils/generic_utils.py` | ソース | serialize/deserializeユーティリティ |
| math_ops.py | `tensorflow/python/ops/math_ops.py` | ソース | reduce_sum, abs, square等の数学演算 |
| backend.py | `tensorflow/python/keras/backend.py` | ソース | cast_to_floatx, constant等のバックエンド関数 |
