# 機能設計書 46-データセットキャッシュ

## 概要

本ドキュメントは、TensorFlowにおけるデータセットキャッシュ機能の設計を記述する。データセットのメモリ/ディスクキャッシュ機能を提供し、データ前処理の繰返し実行を回避する。

### 本機能の処理概要

本機能は、tf.data.Datasetのcache()メソッドとして提供され、データセットの要素を最初のイテレーション時にメモリまたはディスクにキャッシュし、以降のイテレーションではキャッシュから高速にデータを供給する。

**業務上の目的・背景**：機械学習の訓練では、同じデータセットを複数エポックにわたって繰り返し処理する。データの読み込みや前処理（パース、デコード、リサイズ等）は計算コストが高い場合があり、キャッシュにより2エポック目以降の処理を大幅に高速化できる。特にファイルI/Oがボトルネックとなるパイプラインで効果が大きい。

**機能の利用シーン**：（1）小〜中規模データセットをメモリにキャッシュしてエポック間のI/Oを削減する場面、（2）大規模データセットをディスクにキャッシュしてパース/デコード処理を省略する場面、（3）map変換後にキャッシュして前処理の再実行を防ぐ場面。

**主要な処理内容**：
1. CacheDatasetクラスによるキャッシュデータセットの構築
2. filenameが空文字列の場合：メモリキャッシュ（DummyMemoryCache使用）
3. filenameが指定された場合：ディスクキャッシュ（指定パスにファイル保存）
4. TF2環境ではcache_dataset_v2 Opを使用、レガシー環境ではcache_dataset Opを使用

**関連システム・外部連携**：tf.data.Dataset API（No.44）の変換操作の一つ。ファイルシステム（ディスクキャッシュ時）。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに本機能の関連画面なし |

## 機能種別

計算処理 / データ連携

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input_dataset | Dataset | Yes | キャッシュ対象のデータセット | 有効なDatasetオブジェクト |
| filename | string | No | ディスクキャッシュのファイルパス | 空文字列でメモリキャッシュ |
| name | string | No | データセットの名前 | None可 |

### 入力データソース

上流のtf.data.Datasetパイプラインからの出力。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| cached_dataset | tf.data.Dataset | キャッシュ機能付きデータセット |

### 出力先

下流のtf.dataパイプラインに接続される。

## 処理フロー

### 処理シーケンス

```
1. CacheDataset.__init__()
   └─ filenameをstring型テンソルに変換
   └─ TF2環境かどうかを判定
2. TF2環境の場合:
   └─ gen_dataset_ops.dummy_memory_cache()でキャッシュリソース生成
   └─ gen_dataset_ops.cache_dataset_v2()でvariant_tensor生成
3. レガシー環境の場合:
   └─ gen_dataset_ops.cache_dataset()でvariant_tensor生成
4. 最初のイテレーション時:
   └─ 上流データセットから要素を取得し、キャッシュに格納
5. 2回目以降のイテレーション時:
   └─ キャッシュから直接要素を取得
```

### フローチャート

```mermaid
flowchart TD
    A[Dataset.cache filename] --> B{filename指定?}
    B -->|空文字列| C[メモリキャッシュモード]
    B -->|パス指定| D[ディスクキャッシュモード]
    C --> E{TF2環境?}
    D --> E
    E -->|Yes| F[cache_dataset_v2 + dummy_memory_cache]
    E -->|No| G[cache_dataset]
    F --> H{初回イテレーション?}
    G --> H
    H -->|Yes| I[上流から取得してキャッシュに格納]
    H -->|No| J[キャッシュから直接取得]
    I --> K[要素を出力]
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-46-01 | メモリキャッシュ | filenameが空文字列の場合、メモリにキャッシュする | filename="" |
| BR-46-02 | ディスクキャッシュ | filenameが指定された場合、指定パスにファイルとしてキャッシュ | filename指定時 |
| BR-46-03 | キャッシュ構造不変 | CacheDatasetはUnaryUnchangedStructureDatasetを継承し、入力と同じelement_specを持つ | 常時 |
| BR-46-04 | V2 Op使用条件 | TF2有効かつeager/tf.function内でcache_dataset_v2を使用 | TF2環境 |

### 計算ロジック

特に複雑な計算ロジックはなし。キャッシュの格納と取得はC++カーネルレベルで実行される。

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作なし（ファイルI/Oはファイルシステムレベル） |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| NotFoundError | I/Oエラー | ディスクキャッシュの書込みパスが存在しない | 有効なディレクトリパスを指定 |
| ResourceExhaustedError | リソース枯渇 | メモリキャッシュでメモリ不足 | ディスクキャッシュに変更 |

### リトライ仕様

リトライ不要。

## トランザクション仕様

トランザクション管理なし。

## パフォーマンス要件

- メモリキャッシュは最高速だが、データセットサイズがメモリに収まる必要がある
- ディスクキャッシュはI/O速度に依存するが、メモリ制約なし
- キャッシュはmap等の前処理後に配置するのが効果的

## セキュリティ考慮事項

- ディスクキャッシュファイルには訓練データが平文で保存される可能性がある

## 備考

- cache()はshuffle()の前に配置すると、シャッフルの再現性に影響する可能性がある
- データセットの要素数が変動する場合、キャッシュの整合性に注意が必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | cache_op.py | `tensorflow/python/data/ops/cache_op.py` | CacheDatasetクラスの構造 |

**読解のコツ**: CacheDatasetはUnaryUnchangedStructureDatasetを継承しており、入力データセットの型構造をそのまま保持する。filenameの値によってメモリ/ディスクキャッシュが切り替わる。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | cache_op.py | `tensorflow/python/data/ops/cache_op.py` | _cache関数とCacheDataset.__init__ |

**主要処理フロー**:
1. **25-26行目**: _cache関数 - CacheDatasetのファクトリ
2. **32-33行目**: CacheDataset.__init__ - 入力データセットとfilenameの保持
3. **35-36行目**: filenameをstring型テンソルに変換
4. **38-43行目**: TF2環境判定 → cache_dataset_v2 + dummy_memory_cache
5. **45-48行目**: レガシー環境 → cache_dataset

### プログラム呼び出し階層図

```
Dataset.cache(filename="")
    │
    └─ cache_op._cache(input_dataset, filename, name)
           │
           └─ CacheDataset.__init__()
                  ├─ ops.convert_to_tensor(filename, dtype=string)
                  │
                  ├─ [TF2] gen_dataset_ops.cache_dataset_v2(
                  │         input_variant, filename,
                  │         cache=gen_dataset_ops.dummy_memory_cache())
                  │
                  └─ [Legacy] gen_dataset_ops.cache_dataset(
                              input_variant, filename)
```

### データフロー図

```
[入力]                    [処理]                      [出力]

上流Dataset ──────▶ CacheDataset ──────▶ キャッシュ済みDataset
                        │                        │
              初回: 上流→キャッシュ格納     2回目〜: キャッシュ→出力
                        │
              ┌─────────┴──────────┐
              │ メモリキャッシュ   │ ディスクキャッシュ
              │ (filename="")      │ (filename指定)
              └────────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| cache_op.py | `tensorflow/python/data/ops/cache_op.py` | ソース | CacheDatasetの実装 |
| dataset_ops.py | `tensorflow/python/data/ops/dataset_ops.py` | ソース | UnaryUnchangedStructureDataset基底クラス |
| gen_dataset_ops.py | `tensorflow/python/ops/gen_dataset_ops.py` | 自動生成 | cache_dataset/cache_dataset_v2 Opバインディング |
