# 機能設計書 55-ステートフル乱数生成

## 概要

本ドキュメントは、TensorFlowにおけるステートフル乱数生成機能の設計を記述する。内部状態（シード）を保持し、呼び出しごとに異なる乱数列を生成するオペレーション群を提供する。

### 本機能の処理概要

**業務上の目的・背景**：ニューラルネットワークの学習においては、重み初期化（一様分布・正規分布）、ドロップアウト（ベルヌーイ分布）、データシャッフル、確率的サンプリング（多項分布・ポアソン分布・ガンマ分布）など、多様な確率分布からの乱数生成が不可欠である。ステートフル乱数生成は、グローバルシード+オペレーションシードによる再現性と、呼び出しごとに異なる結果を返す性質を両立する。

**機能の利用シーン**：重み初期化（RandomUniform、RandomStandardNormal、TruncatedNormal）、ドロップアウト層での乱数マスク生成、データパイプラインでのシャッフル（RandomShuffle）、強化学習でのアクション選択（Multinomial）で利用される。

**主要な処理内容**：
1. random_ops.cc定義のオペレーション：
   - RandomUniform: 一様分布 U[0,1) の乱数生成
   - RandomUniformInt: 整数一様分布 U[minval, maxval) の乱数生成
   - RandomStandardNormal: 標準正規分布 N(0,1) の乱数生成
   - TruncatedNormal: 切断正規分布の乱数生成
   - ParameterizedTruncatedNormal: パラメータ付き切断正規分布
   - RandomShuffle: テンソルの第1軸に沿ったシャッフル
   - Multinomial: 多項分布からのサンプリング
   - RandomGamma/RandomGammaGrad: ガンマ分布からのサンプリングとその勾配
   - RandomPoisson/RandomPoissonV2: ポアソン分布からのサンプリング
2. stateful_random_ops.cc定義のオペレーション：
   - StatefulUniform/StatefulUniformFullInt: リソース管理型一様分布乱数
   - StatefulStandardNormalV2: リソース管理型標準正規分布乱数
   - StatefulTruncatedNormal: リソース管理型切断正規分布乱数
   - StatefulUniformInt: リソース管理型整数一様分布乱数
   - StatefulRandomBinomial: リソース管理型二項分布乱数
   - RngSkip/RngReadAndSkip: 乱数状態のスキップ操作
   - NonDeterministicInts: 非決定論的整数生成

**関連システム・外部連携**：特になし。内部計算パイプラインの基盤。

**権限による制御**：特段の権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面と直接関連しない基盤機能である |

## 機能種別

計算処理 / 乱数生成

## 入力仕様

### 入力パラメータ（random_ops.cc）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| shape | T (int32/int64) | Yes | 出力テンソルの形状 | ShapeTensor |
| seed | int（属性） | No | グローバルシード（デフォルト0=ランダム） | 整数 |
| seed2 | int（属性） | No | オペレーションシード（デフォルト0=ランダム） | 整数 |
| dtype | 型属性 | No | 出力データ型 | half/bfloat16/float/double |
| minval | Tout | Yes（RandomUniformInt） | 最小値（スカラー） | rank=0 |
| maxval | Tout | Yes（RandomUniformInt） | 最大値（スカラー） | rank=0 |
| logits | T | Yes（Multinomial） | ログ確率テンソル [batch, classes] | rank=2 |
| num_samples | int32 | Yes（Multinomial） | サンプル数 | rank=0 |

### 入力パラメータ（stateful_random_ops.cc）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| resource | resource | Yes | 乱数状態を保持するリソースハンドル | - |
| algorithm | int64 | Yes | 乱数アルゴリズム識別子 | rank=0 |
| shape | shape_dtype | Yes | 出力テンソルの形状 | ShapeTensor |

### 入力データソース

計算グラフ内で直接パラメータ指定。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output | dtype | 指定形状の乱数テンソル（RandomUniform等） |
| output | Tout | 整数乱数テンソル（RandomUniformInt） |
| output | T | シャッフル済みテンソル（RandomShuffle） |
| output | output_dtype | 多項分布サンプル [batch, num_samples]（Multinomial） |

### 出力先

後続の計算グラフ（モデルの初期化、ドロップアウトマスク、シャッフル済みデータ）。

## 処理フロー

### 処理シーケンス

```
1. 乱数生成リクエスト
   ├─ random_ops: shape+seed属性から乱数テンソル生成
   └─ stateful_random_ops: resource+algorithm+shapeから乱数テンソル生成
2. 内部状態の更新
   ├─ random_ops: シードカウンタの暗黙的更新（SetIsStateful）
   └─ stateful_random_ops: リソース内カウンタの明示的更新
3. 出力テンソルの返却
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-55-01 | ステートフル | 全てSetIsStateful()が指定され、呼び出しごとに異なる結果を返す | 全Op |
| BR-55-02 | シード制御 | seed=0かつseed2=0の場合は毎回異なるシーケンス、それ以外は再現可能 | random_ops |
| BR-55-03 | リソース管理 | stateful_random_ops系はresourceハンドルで状態を管理 | stateful_random_ops |
| BR-55-04 | minval/maxval検証 | RandomUniformIntのminval/maxvalはスカラーでなければならない | RandomUniformInt |
| BR-55-05 | 非推奨 | RandomPoissonはバージョン25でRandomPoissonV2に置換 | RandomPoisson |
| BR-55-06 | 非推奨 | StatefulStandardNormalはバージョン29でV2に置換 | StatefulStandardNormal |

### 計算ロジック

- **RandomUniform**: PhiloxまたはThreeFryアルゴリズムで一様分布 [0,1) を生成
- **TruncatedNormal**: 標準正規分布から-2σ〜+2σの範囲に切断
- **Multinomial**: ログ確率にGumbel分布ノイズを加えてargmaxでサンプリング

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgument | スカラーエラー | minval/maxvalがスカラーでない | スカラー値を入力 |
| InvalidArgument | ランクエラー | Multinomialのlogitsがrank=2でない | [batch, classes]形状のテンソルを入力 |

### リトライ仕様

ステートフルであるため、リトライ時は異なる結果が返る。再現性が必要な場合はシードを明示的に設定する。

## トランザクション仕様

トランザクション制御は行わない。

## パフォーマンス要件

学習の各ステップで呼び出されるため、高速な乱数生成が求められる。GPU上でのPhiloxカウンタベース生成で高速化が可能。

## セキュリティ考慮事項

暗号用途には使用しないこと。NonDeterministicIntsは暗号的に安全な乱数ではない。

## 備考

random_ops.ccは約195行で12のオペレーション、stateful_random_ops.ccは約160行で9のオペレーションを定義している。stateful_random_ops.ccはREGISTER_STATEFUL_OPマクロで4つのOpを一括登録している。

---

## コードリーディングガイド

### 推奨読解順序

#### Step 1: random_ops.ccのオペレーション構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | random_ops.cc | `tensorflow/core/ops/random_ops.cc` | 26-34行目: RandomUniform - 基本的なステートフル乱数Op構造（shape入力、seed/seed2属性、SetIsStateful） |
| 1-2 | random_ops.cc | `tensorflow/core/ops/random_ops.cc` | 36-61行目: RandomUniformInt - minval/maxvalのスカラー検証付き整数乱数 |
| 1-3 | random_ops.cc | `tensorflow/core/ops/random_ops.cc` | 114-132行目: Multinomial - logits[batch,classes]+num_samples→output[batch,num_samples] |

#### Step 2: stateful_random_ops.ccのリソース管理構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stateful_random_ops.cc | `tensorflow/core/ops/stateful_random_ops.cc` | 23-33行目: StatefulRandomShape - resource+algorithm+shape→output形状推論 |
| 2-2 | stateful_random_ops.cc | `tensorflow/core/ops/stateful_random_ops.cc` | 35-48行目: REGISTER_STATEFUL_OPマクロ - 4つのOpを一括登録 |
| 2-3 | stateful_random_ops.cc | `tensorflow/core/ops/stateful_random_ops.cc` | 83-105行目: RngSkip/RngReadAndSkip - 乱数状態のスキップ操作 |

### プログラム呼び出し階層図

```
random_ops.cc
    +-- RandomUniform (26行目) → RandomShape
    +-- RandomUniformInt (36行目) → スカラー検証 + RandomShape
    +-- RandomStandardNormal (63行目) → RandomShape
    +-- ParameterizedTruncatedNormal (73行目) → ランク検証 + RandomShape
    +-- TruncatedNormal (95行目) → RandomShape
    +-- RandomShuffle (105行目) → UnchangedShape
    +-- Multinomial (114行目) → カスタム形状推論
    +-- RandomGamma (134行目) → カスタム形状推論
    +-- RandomPoisson (158行目, 非推奨)
    +-- RandomPoissonV2 (176行目)

stateful_random_ops.cc
    +-- StatefulRandomShape関数 (23行目)
    +-- REGISTER_STATEFUL_OPマクロ (35行目)
    |    +-- StatefulUniform (45行目)
    |    +-- StatefulUniformFullInt (46行目)
    |    +-- StatefulStandardNormalV2 (47行目)
    |    +-- StatefulTruncatedNormal (48行目)
    +-- StatefulUniformInt (50行目)
    +-- RngSkip (83行目)
    +-- RngReadAndSkip (94行目)
    +-- NonDeterministicInts (107行目)
    +-- StatefulRandomBinomial (121行目)
    +-- StatefulStandardNormal (143行目, 非推奨)
```

### データフロー図

```
[入力]                          [処理]                         [出力]

shape [N] ---------> RandomUniform ---------> output [shape] (float)
shape + min + max -> RandomUniformInt -------> output [shape] (int)
logits [B,C] ------> Multinomial ------------> output [B, num_samples] (int)
shape + alpha -----> RandomGamma ------------> output [shape + alpha.shape]

resource + alg + shape -> StatefulUniform ---> output [shape] (float)
resource + alg + delta -> RngSkip -----------> (状態更新のみ)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| random_ops.cc | `tensorflow/core/ops/random_ops.cc` | ソース | ステートフル乱数Op定義（12 Op） |
| stateful_random_ops.cc | `tensorflow/core/ops/stateful_random_ops.cc` | ソース | リソース管理型乱数Op定義（9 Op） |
| rng_alg.h | `tensorflow/core/framework/rng_alg.h` | ヘッダ | RNG_MAX_COUNTER_SIZE, RNG_KEY_SIZE定数 |
| common_shape_fns.h | `tensorflow/core/framework/common_shape_fns.h` | ヘッダ | RandomShape等の共通形状推論関数 |
