# 機能設計書 56-ステートレス乱数生成

## 概要

本ドキュメントは、TensorFlowにおけるステートレス乱数生成機能の設計を記述する。外部から渡されるシード値のみで乱数を決定するため、同一シードに対して常に同一結果を返す決定論的な乱数生成オペレーション群を提供する。

### 本機能の処理概要

**業務上の目的・背景**：分散学習やモデルの再現性保証において、オペレーション単位で完全に決定論的な乱数生成が必要となる場面がある。ステートレス乱数はグローバル状態に依存せず、入力シード値のみで出力が一意に決まるため、関数型プログラミングパラダイムとの親和性が高い。

**機能の利用シーン**：tf.functionでの決定論的乱数生成、分散学習での同一初期化保証、テスト時の再現性確保で利用される。

**主要な処理内容**：
1. stateless_random_ops.cc（V1）：seed[2]ベースのステートレス乱数
   - StatelessRandomUniform/Normal/TruncatedNormal: 基本分布
   - StatelessRandomUniformInt: 整数一様分布
   - StatelessRandomUniformFullInt: 全範囲整数乱数
   - StatelessMultinomial: 多項分布
   - StatelessRandomBinomial: 二項分布
   - StatelessParameterizedTruncatedNormal: パラメータ付き切断正規分布
   - StatelessRandomPoisson: ポアソン分布
   - StatelessRandomGammaV2: ガンマ分布
2. stateless_random_ops_v2.cc（V2）：key+counter+algベースのステートレス乱数
   - StatelessRandomUniformV2/NormalV2/TruncatedNormalV2: 基本分布
   - StatelessRandomUniformIntV2: 整数一様分布
   - StatelessRandomUniformFullIntV2: 全範囲整数乱数
   - StatelessRandomGammaV3: ガンマ分布
   - StatelessShuffle: シャッフル
   - StatelessRandomGetKeyCounterAlg/GetKeyCounter/GetAlg: シード→key+counter変換

**関連システム・外部連携**：特になし。

**権限による制御**：特段の権限制御はない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能は画面と直接関連しない基盤機能である |

## 機能種別

計算処理 / 乱数生成（決定論的）

## 入力仕様

### 入力パラメータ（V1: stateless_random_ops.cc）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| shape | T (int32/int64) | Yes | 出力テンソルの形状 | ShapeTensor |
| seed | Tseed (int32/int64) | Yes | シード値 [2] | rank=1, dim[0]=2 |
| dtype | 型属性 | No | 出力データ型 | half/bfloat16/float/double |

### 入力パラメータ（V2: stateless_random_ops_v2.cc）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| shape | Tshape (int32/int64) | Yes | 出力テンソルの形状 | ShapeTensor |
| key | uint64 | Yes | 乱数キー | rank=1, dim[0]=RNG_KEY_SIZE |
| counter | uint64 | Yes | カウンタ | rank=1 |
| alg | int32 | Yes | アルゴリズム識別子 | rank=0 |

### 入力データソース

計算グラフ内で直接パラメータ指定。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output | dtype | 指定形状の乱数テンソル |
| key | uint64 | 乱数キー（GetKeyCounterAlg） |
| counter | uint64 | カウンタ（GetKeyCounterAlg） |
| alg | int32 | アルゴリズム（GetKeyCounterAlg） |

### 出力先

後続の計算グラフ。

## 処理フロー

### 処理シーケンス

```
V1 (seed[2]ベース):
1. seed[2]を受け取る（rank=1, dim[0]=2）
2. シードからPhiloxカウンタを初期化
3. 指定された確率分布に従い乱数テンソルを生成
4. 出力テンソルを返却

V2 (key+counter+algベース):
1. key(uint64)+counter(uint64)+alg(int32)を受け取る
2. keyのサイズがRNG_KEY_SIZEであることを検証
3. 指定アルゴリズムで乱数テンソルを生成
4. 出力テンソルを返却
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-56-01 | シード形状検証（V1） | seed入力はrank=1、dim[0]=2でなければならない | V1全Op |
| BR-56-02 | キーサイズ検証（V2） | key入力のdim[0]はRNG_KEY_SIZEでなければならない | V2全Op |
| BR-56-03 | アルゴリズムスカラー（V2） | alg入力はrank=0スカラーでなければならない | V2全Op |
| BR-56-04 | 決定論的出力 | 同一入力に対して常に同一出力を返す（SetIsStatefulは指定されていない） | V1/V2基本Op |
| BR-56-05 | GetAlgのステートフル性 | StatelessRandomGetAlgのみSetIsStateful（デバイス依存出力） | GetAlg |

### 計算ロジック

V1: seed[2]から内部でPhiloxカウンタベースの乱数生成器を初期化。
V2: key+counterを直接使用してPhilox/ThreeFryアルゴリズムで生成。V2はより明示的なアルゴリズム指定が可能。

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgument | シード形状エラー | seed入力がrank!=1またはdim[0]!=2 | [2]形状のシードを入力 |
| InvalidArgument | キーサイズエラー | key入力のdim[0]!=RNG_KEY_SIZE | 正しいサイズのキーを入力 |
| InvalidArgument | スカラーエラー | minval/maxvalがスカラーでない | スカラー値を入力 |

### リトライ仕様

ステートレスであり同一入力で同一出力。リトライに特別な考慮は不要。

## トランザクション仕様

トランザクション制御は行わない。

## パフォーマンス要件

ステートフル版と同等の生成速度。

## セキュリティ考慮事項

暗号用途には使用しないこと。

## 備考

stateless_random_ops.ccは約183行で10のオペレーション、stateless_random_ops_v2.ccは約164行で10のオペレーションを定義している。V2はV1のシードベースのインタフェースをkey+counter+algのより明示的なインタフェースに置き換えたもの。

---

## コードリーディングガイド

### 推奨読解順序

#### Step 1: V1のシードベース構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | stateless_random_ops.cc | `tensorflow/core/ops/stateless_random_ops.cc` | 25-37行目: StatelessShape - seed[2]検証とshape→output形状推論 |
| 1-2 | stateless_random_ops.cc | `tensorflow/core/ops/stateless_random_ops.cc` | 39-53行目: REGISTER_STATELESS_OPマクロ - 3つのOpを一括登録 |

#### Step 2: V2のkey+counter構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | stateless_random_ops_v2.cc | `tensorflow/core/ops/stateless_random_ops_v2.cc` | 26-42行目: StatelessShapeV2 - key+counter+alg検証とshape→output形状推論 |
| 2-2 | stateless_random_ops_v2.cc | `tensorflow/core/ops/stateless_random_ops_v2.cc` | 118-163行目: GetKeyCounterAlg/GetKeyCounter/GetAlg - シード変換ユーティリティOp |

**読解のコツ**: V1のStatelessShapeとV2のStatelessShapeV2を比較すると、入力インタフェースの違いが明確になる。V1はseed[2]一つで全てを制御するが、V2はkey/counter/algを分離して明示的に管理する。

### プログラム呼び出し階層図

```
stateless_random_ops.cc (V1)
    +-- StatelessShape (25行目)
    +-- REGISTER_STATELESS_OPマクロ (39行目)
    |    +-- StatelessRandomUniform (49行目)
    |    +-- StatelessRandomNormal (50行目)
    |    +-- StatelessTruncatedNormal (51行目)
    +-- StatelessRandomUniformInt (55行目)
    +-- StatelessRandomUniformFullInt (81行目)
    +-- StatelessMultinomial (90行目)
    +-- StatelessRandomBinomial (115行目)
    +-- StatelessParameterizedTruncatedNormal (127行目)
    +-- StatelessRandomPoisson (162行目)
    +-- StatelessRandomGammaV2 (173行目)

stateless_random_ops_v2.cc (V2)
    +-- StatelessShapeV2 (26行目)
    +-- REGISTER_STATELESS_OPマクロ (44行目)
    |    +-- StatelessRandomUniformV2 (55行目)
    |    +-- StatelessRandomNormalV2 (56行目)
    |    +-- StatelessTruncatedNormalV2 (57行目)
    +-- StatelessRandomUniformIntV2 (61行目)
    +-- StatelessRandomUniformFullIntV2 (88行目)
    +-- StatelessRandomGammaV3 (98行目)
    +-- StatelessShuffle (109行目)
    +-- StatelessRandomGetKeyCounterAlg (118行目)
    +-- StatelessRandomGetKeyCounter (138行目)
    +-- StatelessRandomGetAlg (156行目)
```

### データフロー図

```
V1:
seed [2] + shape ---> StatelessRandomUniform ---> output [shape]
seed [2] + shape ---> StatelessRandomNormal ----> output [shape]
seed [2] + logits --> StatelessMultinomial ------> output [B, num_samples]

V2:
key + counter + alg + shape ---> StatelessRandomUniformV2 ---> output [shape]
seed [2] ----------------------> GetKeyCounterAlg -----------> key + counter + alg
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| stateless_random_ops.cc | `tensorflow/core/ops/stateless_random_ops.cc` | ソース | V1ステートレス乱数Op定義（10 Op） |
| stateless_random_ops_v2.cc | `tensorflow/core/ops/stateless_random_ops_v2.cc` | ソース | V2ステートレス乱数Op定義（10 Op） |
| rng_alg.h | `tensorflow/core/framework/rng_alg.h` | ヘッダ | RNG_KEY_SIZE, RNG_MAX_COUNTER_SIZE定数 |
| common_shape_fns.h | `tensorflow/core/framework/common_shape_fns.h` | ヘッダ | 共通形状推論関数 |
