# 機能設計書 64-チェックポイント保存・復元

## 概要

本ドキュメントは、TensorFlowにおけるモデルの重みをチェックポイントとして保存・復元する機能の設計を記述する。

### 本機能の処理概要

チェックポイント保存・復元機能は、tf.train.Checkpoint APIを通じて、学習中のモデルの状態（変数値、オプティマイザのスロット変数等）をディスクに保存し、必要に応じて復元する機能である。SavedModelとは異なり、計算グラフやシグネチャは保存せず、変数値のみを効率的に保存する。

**業務上の目的・背景**：深層学習モデルの学習は長時間を要し、ハードウェア障害やプロセス異常終了のリスクがある。チェックポイント機能により学習途中の状態を定期的に保存し、障害発生時に最新のチェックポイントから学習を再開できる。また、最良のモデル状態を保持するためにも使用される。

**機能の利用シーン**：学習中の定期的な重み保存、学習再開時の状態復元、EarlyStopping時のベストモデル復元、分散学習でのチェックポイント同期、モデルの重みのみの転送。

**主要な処理内容**：
1. Trackableオブジェクトグラフの走査とオブジェクト依存関係の解決
2. 変数値のシリアライズとチェックポイントファイルへの書き出し
3. チェックポイントファイルからの変数値読み込みとオブジェクトへの割り当て
4. CheckpointManagerによるチェックポイントのライフサイクル管理
5. 部分復元（expect_partial）とステータス検証

**関連システム・外部連携**：ファイルシステムへの読み書き。メトリクスシステムによる性能計測。

**権限による制御**：チェックポイントディレクトリへの読み書き権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに該当なし |

## 機能種別

データ連携（状態シリアライゼーション）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| root | Trackable | No | チェックポイント対象のルートオブジェクト | Trackableインスタンス |
| **kwargs | Trackable | No | 名前付きTrackableオブジェクト | 各値がTrackable |
| save_path (write) | str | Yes(保存時) | チェックポイントファイルのプレフィックスパス | - |
| save_path (restore) | str | Yes(復元時) | 復元するチェックポイントのパス | ファイルが存在すること |
| options | CheckpointOptions | No | チェックポイントオプション | CheckpointOptionsインスタンス |

### 入力データソース

Python上のTrackableオブジェクト（tf.Variable、tf.Module、Kerasモデル、オプティマイザ等）。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| checkpoint.index | ファイル | チェックポイントのインデックスファイル |
| checkpoint.data-00000-of-XXXXX | ファイル | チェックポイントのデータファイル（シャード化） |
| checkpoint | テキストファイル | 最新チェックポイントパスを記録するファイル |

### 出力先

指定されたsave_pathのディレクトリ配下のファイルシステム。

## 処理フロー

### 処理シーケンス

```
1. Checkpoint オブジェクトの初期化
   └─ root と kwargs から Trackable オブジェクトを登録
2. 保存処理（write）
   └─ 2-1. save_counter のインクリメント
   └─ 2-2. ObjectGraphView による依存関係走査
   └─ 2-3. functional_saver でシリアライズ
   └─ 2-4. チェックポイントファイルへの書き出し
   └─ 2-5. メトリクス記録（ファイルサイズ、所要時間）
3. 復元処理（restore）
   └─ 3-1. CheckpointReader でチェックポイント読み込み
   └─ 3-2. TrackableObjectGraph proto のパース
   └─ 3-3. _CheckpointRestoreCoordinator の構築
   └─ 3-4. オブジェクト依存関係マッチング
   └─ 3-5. 変数値のリストア
   └─ 3-6. 遅延復元（未追跡オブジェクトの後追い復元）
```

### フローチャート

```mermaid
flowchart TD
    A[tf.train.Checkpoint 初期化] --> B{操作種別}
    B -->|save/write| C[save_counter インクリメント]
    C --> D[ObjectGraphView 走査]
    D --> E[functional_saver シリアライズ]
    E --> F[チェックポイントファイル書き出し]
    F --> G[メトリクス記録]
    B -->|restore| H[CheckpointReader 読み込み]
    H --> I[TrackableObjectGraph パース]
    I --> J[_CheckpointRestoreCoordinator 構築]
    J --> K[オブジェクトマッチング]
    K --> L[変数値リストア]
    L --> M[遅延復元処理]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-64-01 | save_counter管理 | 保存ごとにsave_counterが1インクリメントされる | write/save呼び出し時 |
| BR-64-02 | 部分復元 | expect_partial()を呼ぶと未マッチの変数に対する警告が抑制される | 転移学習等で一部の変数のみ復元する場合 |
| BR-64-03 | 遅延復元 | 復元時に未追跡のオブジェクトは後から追跡された時点で復元される | 動的に構築されるモデルの場合 |
| BR-64-04 | チェックポイントV2形式 | TF2ではV2形式（index + シャードデータ）で保存される | 常時（TF2） |

### 計算ロジック

チェックポイントファイルサイズの計算:
```python
size = sum(metrics.CalculateFileSize(f) for f in glob.glob(prefix + "*"))
```

## データベース操作仕様

### 操作別データベース影響一覧

データベース操作は行わない。ファイルシステムへの読み書きのみ。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | オブジェクトグラフ不在 | チェックポイントにOBJECT_GRAPH_PROTO_KEYが存在しない | TF2形式のチェックポイントを使用 |
| NotFoundError | ファイル不存在 | 指定されたチェックポイントパスが存在しない | 正しいパスを指定 |
| AssertionError | コールバックパラメータエラー | コールバック関数のパラメータ数が0または1以外 | パラメータ数を0または1に修正 |

### リトライ仕様

ファイルシステムエラーの場合、ユーザー側でリトライが必要。

## トランザクション仕様

チェックポイント保存はファイルシステムレベルでのアトミック性に依存する。index/dataファイルの書き出し順序により、不完全なチェックポイントを回避する設計。

## パフォーマンス要件

- チェックポイント保存時間はモデルサイズに比例
- メトリクスにより保存時間とファイルサイズが計測される
- _END_TIME_OF_LAST_WRITE で前回書き込みからの経過時間を追跡

## セキュリティ考慮事項

チェックポイントファイルには学習済み重みが含まれるため、アクセス制御が必要。

## 備考

- `_CHECKPOINT_V1`、`_CHECKPOINT_V2`のメトリクスラベルでバージョン別の使用状況が追跡される
- `_SESSION_PROVIDER`はKeras統合のためのセッションプロバイダ

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | checkpoint.py | `tensorflow/python/checkpoint/checkpoint.py` | モジュールレベルの定数と共有状態（75-88行目） |
| 1-2 | checkpoint.py | `tensorflow/python/checkpoint/checkpoint.py` | `ObjectGraphProtoPrettyPrinter`クラス（142-183行目） |
| 1-3 | checkpoint.py | `tensorflow/python/checkpoint/checkpoint.py` | `_CheckpointRestoreCoordinator`クラス（241-348行目） |

**読解のコツ**: `_CheckpointRestoreCoordinator`はチェックポイント復元の中核クラスで、弱参照（WeakValueDictionary）を使用してメモリリークを防ぐ設計となっている。`slot_restorations`は遅延スロット変数復元を管理する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | checkpoint.py | `tensorflow/python/checkpoint/checkpoint.py` | `tf.train.Checkpoint`クラスの定義 |

#### Step 3: チェックポイント管理

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | checkpoint_management.py | `tensorflow/python/checkpoint/checkpoint_management.py` | `generate_checkpoint_state_proto`関数（65-100行目） |
| 3-2 | checkpoint_management.py | `tensorflow/python/checkpoint/checkpoint_management.py` | `_GetCheckpointFilename`関数（48-61行目） |

**主要処理フロー**:
- **48-61行目**: チェックポイント状態ファイルのパス生成（デフォルト名: "checkpoint"）
- **65-100行目**: CheckpointStateプロトコルバッファの生成

#### Step 4: ユーティリティ関数

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | checkpoint.py | `tensorflow/python/checkpoint/checkpoint.py` | `object_metadata`関数（543-577行目）：チェックポイントのメタデータ読み取り |
| 4-2 | checkpoint.py | `tensorflow/python/checkpoint/checkpoint.py` | `list_objects`関数（580-594行目）：Trackableオブジェクトの列挙 |

### プログラム呼び出し階層図

```
tf.train.Checkpoint(**kwargs)
    │
    ├─ write(save_path)  [保存]
    │      ├─ save_counter.assign_add(1)
    │      ├─ graph_view.ObjectGraphView(root)
    │      ├─ save_util.objects_ids_and_slot_variables_and_paths()
    │      ├─ functional_saver.MultiDeviceSaver()
    │      └─ saver.save()  [ファイル書き出し]
    │
    ├─ restore(save_path)  [復元]
    │      ├─ py_checkpoint_reader.NewCheckpointReader()
    │      ├─ _CheckpointRestoreCoordinator()
    │      └─ restore_lib.CheckpointPosition()  [変数値割り当て]
    │
    └─ CheckpointManager(checkpoint, directory, max_to_keep)
           ├─ save()
           └─ restore_or_initialize()
```

### データフロー図

```
[入力]                          [処理]                              [出力]

Trackable objects ──────────▶ ObjectGraphView ──▶ オブジェクトグラフ
                                │
Variable values ────────────▶ functional_saver ──▶ checkpoint.data-*
                                │                     checkpoint.index
                                │
CheckpointState ────────────▶ checkpoint_management ──▶ checkpoint (テキスト)

[復元時]
checkpoint files ───────────▶ CheckpointReader ──▶ _CheckpointRestoreCoordinator
                                                       │
                                                       └──▶ Variable.assign() [値復元]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| checkpoint.py | `tensorflow/python/checkpoint/checkpoint.py` | ソース | Checkpointクラスと復元コーディネータ |
| checkpoint_management.py | `tensorflow/python/checkpoint/checkpoint_management.py` | ソース | CheckpointManagerとチェックポイント状態管理 |
| checkpoint_options.py | `tensorflow/python/checkpoint/checkpoint_options.py` | ソース | チェックポイントオプション定義 |
| functional_saver.py | `tensorflow/python/checkpoint/functional_saver.py` | ソース | 関数型チェックポイント保存 |
| graph_view.py | `tensorflow/python/checkpoint/graph_view.py` | ソース | オブジェクトグラフの走査 |
| save_util.py | `tensorflow/python/checkpoint/save_util.py` | ソース | 保存ユーティリティ |
| restore.py | `tensorflow/python/checkpoint/restore.py` | ソース | 復元ロジック |
