# 機能設計書 77-TensorRT統合

## 概要

本ドキュメントは、TensorFlowにおけるTensorRT統合（TF-TRT）機能の設計について記述する。TF-TRTは、NVIDIA TensorRTを使用してTensorFlowグラフのサブグラフを高速化する統合レイヤーである。

### 本機能の処理概要

**業務上の目的・背景**：NVIDIA GPUでの推論パフォーマンスを最大化するため、TensorFlowの計算グラフの一部をTensorRTエンジンに変換して実行する。TensorRTは、レイヤーフュージョン、精度キャリブレーション（FP16/INT8）、カーネル自動チューニングなどの最適化を提供し、推論レイテンシを大幅に削減する。

**機能の利用シーン**：NVIDIA GPU上での推論ワークロードの高速化、SavedModelの最適化変換、FP16/INT8量子化による推論高速化・メモリ削減。

**主要な処理内容**：
1. グラフセグメント化: TF計算グラフからTensorRTに変換可能なサブグラフ（セグメント）を検出
2. TensorRTエンジン構築: サブグラフをTensorRTエンジン（ICudaEngine）に変換
3. TRTEngineOp: TensorRT実行をTFグラフ内にOpとして統合
4. INT8キャリブレーション: キャリブレーションデータを使用したINT8量子化
5. 形状最適化プロファイル: 動的形状対応のための最適化プロファイル管理
6. タイミングキャッシュ: TensorRTビルダーのタイミング情報キャッシュ
7. LRUキャッシュ: TensorRTエンジンのキャッシュ管理

**関連システム・外部連携**：NVIDIA TensorRT、CUDA、cuDNN

**権限による制御**：特段の権限制御はない。NVIDIA GPUが必要。

## 関連画面

本機能はバックエンドの推論最適化エンジンであるため、直接的な関連画面は存在しない。

## 機能種別

推論最適化 / グラフ変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input_saved_model_dir | string | Yes | 入力SavedModelディレクトリ | 有効なSavedModelパス |
| precision_mode | string | No | 精度モード | FP32/FP16/INT8 |
| max_workspace_size_bytes | int | No | TRT最大ワークスペースサイズ | バイト単位 |
| minimum_segment_size | int | No | 最小セグメントサイズ（ノード数） | デフォルト3 |
| maximum_cached_engines | int | No | キャッシュエンジン最大数 | デフォルト1 |
| use_calibration | bool | No | INT8キャリブレーション使用 | INT8モード時 |
| is_dynamic_op | bool | No | 動的形状対応 | True/False |

### 入力データソース

TensorFlowのSavedModel、Frozen Graph、またはConcreteFunction

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 最適化グラフ | GraphDef | TRTEngineOpを含む最適化済みグラフ |
| TRTエンジン | SerializedEngine | シリアライズされたTensorRTエンジン |
| キャリブレーションデータ | CalibrationData | INT8キャリブレーション結果 |

### 出力先

最適化されたSavedModel、またはインメモリのTensorRTエンジン

## 処理フロー

### 処理シーケンス

```
1. グラフ分析
   └─ TF計算グラフの各ノードをTensorRT対応可否で分類
2. セグメント化
   └─ TRT対応ノードを連結し、サブグラフ（セグメント）を形成
   └─ Union-Find アルゴリズムでセグメントをマージ
   └─ minimum_segment_sizeに満たないセグメントを除外
3. TRTEngineOpへの変換
   └─ 各セグメントをTRTEngineOp Opに置換
   └─ 入出力テンソルの接続を維持
4. TensorRTエンジン構築（実行時またはビルド時）
   ├─ FP32: そのまま構築
   ├─ FP16: FP16精度で構築
   └─ INT8: キャリブレーションデータで構築
5. 実行
   └─ TRTEngineOp内でTensorRTエンジンを実行
   └─ 入力形状に応じてエンジンをキャッシュから選択
```

### フローチャート

```mermaid
flowchart TD
    A[TF Graph入力] --> B[ノードのTRT対応分類]
    B --> C[Union-Findセグメント化]
    C --> D{minimum_segment_size超?}
    D -->|Yes| E[TRTEngineOpに変換]
    D -->|No| F[TFネイティブ実行のまま]
    E --> G{精度モード}
    G -->|FP32| H[FP32エンジン構築]
    G -->|FP16| I[FP16エンジン構築]
    G -->|INT8| J[キャリブレーション→INT8エンジン構築]
    H --> K[実行]
    I --> K
    J --> K
    F --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-77-01 | 最小セグメントサイズ | ノード数がminimum_segment_size未満のセグメントはTRT変換しない | セグメント化時 |
| BR-77-02 | OP変換可否 | 各TF OpにはTensorRT向け変換関数（OpConverter）の登録が必要 | グラフ分析時 |
| BR-77-03 | エンジンキャッシュ | maximum_cached_engines数を超えるとLRUで古いエンジンを破棄 | 動的形状時 |
| BR-77-04 | 形状プロファイル | 動的形状時は最小/最適/最大形状のプロファイルを管理 | is_dynamic_op=True |

### 計算ロジック

- セグメント化: Union-Findアルゴリズムで連結コンポーネントを検出
- LRUキャッシュ: Least Recently Used方式でエンジンを管理

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgumentError | TRT非対応Op | セグメント内にTRT非対応Opがある | minimum_segment_sizeを大きくする |
| InternalError | エンジン構築失敗 | TensorRTエンジンの構築が失敗 | ワークスペースサイズの増大 |
| ResourceExhaustedError | GPUメモリ不足 | エンジン構築時のGPUメモリ不足 | max_workspace_size_bytesの削減 |

### リトライ仕様

TensorRTエンジン構築失敗時は、該当セグメントをTFネイティブ実行にフォールバック。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- FP16モードでFP32比1.5-3倍の推論高速化
- INT8モードでFP32比2-4倍の推論高速化
- 初回エンジン構築のオーバーヘッドあり（タイミングキャッシュで軽減可能）

## セキュリティ考慮事項

- シリアライズされたTensorRTエンジンはデバイス固有であり、異なるGPUアーキテクチャでは使用不可

## 備考

- TF-TRTはGrappler最適化パスとして実装（TRTOptimizationPass）
- TRTLayoutOptimizationPassによるテンソルレイアウト最適化（NHWC→NCHW等）
- アルゴリズムセレクタ（AlgorithmSelector）によるTRT実行アルゴリズムの選択
- Experimental Features機能による実験的機能の有効化

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: セグメント化ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | segment.cc | `tensorflow/compiler/tf2tensorrt/segment/segment.cc` | Union-Findベースのグラフセグメント化 |
| 1-2 | union_find.cc | `tensorflow/compiler/tf2tensorrt/segment/union_find.cc` | Union-Findデータ構造 |

#### Step 2: Op変換を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | convert_nodes.cc | `tensorflow/compiler/tf2tensorrt/convert/convert_nodes.cc` | TF Op→TRT Layer変換のメインロジック |
| 2-2 | convert_graph.cc | `tensorflow/compiler/tf2tensorrt/convert/convert_graph.cc` | グラフレベルの変換 |
| 2-3 | op_converter_registry.cc | `tensorflow/compiler/tf2tensorrt/convert/op_converter_registry.cc` | Op変換関数のレジストリ |
| 2-4 | ops/ | `tensorflow/compiler/tf2tensorrt/convert/ops/` | 各Op種別の変換実装（binary_ops, unary_ops等） |

#### Step 3: エンジン管理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | trt_engine_op.cc | `tensorflow/compiler/tf2tensorrt/kernels/trt_engine_op.cc` | TRTEngineOp カーネル実装 |
| 3-2 | trt_engine_resource_ops.cc | `tensorflow/compiler/tf2tensorrt/kernels/trt_engine_resource_ops.cc` | エンジンリソース管理 |
| 3-3 | trt_lru_cache.cc | `tensorflow/compiler/tf2tensorrt/utils/trt_lru_cache.cc` | LRUキャッシュ |

#### Step 4: Grappler統合を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | trt_optimization_pass.cc | `tensorflow/compiler/tf2tensorrt/convert/trt_optimization_pass.cc` | Grappler最適化パス |
| 4-2 | trt_layout_optimization_pass.cc | `tensorflow/compiler/tf2tensorrt/convert/trt_layout_optimization_pass.cc` | レイアウト最適化パス |

### プログラム呼び出し階層図

```
Grappler最適化パイプライン
    │
    └─ TRTOptimizationPass
           │
           ├─ segment.cc (グラフセグメント化)
           │   └─ union_find.cc (Union-Findアルゴリズム)
           │
           ├─ convert_graph.cc (サブグラフ→TRTEngineOp変換)
           │   └─ convert_nodes.cc (個別Op変換)
           │       └─ op_converter_registry.cc (変換関数レジストリ)
           │           └─ ops/*.cc (各Op種別の変換)
           │
           └─ trt_engine_op.cc (TRTEngineOp実行カーネル)
               ├─ trt_lru_cache.cc (エンジンキャッシュ)
               ├─ trt_shape_optimization_profiles.cc (形状プロファイル)
               ├─ trt_int8_calibrator.cc (INT8キャリブレーション)
               └─ trt_allocator.cc (メモリアロケータ)
```

### データフロー図

```
[入力]                       [処理]                              [出力]

TF Graph ──────▶  セグメント化  ──▶  TRT対応セグメント
                      │
                      └─ Op変換  ──▶  TRTEngineOp含むグラフ
                      │
                      └─ エンジン構築  ──▶  TRTエンジン（キャッシュ）
                      │
                      └─ 実行  ──▶  推論結果
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| segment.cc | `tensorflow/compiler/tf2tensorrt/segment/segment.cc` | ソース | グラフセグメント化 |
| union_find.cc | `tensorflow/compiler/tf2tensorrt/segment/union_find.cc` | ソース | Union-Findデータ構造 |
| convert_nodes.cc | `tensorflow/compiler/tf2tensorrt/convert/convert_nodes.cc` | ソース | Op変換メインロジック |
| convert_graph.cc | `tensorflow/compiler/tf2tensorrt/convert/convert_graph.cc` | ソース | グラフレベル変換 |
| op_converter_registry.cc | `tensorflow/compiler/tf2tensorrt/convert/op_converter_registry.cc` | ソース | Op変換レジストリ |
| trt_engine_op.cc | `tensorflow/compiler/tf2tensorrt/kernels/trt_engine_op.cc` | ソース | TRTEngineOpカーネル |
| trt_optimization_pass.cc | `tensorflow/compiler/tf2tensorrt/convert/trt_optimization_pass.cc` | ソース | Grappler最適化パス |
| trt_layout_optimization_pass.cc | `tensorflow/compiler/tf2tensorrt/convert/trt_layout_optimization_pass.cc` | ソース | レイアウト最適化パス |
| trt_lru_cache.cc | `tensorflow/compiler/tf2tensorrt/utils/trt_lru_cache.cc` | ソース | LRUキャッシュ |
| trt_int8_calibrator.cc | `tensorflow/compiler/tf2tensorrt/utils/trt_int8_calibrator.cc` | ソース | INT8キャリブレーション |
| trt_shape_optimization_profiles.cc | `tensorflow/compiler/tf2tensorrt/utils/trt_shape_optimization_profiles.cc` | ソース | 形状プロファイル |
| trt_allocator.cc | `tensorflow/compiler/tf2tensorrt/utils/trt_allocator.cc` | ソース | メモリアロケータ |
| algorithm_selector.cc | `tensorflow/compiler/tf2tensorrt/convert/algorithm_selector.cc` | ソース | アルゴリズム選択 |
| trt_convert_api.cc | `tensorflow/compiler/tf2tensorrt/trt_convert_api.cc` | ソース | 変換API |
| weights.cc | `tensorflow/compiler/tf2tensorrt/convert/weights.cc` | ソース | 重み変換 |
| trt_logger.cc | `tensorflow/compiler/tf2tensorrt/utils/trt_logger.cc` | ソース | ロガー |
| timing_cache.cc | `tensorflow/compiler/tf2tensorrt/convert/timing_cache.cc` | ソース | タイミングキャッシュ |
