# 機能設計書 81-TFLite推論エンジン

## 概要

本ドキュメントは、TensorFlow Liteの軽量推論エンジン機能について、その設計・処理内容・インタフェース仕様を記述した機能設計書である。モバイル・エッジデバイス向けに最適化されたモデル推論の実行基盤を提供する。

### 本機能の処理概要

TFLite推論エンジンは、TFLite形式（FlatBuffer）に変換されたモデルをモバイル・エッジデバイス上で高速に実行するための軽量推論ランタイムを提供する機能である。

**業務上の目的・背景**：モバイルアプリケーションや組込みデバイスにおいて、機械学習モデルをリアルタイムに実行する需要が急速に拡大している。通常のTensorFlowランタイムはサーバ向けに設計されており、モバイルデバイスのメモリ・計算リソース制約下では動作が困難である。TFLite推論エンジンは、この課題を解決するためにモデルサイズの削減、起動時間の短縮、低メモリフットプリントを実現する軽量ランタイムとして設計されている。

**機能の利用シーン**：Androidアプリでのリアルタイム画像分類・物体検出、iOSアプリでの音声認識、IoTデバイスでのセンサーデータ推論、エッジコンピューティング環境でのモデルデプロイなど、リソース制約のある環境でのモデル推論実行時に利用される。

**主要な処理内容**：
1. FlatBuffer形式のモデルファイルの読み込みとパース（FlatBufferModel）
2. InterpreterBuilderによるInterpreterインスタンスの構築
3. OpResolverによる演算カーネルの解決・登録
4. テンソルメモリの割り当て（AllocateTensors）
5. 入力テンソルへのデータ設定
6. 推論の実行（Invoke）
7. 出力テンソルからの結果取得
8. SignatureRunnerによる名前付きシグネチャベースの推論実行
9. 非同期推論（AsyncSignatureRunner）のサポート

**関連システム・外部連携**：GPU/NNAPI/CoreML/XNNPACKなどのハードウェアデリゲート（機能No.82）、TFLiteモデル変換（機能No.80）と連携する。Android/iOS/Linuxプラットフォームのネイティブライブラリとして動作する。

**権限による制御**：特に権限による制御はないが、デリゲートの利用可否はハードウェアおよびOS APIのサポート状況に依存する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | 画像分類画面 | 主画面 | TensorFlowImageClassifierを通じてInceptionモデルによる画像分類推論を実行 |
| 2 | 物体検出画面 | 主画面 | TF Object Detection API / MultiBox / YOLOモデルによる物体検出推論を実行 |
| 3 | スタイル変換画面 | 主画面 | stylize_quantized.pbモデルによるNeural Style Transfer推論を実行 |
| 4 | 音声認識画面 | 主画面 | conv_actions_frozen.pbモデルによる音声コマンド認識推論を実行 |
| 9 | TFLiteカメラプレビューフラグメント | 主画面 | ImageClassifierを通じてTFLite推論エンジンで画像分類推論を実行 |
| 10 | OVICベンチマーク画面 | 主画面 | OvicClassifierBenchmarkerを通じてTFLiteモデルのベンチマーク推論を実行 |
| 11 | TFLiteベンチマーク実行画面 | 主画面 | BenchmarkModel.run()を通じてTFLiteモデルのベンチマーク実行 |
| 12 | デリゲート性能ベンチマーク（レイテンシ）画面 | 主画面 | BenchmarkLatencyImplを通じてレイテンシベンチマークを実行 |
| 13 | デリゲート性能ベンチマーク（精度）画面 | 主画面 | BenchmarkAccuracyImplを通じて精度ベンチマークを実行 |
| 14 | Firebase Game Loopベンチマーク画面 | 主画面 | Firebase Test Lab上でTFLiteモデルのベンチマーク実行 |

## 機能種別

計算処理（モデル推論実行）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| model_path | const char* | Yes | TFLiteモデルファイルのパス | ファイル存在確認、FlatBuffer形式検証 |
| model_buffer | const void* | Yes（パス指定しない場合） | モデルデータのメモリバッファ | バッファサイズ > 0、FlatBuffer形式検証 |
| num_threads | int | No | 推論に使用するスレッド数（デフォルト: 1） | >= 1 |
| input_tensors | TfLiteTensor* | Yes | 推論入力テンソル配列 | テンソル型・形状がモデル定義と一致 |
| options | TfLiteInterpreterOptions* | No | インタプリタオプション | NULL許可 |

### 入力データソース

FlatBuffer形式のTFLiteモデルファイル（.tflite）またはメモリ上のモデルバッファ。入力テンソルデータはアプリケーション層から提供される（画像データ、音声データ、テキスト特徴量など）。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output_tensors | const TfLiteTensor* | 推論結果テンソル配列 |
| status | TfLiteStatus | 推論実行結果ステータス（kTfLiteOk/kTfLiteError/kTfLiteDelegateError） |
| profiling_data | ProfileEvent* | プロファイリングデータ（有効化時） |

### 出力先

アプリケーション層（推論結果テンソル）。プロファイリングデータはProfilerを通じて取得可能。

## 処理フロー

### 処理シーケンス

```
1. モデル読み込み
   └─ FlatBufferModel::BuildFromFile() または BuildFromBuffer() でモデルをロード
2. Interpreter構築
   └─ InterpreterBuilder(model, op_resolver) でインタプリタを生成
3. デリゲート適用（オプション）
   └─ Interpreter::ModifyGraphWithDelegate() でハードウェアデリゲートを適用
4. テンソルメモリ割り当て
   └─ Interpreter::AllocateTensors() で入出力テンソルのメモリを確保
5. 入力データ設定
   └─ TfLiteTensorCopyFromBuffer() で入力テンソルにデータをコピー
6. 推論実行
   └─ Interpreter::Invoke() で全サブグラフの演算を順次実行
7. 出力データ取得
   └─ TfLiteTensorCopyToBuffer() で出力テンソルから結果を取得
8. リソース解放
   └─ Interpreter / Model オブジェクトの破棄
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[FlatBufferModel::BuildFromFile]
    B --> C{モデル読み込み成功?}
    C -->|No| Z[エラー返却]
    C -->|Yes| D[InterpreterBuilder構築]
    D --> E{Interpreter生成成功?}
    E -->|No| Z
    E -->|Yes| F{デリゲート適用?}
    F -->|Yes| G[ModifyGraphWithDelegate]
    F -->|No| H[AllocateTensors]
    G --> H
    H --> I{メモリ割り当て成功?}
    I -->|No| Z
    I -->|Yes| J[入力テンソルにデータ設定]
    J --> K[Invoke実行]
    K --> L{推論成功?}
    L -->|No| Z
    L -->|Yes| M[出力テンソルから結果取得]
    M --> N[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-81-01 | モデル寿命管理 | FlatBufferModelインスタンスはInterpreterの寿命以上に存在する必要がある | 常時 |
| BR-81-02 | スレッドセーフティ | Interpreterはスレッドセーフではなく、呼び出し元がシリアライズを保証する必要がある | マルチスレッド環境 |
| BR-81-03 | テンソル型一致 | 入力テンソルのデータ型・形状はモデル定義と一致する必要がある | 推論実行時 |
| BR-81-04 | メモリ再割り当て | 入力テンソルのサイズ変更後はAllocateTensorsの再呼び出しが必要 | 動的形状テンソル使用時 |

### 計算ロジック

推論実行はモデルグラフのトポロジカル順序に従い、各ノード（オペレータ）のカーネル関数を順次呼び出す。各カーネルは入力テンソルを読み取り、演算結果を出力テンソルに書き込む。メモリプランナーがテンソル間のメモリ共有を最適化する。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（TFLite推論エンジンはデータベースを使用しない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| kTfLiteError | 一般エラー | モデルファイル不正、メモリ不足、演算エラー | エラーレポータのメッセージを確認し原因を特定 |
| kTfLiteDelegateError | デリゲートエラー | デリゲートの初期化失敗、未サポート演算 | デリゲートなしでフォールバック実行を検討 |
| kTfLiteApplicationError | アプリケーションエラー | Cancel()による推論中断 | 必要に応じて再実行 |
| kTfLiteUnresolvedOps | 未解決演算 | OpResolverにカスタムOp未登録 | カスタムOpの登録を追加 |

### リトライ仕様

推論エラー時の自動リトライは実装されていない。アプリケーション層での明示的な再実行が必要。デリゲートエラー時はCPUフォールバックが推奨される。

## トランザクション仕様

該当なし（TFLite推論エンジンはトランザクション管理を行わない）

## パフォーマンス要件

- モデル読み込み時間: モデルサイズ依存（典型的には数十〜数百ミリ秒）
- 推論レイテンシ: モデル複雑度・デバイス性能に依存（MobileNetV2で典型的に数〜数十ミリ秒）
- メモリフットプリント: モデルサイズ + テンソルバッファサイズ（典型的に数〜数十MB）
- XNNPACK最適化によりCPU推論性能が大幅に向上

## セキュリティ考慮事項

- モデルファイルのFlatBuffer形式検証（Verifier）による不正データ検出
- バッファオーバーフロー防止のためのテンソルサイズ検証
- 信頼できないソースからのモデル読み込み時はサンドボックス環境での実行を推奨

## 備考

- C API（c_api.h）は安定したABI境界を提供し、言語バインディングの基盤となる
- Python/Java/Swift/Objective-C等のラッパーはC APIを通じてアクセス
- TFLite 2.xではInterpreterBuilderの使用が推奨（直接コンストラクタは非推奨）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、TFLiteの基本データ型とテンソル構造を理解することが重要である。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | common.h | `tensorflow/lite/core/c/common.h` | TfLiteTensor、TfLiteNode、TfLiteContext、TfLiteRegistrationなどの基本構造体定義 |
| 1-2 | c_api_types.h | `tensorflow/lite/core/c/c_api_types.h` | TfLiteStatus列挙型、TfLiteType列挙型の定義 |
| 1-3 | builtin_op_data.h | `tensorflow/lite/core/c/builtin_op_data.h` | 組込み演算のパラメータ構造体（TfLiteConvParams等） |

**読解のコツ**: TFLiteはC/C++混在のコードベースである。C APIは`extern "C"`ブロック内に定義され、ABI安定性が保証される。構造体は`TfLite`プレフィックスで統一されている。

#### Step 2: エントリーポイントを理解する

C APIのエントリーポイントから推論フローを追う。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | c_api.h | `tensorflow/lite/core/c/c_api.h` | C APIの公開インタフェース（TfLiteModelCreate、TfLiteInterpreterCreate、TfLiteInterpreterInvoke） |

**主要処理フロー**:
1. **48-77行目**: C APIの使用例（モデル生成→インタプリタ生成→テンソル割り当て→推論実行→結果取得→解放）

#### Step 3: モデル読み込み層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | model_builder.h | `tensorflow/lite/core/model_builder.h` | FlatBufferModelクラスの定義。BuildFromFile/BuildFromBufferによるモデルロード |

#### Step 4: Interpreter構築層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | interpreter_builder.h | `tensorflow/lite/core/interpreter_builder.h` | InterpreterBuilderクラス。FlatBufferModel + OpResolverからInterpreterを構築 |
| 4-2 | op_resolver.h | `tensorflow/lite/core/api/op_resolver.h` | OpResolverインタフェース。演算名から登録情報（TfLiteRegistration）を解決 |
| 4-3 | register.h | `tensorflow/lite/core/kernels/register.h` | BuiltinOpResolverクラス。組込み演算カーネルの一括登録 |

**主要処理フロー**:
- **74-92行目（interpreter_builder.h）**: InterpreterBuilderのコンストラクタとoperator()による構築処理

#### Step 5: Interpreter実行層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | interpreter.h | `tensorflow/lite/core/interpreter.h` | Interpreterクラスの定義。AllocateTensors、Invoke、入出力テンソルアクセス |
| 5-2 | subgraph.h | `tensorflow/lite/core/subgraph.h` | Subgraphクラス。実際の演算グラフ実行を担当 |

**主要処理フロー**:
- **87-122行目（interpreter.h）**: Interpreterの使用例とクラス定義
- **128-249行目（interpreter.h）**: Interpreterの公開メソッド（SetInputs、SetOutputs、AllocateTensors等）
- **73-100行目（subgraph.h）**: Subgraphクラスのコンストラクタとメンバ関数

#### Step 6: SignatureRunnerを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 6-1 | signature_runner.h | `tensorflow/lite/core/signature_runner.h` | 名前付きシグネチャベースの推論実行インタフェース |
| 6-2 | async_signature_runner.h | `tensorflow/lite/core/async/async_signature_runner.h` | 非同期推論のSignatureRunner |

### プログラム呼び出し階層図

```
TfLiteModelCreateFromFile (C API)
    |
    +-- FlatBufferModel::BuildFromFile
    |       +-- FlatBuffer Verifier
    |
TfLiteInterpreterCreate (C API)
    |
    +-- InterpreterBuilder::operator()
    |       +-- OpResolver::FindOp (各演算のカーネル解決)
    |       +-- Subgraph::AddTensors
    |       +-- Subgraph::AddNodeWithParameters
    |
TfLiteInterpreterAllocateTensors (C API)
    |
    +-- Interpreter::AllocateTensors
            +-- Subgraph::AllocateTensors
                    +-- MemoryPlanner::PlanAllocations
                    +-- TfLiteRegistration::prepare (各Opのprepare)
    |
TfLiteInterpreterInvoke (C API)
    |
    +-- Interpreter::Invoke
            +-- Subgraph::Invoke
                    +-- TfLiteRegistration::invoke (各Opのinvoke)
                            +-- 組込みカーネル実行
                            +-- デリゲートカーネル実行
```

### データフロー図

```
[入力]                          [処理]                              [出力]

モデルファイル (.tflite) ──▶ FlatBufferModel ──▶ InterpreterBuilder
                                                        |
                                                        v
入力データ (画像/音声等) ──▶ Interpreter::Invoke ──▶ 出力テンソル (推論結果)
                               |         |
                               v         v
                          Subgraph    MemoryPlanner
                               |
                               v
                        各Op Kernel実行
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| interpreter.h | `tensorflow/lite/core/interpreter.h` | ソース（ヘッダ） | Interpreterクラスの定義 |
| interpreter_builder.h | `tensorflow/lite/core/interpreter_builder.h` | ソース（ヘッダ） | InterpreterBuilderクラスの定義 |
| subgraph.h | `tensorflow/lite/core/subgraph.h` | ソース（ヘッダ） | Subgraphクラスの定義 |
| model_builder.h | `tensorflow/lite/core/model_builder.h` | ソース（ヘッダ） | FlatBufferModelクラスの定義 |
| c_api.h | `tensorflow/lite/core/c/c_api.h` | ソース（ヘッダ） | C言語公開API |
| common.h | `tensorflow/lite/core/c/common.h` | ソース（ヘッダ） | 基本データ構造定義 |
| c_api_types.h | `tensorflow/lite/core/c/c_api_types.h` | ソース（ヘッダ） | 型定義・列挙型 |
| op_resolver.h | `tensorflow/lite/core/api/op_resolver.h` | ソース（ヘッダ） | OpResolverインタフェース |
| register.h | `tensorflow/lite/core/kernels/register.h` | ソース（ヘッダ） | 組込みOp登録 |
| signature_runner.h | `tensorflow/lite/core/signature_runner.h` | ソース（ヘッダ） | SignatureRunner定義 |
| async_signature_runner.h | `tensorflow/lite/core/async/async_signature_runner.h` | ソース（ヘッダ） | 非同期推論 |
| profiler.h | `tensorflow/lite/core/api/profiler.h` | ソース（ヘッダ） | プロファイラインタフェース |
| verifier.h | `tensorflow/lite/core/tools/verifier.h` | ソース（ヘッダ） | モデル検証 |
