# 機能設計書 86-自動混合精度

## 概要

本ドキュメントは、TensorFlow/Kerasの自動混合精度（Auto Mixed Precision）機能について、その設計・処理内容・インタフェース仕様を記述した機能設計書である。FP16/BF16への自動精度変換によるモデル学習・推論の高速化を提供する。

### 本機能の処理概要

自動混合精度機能は、モデルの計算において一部の演算をFP32からFP16（float16）またはBF16（bfloat16）に自動変換することで、計算速度を向上させつつ数値精度を維持する機能である。Keras層のdtypeポリシーとGrapplerの自動混合精度パスの2つのレベルで提供される。

**業務上の目的・背景**：ディープラーニングの学習・推論において、全ての演算をFP32で行うことは計算リソースの無駄になることがある。NVIDIA Volta以降のGPUやGoogle TPUはFP16/BF16の演算を高速に実行できるテンソルコアを搭載しており、これらを活用することで学習速度を2〜3倍に向上させることが可能である。ただし、一部の演算（損失計算、勾配計算等）はFP32の精度が必要であり、適切な精度管理が不可欠である。

**機能の利用シーン**：GPU/TPU上でのモデル学習高速化、推論の高速化、メモリ使用量の削減（FP16テンソルはFP32の半分のメモリ）。

**主要な処理内容**：
1. Policyクラスによるdtypeポリシーの定義（mixed_float16 / mixed_bfloat16）
2. set_global_policyによるグローバルポリシー設定
3. AutocastVariableによる変数の自動型変換
4. LossScaleOptimizerによる損失スケーリング（FP16使用時のアンダーフロー防止）
5. デバイス互換性チェック（GPU Compute Capability確認）
6. Keras層の入力テンソル自動キャスト

**関連システム・外部連携**：Kerasモデル構築API（No.131）、Grappler最適化（No.84）、各オプティマイザ（No.27-34）と連携する。

**権限による制御**：特になし。ただし、GPU/TPUの対応状況に依存する。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | バックエンドの学習最適化機能であり、直接的な画面関連はない |

## 機能種別

計算処理（精度管理・学習最適化）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| name | string | Yes | ポリシー名（'mixed_float16'/'mixed_bfloat16'/'float32'等） | 有効なdtype名またはmixedポリシー名 |
| loss_scale | LossScale/string/float | No | 損失スケール（'dynamic'等） | - |

### 入力データソース

アプリケーション層からのポリシー設定。Keras層のdtype引数。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| compute_dtype | string | 計算に使用するデータ型 |
| variable_dtype | string | 変数に使用するデータ型 |
| loss_scale | LossScale | 損失スケーリング設定 |

### 出力先

Keras層の計算パイプライン。オプティマイザの勾配計算。

## 処理フロー

### 処理シーケンス

```
1. ポリシー設定
   └─ set_global_policy('mixed_float16') でグローバルポリシーを設定
2. デバイス互換性チェック
   └─ GPUのCompute Capabilityを確認（7.0以上推奨）
3. 層構築時
   └─ 各Keras層がポリシーのcompute_dtypeとvariable_dtypeを取得
4. 順伝搬時
   └─ 入力テンソルをcompute_dtypeにキャスト→計算→出力
5. 変数アクセス時
   └─ AutocastVariableがvariable_dtype(FP32)で保持しcompute_dtype(FP16)で読み出し
6. 損失スケーリング
   └─ LossScaleOptimizerが勾配をスケールアップ→更新→スケールダウン
7. 動的損失スケール調整
   └─ 勾配のInf/NaN検出→スケール調整
```

### フローチャート

```mermaid
flowchart TD
    A[set_global_policy] --> B[デバイス互換性チェック]
    B --> C[Keras層構築]
    C --> D[順伝搬: 入力をFP16にキャスト]
    D --> E[FP16で計算実行]
    E --> F[損失計算: FP32で実行]
    F --> G[LossScaleOptimizer: 勾配スケールアップ]
    G --> H{勾配にInf/NaN?}
    H -->|Yes| I[スケール縮小・ステップスキップ]
    H -->|No| J[FP32で変数更新]
    J --> K[スケール拡大]
    I --> D
    K --> D
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-86-01 | compute/variable dtype分離 | mixed_float16ではcompute=float16、variable=float32 | mixed_*ポリシー使用時 |
| BR-86-02 | 自動LossScaleOptimizer | Model.compileはmixed_float16使用時にオプティマイザを自動ラップ | Model.compile使用時 |
| BR-86-03 | カスタムループでの手動ラップ | カスタムトレーニングループではLossScaleOptimizerの明示的使用が必要 | カスタムループ使用時 |
| BR-86-04 | 入力テンソルキャスト | call()の第1引数のみ自動キャストされる | Keras層のcall()呼び出し時 |

### 計算ロジック

- **損失スケーリング**：損失値をscale_factorで乗算→逆伝搬→勾配をscale_factorで除算
- **動的スケール調整**：N回連続で有限勾配→スケール2倍、Inf/NaN検出→スケール1/2
- **is_finite判定**：全勾配に対してmath_ops.reduce_all(math_ops.is_finite(...))で判定

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| 数値アンダーフロー | 計算エラー | FP16でのアンダーフロー | 損失スケーリングの適用を確認 |
| GPU非対応警告 | 警告 | Compute Capability < 7.0 | 対応GPUの使用またはmixed_bfloat16の使用 |

### リトライ仕様

動的損失スケーリングにより、Inf/NaN勾配検出時はステップがスキップされスケールが調整される（自動リカバリ）。

## トランザクション仕様

該当なし

## パフォーマンス要件

- NVIDIA V100/A100等のTensor Core対応GPUで学習速度2〜3倍向上
- メモリ使用量はFP16テンソルにより約半分に削減可能

## セキュリティ考慮事項

- 混合精度は数値精度に影響するため、セキュリティクリティカルな計算には精度検証が必要

## 備考

- mixed_bfloat16はTPUおよびAmpere以降のGPUで推奨
- BF16は損失スケーリングが不要（指数部がFP32と同じ）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | policy.py | `tensorflow/python/keras/mixed_precision/policy.py` | Policyクラス - compute_dtype / variable_dtypeの定義 |

**読解のコツ**: Policyは「dtype名」で構成される軽量オブジェクト。'mixed_float16'はcompute=float16、variable=float32を意味する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | policy.py | `tensorflow/python/keras/mixed_precision/policy.py` | Policyクラスとset_global_policy |

**主要処理フロー**:
1. **30-68行目**: Policyクラスのdocstring - 混合精度の概要と使用例
2. **39-43行目**: nameパラメータの説明（'mixed_float16'/'mixed_bfloat16'）
3. **53-62行目**: 使用例（set_global_policy→層構築→自動適用）
4. **70-74行目**: Model.compileの自動LossScaleOptimizerラップ

#### Step 3: AutocastVariableを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | autocast_variable.py | `tensorflow/python/keras/mixed_precision/autocast_variable.py` | 変数の自動型変換メカニズム |

#### Step 4: LossScaleOptimizerを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | loss_scale_optimizer.py | `tensorflow/python/keras/mixed_precision/loss_scale_optimizer.py` | 損失スケーリングオプティマイザ |

**主要処理フロー**:
- **47-57行目**: _UnwrapPreventerクラス - DistributionStrategy対応
- **65-75行目**: _is_all_finite関数 - 全勾配の有限性チェック

#### Step 5: デバイス互換性チェックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | device_compatibility_check.py | `tensorflow/python/keras/mixed_precision/device_compatibility_check.py` | GPU互換性チェック |

### プログラム呼び出し階層図

```
tf.keras.mixed_precision.set_global_policy('mixed_float16')
    |
    +-- Policy('mixed_float16')
    |       +-- compute_dtype = 'float16'
    |       +-- variable_dtype = 'float32'
    |
    +-- device_compatibility_check
            +-- GPU Compute Capability確認

Keras Layer.call(inputs)
    |
    +-- 入力をcompute_dtype(float16)にキャスト
    +-- AutocastVariable.read_value() → float16で読み出し
    +-- 計算実行(float16)
    +-- 出力(float16)

Model.compile(optimizer)
    |
    +-- LossScaleOptimizer(optimizer) でラップ
            +-- apply_gradients()
                    +-- _is_all_finite(grads)
                    +-- 動的スケール調整
```

### データフロー図

```
[入力]                        [処理]                          [出力]

Policy設定 ──────────▶ compute_dtype / variable_dtype
                              |
入力テンソル(FP32) ──▶ 自動キャスト(FP16) ──▶ Keras層計算(FP16) ──▶ 出力(FP16)
                              |
変数(FP32) ────────▶ AutocastVariable ──▶ 読み出し(FP16)
                              |
損失(FP32) ────────▶ LossScaleOptimizer ──▶ スケール済み勾配 ──▶ 変数更新(FP32)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| policy.py | `tensorflow/python/keras/mixed_precision/policy.py` | ソース | Policyクラス・グローバルポリシー管理 |
| autocast_variable.py | `tensorflow/python/keras/mixed_precision/autocast_variable.py` | ソース | 自動型変換変数 |
| loss_scale_optimizer.py | `tensorflow/python/keras/mixed_precision/loss_scale_optimizer.py` | ソース | 損失スケーリングオプティマイザ |
| loss_scale.py | `tensorflow/python/keras/mixed_precision/loss_scale.py` | ソース | 損失スケール管理 |
| device_compatibility_check.py | `tensorflow/python/keras/mixed_precision/device_compatibility_check.py` | ソース | デバイス互換性チェック |
| get_layer_policy.py | `tensorflow/python/keras/mixed_precision/get_layer_policy.py` | ソース | 層からポリシー取得 |
| __init__.py | `tensorflow/python/keras/mixed_precision/__init__.py` | ソース | パッケージ初期化 |
