# 機能設計書 88-tf.function（関数トレーシング）

## 概要

本ドキュメントは、TensorFlowのtf.functionデコレータによるPython関数のグラフトレーシング・変換機能について、その設計・処理内容・インタフェース仕様を記述した機能設計書である。

### 本機能の処理概要

tf.functionは、Python関数をTensorFlowの計算グラフにトレース・変換するデコレータである。関数を一度トレースしてGraphDefに変換し、以降の呼び出しではコンパイル済みグラフを再利用することで、Eager実行のオーバーヘッドを排除し高速実行を実現する。

**業務上の目的・背景**：Eager Execution（No.87）はデバッグが容易だが、Pythonインタプリタのオーバーヘッドにより大規模モデルでは性能が低下する。tf.functionはPythonの書きやすさとグラフ実行の高速性を両立させる。関数の入力シグネチャ（引数の型・形状）ごとにグラフをキャッシュする「多相関数」として動作し、Variable初期化の特殊パターンもサポートする。

**機能の利用シーン**：モデルの学習ループの高速化、推論関数のグラフ変換・最適化、SavedModelへのエクスポート、XLA JITコンパイルとの組み合わせなど。

**主要な処理内容**：
1. @tf.functionデコレータの適用
2. 初回呼び出し時のPython関数トレーシング
3. ConcreteFunction（具体的なグラフ）の生成
4. 入力シグネチャに基づくFunction Cache管理
5. Variable初期化パターンのサポート（リフティング/UnliftedInitializerVariable）
6. AutoGraph変換（No.89）との統合
7. jit_compileオプションによるXLA JITコンパイル
8. input_signatureによる入力仕様の明示指定
9. 再トレーシングの管理と警告

**関連システム・外部連携**：AutoGraph変換（No.89）、Eager実行コンテキスト（No.87）、Grappler最適化（No.84）、XLAコンパイル（No.75）と連携する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | バックエンドのグラフ変換機能であり、直接的な画面関連はない |

## 機能種別

計算処理（関数トレーシング・グラフ変換）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| func | callable | Yes | トレース対象のPython関数 | callable |
| input_signature | list[TensorSpec] | No | 入力テンソルの仕様 | TensorSpecのリスト |
| autograph | bool | No | AutoGraph変換の有効/無効 | デフォルト: True |
| jit_compile | bool | No | XLA JITコンパイルの有効/無効 | デフォルト: None |
| reduce_retracing | bool | No | 再トレーシングの削減 | デフォルト: False |
| experimental_autograph_options | AutoGraphOptions | No | AutoGraphオプション | - |

### 入力データソース

Python関数オブジェクトとその引数。引数の型・形状に基づいてトレースが行われる。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| function | Function | 多相関数オブジェクト（ConcreteFunctionのキャッシュを保持） |
| concrete_function | ConcreteFunction | 特定の入力シグネチャに対する具体的なグラフ |
| result | Tensor/structure | 関数の実行結果 |

### 出力先

呼び出し元（関数の戻り値）。

## 処理フロー

### 処理シーケンス

```
1. @tf.functionデコレータ適用
   └─ Function（多相関数）オブジェクトの生成
2. 初回呼び出し
   └─ 引数からTraceTypeを生成
3. キャッシュ検索
   └─ Function Cacheに該当するConcreteFunctionがあるか確認
4. キャッシュミス時：トレーシング
   └─ Python関数をトレース実行
   └─ AutoGraph変換（autograph=True時）
   └─ FuncGraphの構築
   └─ Variable初期化のリフティング
5. Grappler最適化
   └─ 生成されたグラフにGrappler最適化を適用
6. XLA JITコンパイル（jit_compile=True時）
   └─ グラフをXLAでコンパイル
7. ConcreteFunction実行
   └─ コンパイル済みグラフで推論/学習を実行
8. 結果返却
   └─ 出力テンソルを返却
```

### フローチャート

```mermaid
flowchart TD
    A["@tf.function適用"] --> B["Function(多相関数)生成"]
    B --> C["関数呼び出し"]
    C --> D["引数からTraceType生成"]
    D --> E{"Function Cacheヒット?"}
    E -->|Yes| F["キャッシュ済みConcreteFunction使用"]
    E -->|No| G["Python関数トレース実行"]
    G --> H{"autograph=True?"}
    H -->|Yes| I["AutoGraph変換"]
    H -->|No| J["FuncGraph構築"]
    I --> J
    J --> K{"Variable初期化あり?"}
    K -->|Yes| L["初期化サブグラフのリフティング"]
    K -->|No| M["ConcreteFunction生成"]
    L --> M
    M --> N["Function Cacheに格納"]
    N --> F
    F --> O["ConcreteFunction実行"]
    O --> P["結果返却"]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-88-01 | 引数依存トレーシング | 異なる型・形状の引数は異なるConcreteFunctionを生成する | 関数呼び出し時 |
| BR-88-02 | Variable初期化サポート | 初回呼び出し時のVariable作成をサポート（2回目のトレースで初期化を分離） | Variable作成パターン |
| BR-88-03 | 再トレーシング警告 | 同一関数で多数の異なるシグネチャでの呼び出しは警告が出力される | 再トレーシング過多時 |
| BR-88-04 | Pythonの副作用 | tf.function内のPython副作用（print等）はトレース時のみ実行される | 常時 |

### 計算ロジック

トレーシングでは、Python関数の実行パスをTensorFlowの演算グラフとして記録する。テンソル引数は「プレースホルダ」として扱われ、具体的な値ではなく型と形状のみが記録される。Function Cacheは入力のTraceType（型・形状のハッシュ）をキーとして管理される。

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TypeError | 型エラー | 非callableオブジェクトへのデコレータ適用 | callable関数を渡す |
| ValueError | 引数エラー | input_signatureとの不一致 | input_signatureを修正 |
| OperatorNotAllowedInGraphError | グラフ制約 | グラフ内でPython制御フローを直接使用 | AutoGraphの使用またはtf.condへの書き換え |

### リトライ仕様

トレーシングエラーの自動リトライはない。

## トランザクション仕様

該当なし

## パフォーマンス要件

- 初回トレーシング: 関数の複雑度に依存（通常数百ミリ秒〜数秒）
- キャッシュヒット時: Eager実行と比較して大幅な高速化（10倍以上の場合あり）
- XLA JITコンパイル時: 初回コンパイルは数秒〜数十秒、以降は高速実行

## セキュリティ考慮事項

- トレーシングはPython関数を実行するため、信頼できない関数のトレースには注意

## 備考

- def_function.pyは旧APIの互換レイヤーであり、実装本体はpolymorphic_function.pyに移行
- _tf_function_counterでtf.function呼び出し回数をモニタリング

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | function_cache.py | `tensorflow/core/function/polymorphism/function_cache.py` | Function Cacheの構造（TraceTypeをキーとするキャッシュ） |
| 1-2 | trace_type | `tensorflow/core/function/trace_type/` | TraceType（引数の型・形状表現） |
| 1-3 | capture_container.py | `tensorflow/core/function/capture/capture_container.py` | キャプチャコンテナ（外部変数のキャプチャ管理） |

**読解のコツ**: tf.functionは「多相関数」パターンを実装している。入力の「トレースタイプ」がキャッシュのキーとなり、同じトレースタイプに対しては同じConcreteFunctionが再利用される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | def_function.py | `tensorflow/python/eager/def_function.py` | 互換レイヤー（旧API→新APIへの転送） |
| 2-2 | polymorphic_function.py | `tensorflow/python/eager/polymorphic_function/polymorphic_function.py` | 実装本体 |

**主要処理フロー**:
1. **20-21行目（def_function.py）**: run_functions_eagerly / functions_run_eagerly のインポート
2. **24-25行目（def_function.py）**: Function / function のインポート
3. **16-61行目（polymorphic_function.py）**: Variable初期化パターンのドキュメント
4. **63-100行目（polymorphic_function.py）**: インポートと依存モジュール

#### Step 3: トレーシングコンパイル処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | tracing_compilation.py | `tensorflow/python/eager/polymorphic_function/tracing_compilation.py` | トレーシングコンパイルの実行ロジック |
| 3-2 | function_type_utils.py | `tensorflow/python/eager/polymorphic_function/function_type_utils.py` | 関数型ユーティリティ |
| 3-3 | autograph_util.py | `tensorflow/python/eager/polymorphic_function/autograph_util.py` | AutoGraph統合ユーティリティ |

#### Step 4: AutoGraph統合を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | autograph_util.py | `tensorflow/python/eager/polymorphic_function/autograph_util.py` | tf.functionからAutoGraphへの呼び出し |

### プログラム呼び出し階層図

```
@tf.function (デコレータ)
    |
    +-- polymorphic_function.function() → Function オブジェクト生成
            |
            +-- Function.__call__(args, kwargs)
                    |
                    +-- TraceType生成
                    +-- function_cache.lookup()
                    |       |
                    |       +-- キャッシュヒット → ConcreteFunction返却
                    |
                    +-- tracing_compilation.trace()  (キャッシュミス時)
                    |       +-- autograph_util.convert() (autograph=True)
                    |       +-- func_graph.trace() (FuncGraph構築)
                    |       +-- Variable初期化リフティング
                    |       +-- ConcreteFunction生成
                    |
                    +-- ConcreteFunction.__call__()
                            +-- Grappler最適化
                            +-- XLA JITコンパイル (jit_compile=True)
                            +-- 実行・結果返却
```

### データフロー図

```
[入力]                        [処理]                              [出力]

Python関数 ──────────▶ トレーシング ──────▶ FuncGraph
                              |
引数(型・形状) ──────▶ TraceType生成 ──▶ Function Cache
                              |
                              v
                     ConcreteFunction ──▶ Grappler最適化 ──▶ 最適化グラフ
                              |
入力テンソル ─────────▶ グラフ実行 ────────▶ 出力テンソル
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| def_function.py | `tensorflow/python/eager/def_function.py` | ソース | 互換レイヤー |
| polymorphic_function.py | `tensorflow/python/eager/polymorphic_function/polymorphic_function.py` | ソース | Function/function実装本体 |
| tracing_compilation.py | `tensorflow/python/eager/polymorphic_function/tracing_compilation.py` | ソース | トレーシングコンパイル |
| function_type_utils.py | `tensorflow/python/eager/polymorphic_function/function_type_utils.py` | ソース | 関数型ユーティリティ |
| autograph_util.py | `tensorflow/python/eager/polymorphic_function/autograph_util.py` | ソース | AutoGraph統合 |
| eager_function_run.py | `tensorflow/python/eager/polymorphic_function/eager_function_run.py` | ソース | Eager関数実行制御 |
| attributes.py | `tensorflow/python/eager/polymorphic_function/attributes.py` | ソース | 関数属性 |
| compiler_ir.py | `tensorflow/python/eager/polymorphic_function/compiler_ir.py` | ソース | コンパイラIR |
| tf_method_target.py | `tensorflow/python/eager/polymorphic_function/tf_method_target.py` | ソース | メソッドターゲット |
| function_cache.py | `tensorflow/core/function/polymorphism/function_cache.py` | ソース | 関数キャッシュ |
| func_graph.py | `tensorflow/python/framework/func_graph.py` | ソース | 関数グラフ |
