# 機能設計書 89-AutoGraph変換

## 概要

本ドキュメントは、TensorFlowのAutoGraph変換機能について、その設計・処理内容・インタフェース仕様を記述した機能設計書である。PythonのifやforループをTensorFlowのグラフ操作に自動変換する機能を提供する。

### 本機能の処理概要

AutoGraphは、Pythonの通常の制御フロー構文（if/else、for/while、論理演算等）をTensorFlowのグラフ操作（tf.cond、tf.while_loop等）に自動変換するソースコード変換（トランスパイラ）機能である。tf.function（No.88）と統合されており、デフォルトで有効化されている。

**業務上の目的・背景**：TensorFlowのグラフモードでは、Pythonのif/forをそのまま使用するとトレース時に固定化されてしまい、テンソル値に基づく動的な制御フローが実現できない。AutoGraphはこの問題を解決し、Pythonの自然な記法でテンソル依存の制御フローを記述できるようにする。

**機能の利用シーン**：tf.function内でのテンソル値に基づく条件分岐、動的ループ、リスト操作、例外処理など。tf.functionのautograph=Trueオプション（デフォルト）で自動適用される。

**主要な処理内容**：
1. Python関数のソースコード取得とAST（抽象構文木）解析
2. 各種コンバーターによるAST変換
   - control_flow: if→tf.cond、for/while→tf.while_loop
   - logical_expressions: and/or/not→tf.logical_and/or/not
   - break_statements: break→特殊フラグ制御
   - continue_statements: continue→特殊フラグ制御
   - return_statements: 複数return→単一return変換
   - lists: list→TensorArray変換
   - conditional_expressions: 三項演算子変換
   - asserts: assert→tf.Assert
   - slices: スライス操作変換
   - functions: ネスト関数変換
   - directives: ag.set_loop_options等のディレクティブ処理
   - call_trees: 関数呼び出しツリー変換
   - variables: 変数のリネーム・スコープ管理
3. 変換後のPythonコードの生成と実行
4. エラーメッセージの元ソースへの逆マッピング

**関連システム・外部連携**：tf.function（No.88）と統合。Eager実行コンテキスト（No.87）と連携する。

**権限による制御**：環境変数AUTOGRAPH_STRICT_CONVERSIONで厳格変換モードを制御可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | バックエンドのコード変換機能であり、直接的な画面関連はない |

## 機能種別

計算処理（ソースコード変換・トランスパイラ）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| func | callable | Yes | 変換対象のPython関数 | callable、ソースコード取得可能 |
| recursive | bool | No | 再帰的変換 | デフォルト: True |
| optional_features | Feature | No | オプション機能フラグ | - |

### 入力データソース

Python関数オブジェクトとそのソースコード。inspect.getsource()でソースコードを取得。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| converted_func | callable | TensorFlowグラフ操作に変換されたPython関数 |
| source_map | dict | 変換後コードから元ソースへのマッピング |

### 出力先

tf.functionのトレーシングパイプライン。

## 処理フロー

### 処理シーケンス

```
1. 変換対象判定
   └─ is_unsupported()でwrapt等の非対応ライブラリを検出
2. キャッシュ確認
   └─ 変換済み関数がキャッシュにあるか確認
3. ソースコード取得
   └─ inspect.getsource()でPythonソースを取得
4. AST解析
   └─ ast.parse()で抽象構文木を生成
5. 静的解析
   └─ activity分析、reaching_definitions分析
6. コンバーター適用
   └─ 各コンバーターを順次適用（control_flow、logical_expressions等）
7. コード生成
   └─ 変換後ASTからPythonコードを生成
8. 関数オブジェクト生成
   └─ 変換後コードをコンパイル・実行して関数オブジェクトを生成
9. 変換結果キャッシュ
   └─ 次回利用のためキャッシュに格納
```

### フローチャート

```mermaid
flowchart TD
    A[関数入力] --> B{非対応ライブラリ?}
    B -->|Yes| C[変換スキップ・警告]
    B -->|No| D{キャッシュヒット?}
    D -->|Yes| E[キャッシュ済み関数返却]
    D -->|No| F[ソースコード取得]
    F --> G[AST解析]
    G --> H[静的解析]
    H --> I[コンバーター順次適用]
    I --> J[変換後コード生成]
    J --> K[関数オブジェクト生成]
    K --> L[キャッシュ格納]
    L --> E
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-89-01 | wrapt非対応 | wrapt.FunctionWrapper/BoundFunctionWrapperは変換されず警告が出力される | wrapt使用時 |
| BR-89-02 | ソースコード必須 | 変換にはソースコードの取得が必要（lambda等は制限あり） | 全変換時 |
| BR-89-03 | 制御変数の型制約 | ループ変数はループ前後で型が変わってはならない | ループ変換時 |
| BR-89-04 | 副作用の制限 | Pythonの副作用（print等）はグラフモードでは予期しない動作になる可能性がある | 常時 |

### 計算ロジック

- **if変換**: `if tensor_cond:` → `tf.cond(tensor_cond, true_fn, false_fn)`
- **forループ変換**: `for x in dataset:` → `tf.while_loop(cond, body, loop_vars)`
- **whileループ変換**: `while tensor_cond:` → `tf.while_loop(cond, body, loop_vars)`
- **論理式変換**: `a and b` → `tf.logical_and(a, b)`（テンソル引数時）

## データベース操作仕様

### 操作別データベース影響一覧

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| AutoGraphError | 変換エラー | AutoGraph全般の変換エラー | エラーメッセージを確認 |
| ConversionError | 変換エラー | ソースコード変換時のエラー | 変換対象コードを修正 |
| StagingError | 実行エラー | 変換後コードのPython実行時エラー | 元のPythonコードを確認 |

### リトライ仕様

変換エラーの自動リトライはない。

## トランザクション仕様

該当なし

## パフォーマンス要件

- 変換処理は初回のみ実行（結果はキャッシュ）
- 変換時間は関数の複雑度に依存（通常数十〜数百ミリ秒）

## セキュリティ考慮事項

- ソースコード変換はPythonのAST操作で行われるため、コードインジェクションリスクは低い
- 変換後のコードはTensorFlowの演算グラフとして実行される

## 備考

- AUTOGRAPH_STRICT_CONVERSION=1で厳格変換モードが有効化される
- 変換不要な関数（@tf.autograph.experimental.do_not_convert）もサポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ag_ctx.py | `tensorflow/python/autograph/core/ag_ctx.py` | AutoGraphコンテキスト管理 |
| 1-2 | converter.py | `tensorflow/python/autograph/core/converter.py` | コンバーター基底クラスと設定 |

**読解のコツ**: AutoGraphはPythonのAST（抽象構文木）操作に基づくトランスパイラである。ast.NodeVisitor/NodeTransformerパターンが多用される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | api.py | `tensorflow/python/autograph/impl/api.py` | AutoGraphの公開API |

**主要処理フロー**:
1. **25-63行目**: インポート - 全コンバーター、AST操作ユーティリティ
2. **66-67行目**: AUTOGRAPH_STRICT_CONVERSION環境変数チェック
3. **76-88行目**: AutoGraphError / ConversionError / StagingError定義
4. **91-100行目**: _ErrorMetadataクラス（エラーのソースマッピング）

#### Step 3: 変換パイプラインを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | conversion.py | `tensorflow/python/autograph/impl/conversion.py` | 変換ロジックのコア |

**主要処理フロー**:
- **30行目**: _ALLOWLIST_CACHE - 変換不要関数のキャッシュ
- **33-39行目**: _is_of_known_loaded_module - 既知モジュール関数の判定
- **42-66行目**: _is_known_loaded_type - 既知型の判定
- **69-80行目**: is_unsupported - 非対応エンティティの判定（wrapt等）

#### Step 4: 各コンバーターを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | control_flow.py | `tensorflow/python/autograph/converters/control_flow.py` | if/for/while変換 |
| 4-2 | logical_expressions.py | `tensorflow/python/autograph/converters/logical_expressions.py` | and/or/not変換 |
| 4-3 | break_statements.py | `tensorflow/python/autograph/converters/break_statements.py` | break変換 |
| 4-4 | continue_statements.py | `tensorflow/python/autograph/converters/continue_statements.py` | continue変換 |
| 4-5 | return_statements.py | `tensorflow/python/autograph/converters/return_statements.py` | return変換 |
| 4-6 | lists.py | `tensorflow/python/autograph/converters/lists.py` | list→TensorArray変換 |

#### Step 5: ランタイムオペレーターを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 5-1 | control_flow.py | `tensorflow/python/autograph/operators/control_flow.py` | ランタイム制御フローオペレーター |
| 5-2 | logical.py | `tensorflow/python/autograph/operators/logical.py` | ランタイム論理オペレーター |
| 5-3 | data_structures.py | `tensorflow/python/autograph/operators/data_structures.py` | ランタイムデータ構造オペレーター |

### プログラム呼び出し階層図

```
tf.function(autograph=True)
    |
    +-- autograph_util.convert()
            |
            +-- api.converted_call() / api.to_graph()
                    |
                    +-- conversion.is_unsupported() (非対応チェック)
                    +-- conversion.convert() (変換本体)
                    |       |
                    |       +-- inspect.getsource() (ソース取得)
                    |       +-- ast.parse() (AST生成)
                    |       +-- static_analysis (静的解析)
                    |       |       +-- activity分析
                    |       |       +-- reaching_definitions分析
                    |       |
                    |       +-- converters適用 (順次)
                    |       |       +-- control_flow
                    |       |       +-- logical_expressions
                    |       |       +-- break_statements
                    |       |       +-- continue_statements
                    |       |       +-- return_statements
                    |       |       +-- lists
                    |       |       +-- slices
                    |       |       +-- asserts
                    |       |       +-- conditional_expressions
                    |       |       +-- functions
                    |       |       +-- directives
                    |       |       +-- call_trees
                    |       |       +-- variables
                    |       |
                    |       +-- code_gen() (コード生成)
                    |
                    +-- 変換後関数キャッシュ格納
```

### データフロー図

```
[入力]                           [処理]                         [出力]

Python関数 ──────────▶ ソースコード取得
                              |
ソースコード ────────▶ AST解析 ──▶ 静的解析 ──▶ コンバーター適用
                                                        |
                                                        v
                                              変換後AST ──▶ コード生成 ──▶ 変換後関数
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| api.py | `tensorflow/python/autograph/impl/api.py` | ソース | 公開API |
| conversion.py | `tensorflow/python/autograph/impl/conversion.py` | ソース | 変換コアロジック |
| ag_ctx.py | `tensorflow/python/autograph/core/ag_ctx.py` | ソース | AutoGraphコンテキスト |
| converter.py | `tensorflow/python/autograph/core/converter.py` | ソース | コンバーター基底 |
| config.py | `tensorflow/python/autograph/core/config.py` | ソース | 設定 |
| function_wrappers.py | `tensorflow/python/autograph/core/function_wrappers.py` | ソース | 関数ラッパー |
| control_flow.py | `tensorflow/python/autograph/converters/control_flow.py` | ソース | 制御フロー変換 |
| logical_expressions.py | `tensorflow/python/autograph/converters/logical_expressions.py` | ソース | 論理式変換 |
| break_statements.py | `tensorflow/python/autograph/converters/break_statements.py` | ソース | break変換 |
| continue_statements.py | `tensorflow/python/autograph/converters/continue_statements.py` | ソース | continue変換 |
| return_statements.py | `tensorflow/python/autograph/converters/return_statements.py` | ソース | return変換 |
| lists.py | `tensorflow/python/autograph/converters/lists.py` | ソース | リスト変換 |
| slices.py | `tensorflow/python/autograph/converters/slices.py` | ソース | スライス変換 |
| asserts.py | `tensorflow/python/autograph/converters/asserts.py` | ソース | assert変換 |
| control_flow.py | `tensorflow/python/autograph/operators/control_flow.py` | ソース | ランタイム制御フロー |
| logical.py | `tensorflow/python/autograph/operators/logical.py` | ソース | ランタイム論理演算 |
